library(xtable)
library(ggplot2)
library(munsell)
library(scales)
library(tidyr)
library(dplyr)
library(stringr)

## Import 4all data

# Read from csv
denoise <- tbl_df(read.csv('denoise_4all.csv.gz'))

# Save out to SQLite
my_db <- src_sqlite("denoise_4all.sqlite3", create = T)
denoise_sqlite <- copy_to(my_db, denoise, temporary = FALSE)

# # Read in from SQLite
# my_db <- src_sqlite("denoise_4all.sqlite3", create = F)
# denoise <- tbl(my_db, "denoise")

## Summarize
denoise.mse <- function(df) {
  df %>%
    mutate(rsdr=-20*log10(rel_error), gain=(-20*log10(rel_error) - noise_level)) %>%
    group_by(shape, inner_type, rank, noise_level, regularizer, solver, solve_rank, relconvergetol, penconst_offset) %>%
    summarize(MSE=mean(abs_error^2), maxSE=max(abs_error^2), minSE=min(abs_error^2), origMSE=mean(orig_error^2),
              maxrsdr=max(rsdr), minrsdr=min(rsdr), rsdr=mean(rsdr),
              maxgain=max(gain), mingain=min(gain), gain=mean(gain),
              maxiters=max(outer_iters), miniters=min(outer_iters), iters=mean(outer_iters))
}
# Use empirically best penconst
denoise.bestmse <- denoise %>% denoise.mse() %>%
  collect(n=Inf) %>%
  filter(row_number(MSE) == 1) # returns a tbl_df now

## Set up color scales
noise_level.scale <- mnsl(c("2.5PB 2/4", "2.5PB 7/10"))
rank.scale <- mnsl(c("2.5YR 2/4", "2.5YR 7/10"))
solve_rank.scale <- mnsl(c("2.5G 2/4", "2.5G 7/10"))
relconvergetol.scale <- mnsl(c("2.5RP 2/4", "2.5RP 7/10"))

## Set up labellers
snr_labels <- function(value) {
  temp <- str_c(value, 'dB')
  str_replace_all(temp, 'InfdB', 'noiseless')
}
reg_labels <- function(value) {
  temp <- str_match(value, '.*:\\s(.*)')[,2]
  str_replace_all(temp, 'l', 'L')
}
norm_labels <- function(value) {
  str_replace_all(value, 'l', 'L')
}
solverank_labels <- function(value) {
  temp <- paste(value, 'dyads')
  temp <- str_replace_all(temp, 'NA dyads', 'convex')
  temp <- str_replace_all(temp, '1 dyads', '1 dyad')
}
method_labels <- c(`altmin_SCS: NucNorm_Prod` = 'nonconvex: product',
                   `altmin_SCS: NucNorm_Sum` = 'nonconvex: sum',
                   `mat_SCS: NucNorm_Prod` = 'convex')

## Genereate MSE vs. penconst figures
MSE.vs.penconst.by.noiselevel <- function(df) {
  df %>%
    collect(n=Inf) %>%
    ggplot(aes(x=factor(penconst_offset), y=MSE, color=noise_level, group=noise_level)) +
    geom_point(size=0.5, alpha=0.75) +
    geom_linerange(aes(ymin=minSE, ymax=maxSE), size=0.25, alpha=0.75) +
    geom_line(size=0.25, alpha=0.75, linetype=3) +
    scale_y_log10() +
    scale_color_gradient(low=noise_level.scale[1], high=noise_level.scale[2]) +
    facet_grid(regularizer ~ inner_type, labeller = label_value) +
    labs(x = 'j: penalty constant offset', y = 'avg. squared error', color = 'SNR (dB)') +
    theme_bw() +
    theme(text = element_text(size=8))
}

MSE.vs.penconst.by.rank <- function(df) {
  df %>%
    collect(n=Inf) %>%
    ggplot(aes(x=factor(penconst_offset), y=MSE, color=rank, group=rank)) +
    geom_point(size=0.5, alpha=0.75) +
    geom_linerange(aes(ymin=minSE, ymax=maxSE), size=0.25, alpha=0.75) +
    geom_line(size=0.25, alpha=0.75, linetype=3) +
    scale_y_log10() +
    scale_color_gradient(low=rank.scale[1], high=rank.scale[2]) +
    facet_grid(regularizer ~ inner_type, labeller = label_value) +
    labs(x = 'j: penalty constant offset', y = 'avg. squared error', color = 'rank') +
    theme_bw() +
    theme(text = element_text(size=8))
}

# Generate Figure B.1
MSE.vs.penconst.by.noiselevel(denoise %>% filter(solver == 'mat_SCS' & rank == 1) %>% denoise.mse()) +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/4all-mse-penconst-noiselevel-mat.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure B.2
MSE.vs.penconst.by.noiselevel(denoise %>% filter(solver == 'sdp_SCS' & rank == 1) %>% denoise.mse()) +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/4all-mse-penconst-noiselevel-sdp.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure B.7
regularizers <- denoise %>% select(regularizer) %>% distinct() %>% collect(n=Inf)
reglists <- lapply(split(regularizers, 1:6), function(x) x$regularizer)
altminplots <- lapply(reglists, function(x) MSE.vs.penconst.by.noiselevel(denoise %>% filter(solver == 'altmin_SCS' & solve_rank == 16 & relconvergetol == 5e-4 & rank == 1 & regularizer %in% x) %>% denoise.mse()) + theme(text = element_text(size=8)) +     facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=norm_labels)))
cairo_pdf("figures/4all-mse-penconst-altmin.pdf", width=7*2, height=5.5*2, onefile = TRUE)
invisible(lapply(altminplots, print))
dev.off()

# Generate Figure 5.2
MSE.vs.penconst.by.noiselevel(denoise %>%
                                filter(solver == 'mat_SCS' & rank == 1) %>%
                                filter(regularizer == 'NucNorm_Prod: l1, l1') %>%
                                filter(inner_type %in% c('sparse, sparse', 'sign, sign')) %>%
                                denoise.mse()) +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/body-mse-penconst-snr.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

## Generate gain vs. SNR figures
gain.vs.snr <- function(df) {
  df %>%
    ungroup() %>%
    ggplot(aes(x=noise_level, y=gain, color=rank)) +
    geom_point(size=0.5, alpha=0.75) +
    geom_linerange(aes(ymin=mingain, ymax=maxgain), size=0.25, alpha=0.75) +
    geom_line(aes(group=interaction(rank, solver)), linetype=3, size=0.25, alpha=0.75) +
    geom_hline(yintercept=0, color='gray', alpha=0.75) +
    scale_color_gradient(low=rank.scale[1], high=rank.scale[2]) +
    facet_grid(regularizer ~ inner_type, labeller = label_value) +
    labs(x = 'SNR (dB)', y = 'avg. gain (dB)', color = 'rank') +
    theme_bw() +
    theme(text = element_text(size=8))
}

# Generate Figure B.3
denoise.bestmse %>%
  filter(solver == 'mat_SCS') %>%
  gain.vs.snr() +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/4all-gain-snr-mat.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure B.4
denoise.bestmse %>%
  filter(solver == 'sdp_SCS') %>%
  gain.vs.snr() +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/4all-gain-snr-sdp.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure 5.3
denoise.bestmse %>%
  filter(solver == 'mat_SCS') %>%
  filter(regularizer == 'NucNorm_Prod: l1, l1') %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
  gain.vs.snr() +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/body-gain-snr.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

## Generate Figure 5.4
denoise %>%
  filter(solver == 'altmin_SCS') %>%
  select(outer_iters, relconvergetol) %>%
  collect(n=Inf) %>%
  ggplot(aes(x=outer_iters)) +
  geom_histogram(binwidth=1) +
  facet_wrap(~ relconvergetol) +
  labs(x = 'outer iterations') +
  theme_bw()
ggsave('figures/4all-outeriters-by-relconvergetol.pdf', width=5.5, height=3, units='in', scale=1.5, device=cairo_pdf)

## Generate gain vs. SNR by solver figures
gain.vs.snr.by.solver <- function(df) {
  df %>%
    ungroup() %>%
    ggplot(aes(x=noise_level, y=gain, color=solver.method, shape=solver.method)) +
    geom_point(size=1, alpha=0.75) +
    geom_linerange(aes(ymin=mingain, ymax=maxgain), size=0.25, alpha=0.75) +
    geom_line(aes(group=solver.method), linetype=3, size=0.25, alpha=0.75) +
    geom_hline(yintercept=0, color='gray', alpha=0.75) +
    facet_grid(norm ~ inner_type, labeller = labeller(norm = norm_labels)) +
    labs(x = 'SNR (dB)', y = 'avg. gain (dB)', color = 'method', shape = 'method') +
    theme_bw() +
    theme(text = element_text(size=8))
}

# separate regularizer into method and norm
# get norms handled by mat_SCS
# filter altmin by those norms (excluding SDR)
# combine method and solver
# filter to rank 1 operators and 5e-4 relconvtol & solve_rank 16
# plot gain vs. snr, series by solver-method
denoise.bestmse.altmin.mat <- function(RANK, RELCONVERGETOL, SOLVE_RANK) {
  matnorms <- denoise.bestmse %>%
    ungroup() %>%
    filter(solver == 'mat_SCS') %>%
    separate(regularizer, c('method', 'norm'), sep=': ') %>%
    distinct(norm) %>%
    select(norm)
  matnorms <- matnorms[[1]]
  temp <- denoise.bestmse %>%
    ungroup() %>%
    filter(solver == 'altmin_SCS', relconvergetol==RELCONVERGETOL, rank==RANK, solve_rank==SOLVE_RANK) %>%
    separate(regularizer, c('method', 'norm'), sep=': ') %>%
    filter(method != 'NucNorm_SDR') %>%
    filter(norm %in% matnorms)
  bind_rows(denoise.bestmse %>% ungroup() %>% filter(solver == 'mat_SCS', rank==RANK) %>% separate(regularizer, c('method', 'norm'), sep=': '),
            temp) %>%
    unite(solver.method, solver, method, sep=': ')
}

denoise.bestmse.altmin.sdp <- function(RANK, RELCONVERGETOL, SOLVE_RANK) {
  temp <- denoise.bestmse %>%
    ungroup() %>%
    filter(solver == 'altmin_SCS', relconvergetol==RELCONVERGETOL, rank==RANK, solve_rank==SOLVE_RANK) %>%
    separate(regularizer, c('method', 'norm'), sep=': ') %>%
    filter(method == 'NucNorm_SDR')
  bind_rows(denoise.bestmse %>% ungroup() %>% filter(solver == 'sdp_SCS', rank==RANK) %>% separate(regularizer, c('method', 'norm'), sep=': '),
            temp) %>%
    unite(solver.method, solver, method, sep=': ')
}

# Generate Figure B.5
denoise.bestmse.altmin.mat(1, 5e-4, 16) %>%
  mutate(solver.method = plyr::mapvalues(solver.method, c('altmin_SCS: NucNorm_Prod', 'altmin_SCS: NucNorm_Sum', 'mat_SCS: NucNorm_Prod'),
                                         c('product', 'sum', 'convex'))) %>%
  gain.vs.snr.by.solver()
ggsave('figures/4all-gain-snr-altminmat.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure B.6
denoise.bestmse.altmin.sdp(1, 5e-4, 16) %>%
  mutate(solver.method = plyr::mapvalues(solver.method, c('altmin_SCS: NucNorm_SDR', 'sdp_SCS: NucNorm_SDR'),
                                         c('nonconvex', 'SDP'))) %>%
  gain.vs.snr.by.solver()
ggsave('figures/4all-gain-snr-altminsdp.pdf', width=7, height=5.5, units='in', scale=2.0, device=cairo_pdf)

# Generate Figure 5.5
denoise.bestmse.altmin.mat(1, 5e-4, 16) %>%
  filter(norm == 'l1, l1') %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
  mutate(solver.method = plyr::mapvalues(solver.method, c('altmin_SCS: NucNorm_Prod', 'altmin_SCS: NucNorm_Sum', 'mat_SCS: NucNorm_Prod'),
                                         c('product', 'sum', 'convex'))) %>%
  gain.vs.snr.by.solver()
ggsave('figures/body-gain-snr-altminmat.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

# Generate Figure 5.6
denoise.bestmse.altmin.sdp(1, 5e-4, 16) %>%
  filter(norm == 'linf, linf') %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
  mutate(solver.method = plyr::mapvalues(solver.method, c('altmin_SCS: NucNorm_SDR', 'sdp_SCS: NucNorm_SDR'),
                                         c('nonconvex', 'SDP'))) %>%
  gain.vs.snr.by.solver()
ggsave('figures/body-gain-snr-altminsdp.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

## Generate gain vs. SNR by relconvergetol
gain.vs.snr.by.relconvergetol <- function(df) {
  df %>%
    ungroup() %>%
    ggplot(aes(x=noise_level, y=gain, color=log10(relconvergetol))) +
    geom_point(alpha=0.75, size=0.5) +
    geom_linerange(aes(ymin=mingain, ymax=maxgain, fatten=1), alpha=0.75, size=0.25) +
    geom_line(aes(group=log10(relconvergetol)), linetype=3, size=0.25) +
    geom_hline(yintercept=0, color='gray', alpha=0.75) +
    scale_color_gradient(low=relconvergetol.scale[1], high=relconvergetol.scale[2]) +
    facet_grid(regularizer ~ inner_type, labeller = label_value) +
    labs(x = 'SNR (dB)', y = 'avg. gain (dB)', color = 'tol. (log)') +
    theme_bw() +
    theme(text = element_text(size=8))
}

# Generate Figure B.8
altminplots <- lapply(reglists, function(x) gain.vs.snr.by.relconvergetol(denoise.bestmse %>% filter(solver == 'altmin_SCS' & solve_rank == 16 & rank == 1 & regularizer %in% x)) + theme(text = element_text(size=8)) +     facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=norm_labels)))
cairo_pdf("figures/4all-gain-snr-altmin.pdf", width=8*2, height=6.5*2, onefile = TRUE)
invisible(lapply(altminplots, print))
dev.off()

# Generate Figure 5.7
denoise.bestmse %>%
  filter(solver == 'altmin_SCS') %>% filter(solve_rank == 16) %>% filter(rank == 1) %>%
  filter(relconvergetol %in% c(1e-1, 1e-2, 1e-3)) %>%
  filter(regularizer == 'NucNorm_Sum: l1, l1') %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
  gain.vs.snr.by.relconvergetol() +
  geom_line(data=filter(denoise.bestmse, solver=='mat_SCS', rank==1, regularizer == 'NucNorm_Prod: l1, l1',
                        inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
              ungroup() %>% mutate(regularizer=NULL), linetype=2, color='black', size=0.25) +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/body-gain-snr-relconvtol.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

## Generate MSE vs. penconst by solve rank
MSE.vs.penconst.by.solve_rank <- function(df) {
  df %>%
    collect(n=Inf) %>%
    ggplot(aes(x=factor(penconst_offset), y=MSE, color=solve_rank, group=solve_rank)) +
    geom_point(size=0.5, alpha=0.75) +
    geom_linerange(aes(ymin=minSE, ymax=maxSE), size=0.25, alpha=0.75) +
    geom_line(size=0.25, alpha=0.75) +
    scale_y_log10() +
    scale_color_gradient(low=solve_rank.scale[1], high=solve_rank.scale[2]) +
    facet_grid(regularizer ~ inner_type, labeller = label_value) +
    labs(x = 'j: penalty constant offset', y = 'avg. squared error', color = 'solver rank') +
    theme_bw() +
    theme(text = element_text(size=8))
}

# Generate Figure B.9
altminplots <- lapply(reglists, function(x) MSE.vs.penconst.by.solve_rank(denoise %>% filter(solver == 'altmin_SCS' & noise_level == 10 & relconvergetol == 5e-4 & rank == 1 & regularizer %in% x) %>% denoise.mse()) + theme(text = element_text(size=8)) + facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=norm_labels)))
cairo_pdf("figures/4all-mse-penconst-solverank-altmin.pdf", width=8*2, height=6.5*2, onefile = TRUE)
invisible(lapply(altminplots, print))
dev.off()

# Generate Figure 5.8
denoise %>%
  filter(solver == 'altmin_SCS', noise_level == 10, relconvergetol == 5e-4, rank == 1,
         inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign'), regularizer == 'NucNorm_Sum: l1, l1') %>%
  denoise.mse() %>%
  MSE.vs.penconst.by.solve_rank() +
  facet_grid(regularizer ~ inner_type, labeller = labeller(regularizer=reg_labels))
ggsave('figures/body-mse-penconst-solverank.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

## Generate gain tables for 4all
write.gain.table <- function(r) {
  tab <- denoise.bestmse %>%
    ungroup() %>%
    filter(rank == r) %>%
    filter(solver == 'altmin_SCS') %>%
    filter(relconvergetol == 5e-4) %>% filter(solve_rank == 16) %>%
    filter(noise_level == 15) %>%
    separate(regularizer, c('method', 'norm'), sep=': ') %>%
    filter(method == 'NucNorm_Sum') %>%
    group_by(norm, inner_type) %>%
    summarize(gain) %>%
    spread(norm, gain)
  xtab <- xtable(tab, digits=1)
  print(xtab, file=paste('tables/4all-norm-type-operrank', r, 'table.tex', sep=''), floating=FALSE, include.rownames=FALSE)
  print(xtab, file=paste('tables/4all-norm-type-operrank', r, 'table-rows.tex', sep=''), only.contents=TRUE, include.rownames=FALSE)
}
write.gain.table(1)
write.gain.table(2)
write.gain.table(4)
write.gain.table(8)

## Generate SDP vs. nonconvex figures

# Generate Figure 5.11[top]
denoise.bestmse %>%
  ungroup() %>%
  separate(regularizer, c('method', 'norm'), sep=': ') %>%
  filter(noise_level == 10) %>%
  filter(norm %in% c('l2, linf', 'linf, l2', 'linf, linf')) %>%
  filter(inner_type %in% c('gaussian, gaussian', 'gaussian, sign', 'sign, sign')) %>%
  filter(method %in% c('NucNorm_SDR', 'NucNorm_Sum') & solver == 'altmin_SCS' & relconvergetol == 5e-4 & solve_rank == 16) %>%
  mutate(method = plyr::mapvalues(method, c('NucNorm_SDR', 'NucNorm_Sum'),
                                  c('relaxation', 'nuc. norm'))) %>%
  ggplot(aes(x=rank, y=gain)) +
  geom_point(aes(color=method, shape=method)) +
  geom_linerange(aes(ymin=mingain, ymax=maxgain, fatten=1, color=method, shape=method), alpha=0.75, size=0.25) +
  geom_line(data=transform(filter(separate(ungroup(denoise.bestmse), regularizer, c('method', 'norm'), sep=': '),
                                  noise_level == 10, norm == 'l2, l2', solver == 'mat_SCS',
                                  inner_type %in% c('gaussian, gaussian', 'gaussian, sign', 'sign, sign')),
                           norm=NULL), color='black', linetype=3, size=0.25) +
  geom_hline(aes(yintercept=0), color='gray', linetype=2) +
  facet_grid(norm ~ inner_type, labeller = labeller(norm = norm_labels)) +
  labs(x='rank', y='avg. gain (dB)', shape='method', color='method') +
  theme_bw() +
  theme(text = element_text(size=8))
ggsave('figures/body-gain-rank-sumvssdr.pdf', width=5.5, height=4, units='in', scale=1.0, device=cairo_pdf)

## Load data for 16all
denoise <- tbl_df(read.csv("denoise_16all.csv.gz"))

## Use empiracally best penconsts
denoise.bestmse <- denoise %>% denoise.mse() %>%
  collect(n=Inf) %>%
  filter(row_number(MSE) == 1) # returns a tbl_df now

## Generate gain tables for 16all
write.gain.table <- function(r) {
  tab <- denoise.bestmse %>%
    ungroup() %>%
    filter(rank == r) %>%
    filter(solver == 'altmin_SCS') %>%
    filter(relconvergetol == 1e-3) %>% filter(solve_rank == 16) %>%
    separate(regularizer, c('method', 'norm'), sep=': ') %>%
    filter(method == 'NucNorm_Sum') %>%
    group_by(norm, inner_type) %>%
    summarize(gain) %>%
    spread(norm, gain)
  xtab <- xtable(tab, digits=1)
  print(xtab, file=paste('tables/16all-norm-type-operrank', r, '-table.tex', sep=''), floating=FALSE, include.rownames=FALSE)
  print(xtab, file=paste('tables/16all-norm-type-operrank', r, '-table-rows.tex', sep=''), only.contents=TRUE, include.rownames=FALSE)
}
write.gain.table(1)
write.gain.table(2)
write.gain.table(4)
write.gain.table(8)

## Generate Figure 5.11[bottom]
denoise.bestmse %>%
  ungroup() %>%
  separate(regularizer, c('method', 'norm'), sep=': ') %>%
  filter(noise_level == 10) %>%
  filter(norm %in% c('l2, linf', 'linf, l2', 'linf, linf')) %>%
  filter(inner_type %in% c('gaussian, gaussian', 'gaussian, sign', 'sign, sign')) %>%
  filter(method %in% c('NucNorm_SDR', 'NucNorm_Sum') & solver == 'altmin_SCS' & relconvergetol == 1e-3 & solve_rank == 16) %>%
  mutate(method = plyr::mapvalues(method, c('NucNorm_SDR', 'NucNorm_Sum'),
                                  c('relaxation', 'nuc. norm'))) %>%
  ggplot(aes(x=rank, y=gain)) +
  geom_point(aes(color=method, shape=method)) +
  geom_linerange(aes(ymin=mingain, ymax=maxgain, fatten=1, color=method, shape=method), alpha=0.75, size=0.25) +
  geom_line(data=transform(filter(separate(ungroup(denoise.bestmse), regularizer, c('method', 'norm'), sep=': '),
                                  noise_level == 10, norm == 'l2, l2', solver == 'mat_SCS',
                                  inner_type %in% c('gaussian, gaussian', 'gaussian, sign', 'sign, sign')),
                           norm=NULL), color='black', linetype=3, size=0.25) +
  geom_hline(aes(yintercept=0), color='gray', linetype=2) +
  facet_grid(norm ~ inner_type, labeller = labeller(norm = norm_labels)) +
  labs(x='rank', y='avg. gain (dB)', shape='method', color='method') +
  theme_bw() +
  theme(text = element_text(size=8))
ggsave('figures/body-16all-gain-rank-sumvssdr.pdf', width=5.5, height=4, units='in', scale=1.0, device=cairo_pdf)

## Generate gain vs operator rank figures

# Generate Figure 5.9
denoise.bestmse %>%
  ungroup() %>%
  separate(regularizer, c('method', 'norm'), sep=': ') %>%
  filter(noise_level == 10) %>%
  filter(norm %in% c('l1, l1')) %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, sign', 'sign, sign')) %>%
  filter((method == 'NucNorm_Prod' & solver == 'mat_SCS') |
           (method == 'NucNorm_Sum' & solver == 'altmin_SCS' & relconvergetol == 1e-3 & solve_rank == 16)) %>%
  mutate(solver = plyr::mapvalues(solver, c('altmin_SCS', 'mat_SCS'),
                                  c('nonconvex', 'convex'))) %>%
  ggplot(aes(x=rank, y=gain, color=solver, shape=solver, group=solver)) +
  geom_point() +
  geom_linerange(aes(ymin=mingain, ymax=maxgain, fatten=1, color=solver), alpha=0.75, size=0.25) +
  geom_line(size=0.25, alpha=0.75, linetype=3) +
  geom_hline(aes(yintercept=0), color='gray', linetype=2) +
  facet_grid(norm ~ inner_type, labeller = labeller(norm = norm_labels)) +
  labs(x='rank', y='avg. gain (dB)', color='solver', shape='solver') +
  theme_bw() +
  theme(text = element_text(size=8))
ggsave('figures/body-16all-gain-rank.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)

# Generate Figure 5.10
denoise.bestmse %>%
  ungroup() %>%
  separate(regularizer, c('method', 'norm'), sep=': ') %>%
  filter(noise_level == 10) %>%
  filter(norm %in% c('l1, l2')) %>%
  filter(inner_type %in% c('sparse, sparse', 'sparse, gaussian', 'gaussian, gaussian')) %>%
  filter((method == 'NucNorm_Prod' & solver == 'mat_SCS') |
           (method == 'NucNorm_Sum' & solver == 'altmin_SCS' & relconvergetol == 1e-3 & solve_rank == 16)) %>%
  mutate(solver = plyr::mapvalues(solver, c('altmin_SCS', 'mat_SCS'),
                                  c('nonconvex', 'convex'))) %>%
  ggplot(aes(x=rank, y=gain, color=solver, shape=solver, group=solver)) +
  geom_point() +
  geom_linerange(aes(ymin=mingain, ymax=maxgain, fatten=1, color=solver), alpha=0.75, size=0.25) +
  geom_line(size=0.25, alpha=0.75, linetype=3) +
  geom_hline(aes(yintercept=0), color='gray', linetype=2) +
  facet_grid(norm ~ inner_type, labeller = labeller(norm = norm_labels)) +
  labs(x='rank', y='avg. gain (dB)', color='solver', shape='solver') +
  theme_bw() +
  theme(text = element_text(size=8))
ggsave('figures/body-16all-gain-rank-l1l2.pdf', width=5.5, height=3, units='in', scale=1.0, device=cairo_pdf)
