library(ggplot2)
library(dplyr)
library(broom)
library(stringr)

## Load the data
selfcalib_multiple <- read.csv("selfcalib_multiple.csv.gz")

## Summarize the data
selfcalib_multiple.summarized <- selfcalib_multiple %>%
  group_by(L, N, k, n, M, generate_fn, snr, reg, outer_solver, inner_solver,
           solve_rank, relconvergetol, max_outer_iters, penconst_offset) %>%
  summarize(mean_error=mean(rel_error), success=mean(rsdr > snr), maxouteriters=max(outer_iters))

## For each group, choose the empirically best penconst_offset
selfcalib_multiple.bestsuccess <- selfcalib_multiple.summarized %>%
  filter(row_number(desc(success)) == 1)

## Set up labels for plotting
snr_labels <- function(value) {
  temp <- str_c(value, 'dB')
  str_replace_all(temp, 'InfdB', 'noiseless')
}
reg_labels <- function(value) {
  temp <- str_match(value, '.*:\\s(.*)')[,2]
  str_replace_all(temp, 'l', 'L')
}
solverank_labels <- function(value) {
  temp <- paste(value, 'dyads')
  temp <- str_replace_all(temp, 'NA dyads', 'convex')
  temp <- str_replace_all(temp, '1 dyads', '1 dyad')
}
M_labels <- function(value) {
  temp <- paste(value, 'snapshots')
}
genfn_labels <- c(`sparse_identical_entries` = 'Identical snapshots',
                  `sparse_identical_locs` = 'Simultaneous sparisty',
                  `sparse_nonidentical` = 'Independent snapshots')

## Generate Figure 7.3
generatefn.vs.norm <- function(m) {
  selfcalib_multiple.bestsuccess %>%
    filter(outer_solver == 'altmin') %>%
    filter(solve_rank == 4) %>%
    filter(M == m) %>%
    ggplot(aes(x=n, y=k)) +
    geom_raster(hjust=0, vjust=0, aes(fill=success*100), interpolate=FALSE) +
    scale_fill_gradient(limits=c(0,100), low="black", high="white") +
    coord_cartesian(xlim=c(0,15), ylim=c(0,15)) +
    facet_grid(reg ~ generate_fn, labeller = labeller(reg = as_labeller(reg_labels),
                                                      generate_fn = as_labeller(genfn_labels))) +
    theme_bw() +
    labs(x = 's: nnz(y)', y = 'm: length of x', fill = 'success (%)') +
    theme(aspect.ratio=1, text = element_text(size=10))
}
generatefn.vs.norm(8)
ggsave('figures/multiple-generatefn-norm.pdf', width=5.5, height=4, units='in', scale=1.5, device=cairo_pdf)

## Generate Figure 7.4
M.vs.norm_single <- function(genfn, norm) {
  selfcalib_multiple.bestsuccess %>%
    filter(outer_solver == 'altmin') %>%
    filter(solve_rank == 4) %>%
    filter(generate_fn == genfn) %>%
    filter(reg == norm) %>%
    filter(M > 1) %>%
    ggplot(aes(x=n, y=k)) +
    geom_raster(hjust=0, vjust=0, aes(fill=success*100), interpolate=FALSE) +
    scale_fill_gradient(limits=c(0,100), low="black", high="white") +
    coord_cartesian(xlim=c(0,15), ylim=c(0,15)) +
    facet_grid(. ~ M, labeller = labeller(M = as_labeller(M_labels))) +
    theme_bw() +
    labs(x = 's: nnz(y)', y = 'm: length of x', fill = 'success (%)') +
    theme(aspect.ratio=1, text = element_text(size=10))
}
# Figure 7.4(a)
M.vs.norm_single('sparse_nonidentical', 'NucNorm_Prod: l2, l1')
ggsave('figures/multiple-snapshots-nonidentical.pdf', width=5.5, height=2.5, units='in', scale=1.5, device=cairo_pdf)
# Figure 7.4(b)
M.vs.norm_single('sparse_identical_locs', 'NucNorm_Prod: l2, l1l2')
ggsave('figures/multiple-snapshots-identicallocs.pdf', width=5.5, height=2.5, units='in', scale=1.5, device=cairo_pdf)
# Figure 7.4(c)
M.vs.norm_single('sparse_identical_entries', 'NucNorm_Prod: l2, l1linf')
ggsave('figures/multiple-snapshots-identicalentries.pdf', width=5.5, height=2.5, units='in', scale=1.5, device=cairo_pdf)
