library(ggplot2)
library(dplyr)
library(broom)
library(stringr)

## Load the data
selfcalib_single <- bind_rows(read.csv("selfcalib_single_altmin.csv.gz"),
                              read.csv("selfcalib_single_matnoisy.csv.gz"))
selfcalib_single <- selfcalib_single %>% mutate(success=(rsdr > snr))
selfcalib_single <- bind_rows(selfcalib_single,
                              read.csv("selfcalib_single_matnoiseless.csv.gz") %>% mutate(success=(rel_error < 1e-2)))

## Summarize the data
selfcalib_single.summarized <- selfcalib_single %>%
  group_by(L, N, k, n, M, snr, reg, outer_solver, inner_solver,
           solve_rank, relconvergetol, max_outer_iters, penconst_offset) %>%
  summarize(mean_error=mean(rel_error), success=mean(success), maxouteriters=max(outer_iters))

## For each group, choose the empirically best penconst_offset
selfcalib_single.bestsuccess <- selfcalib_single.summarized %>%
  filter(row_number(desc(success)) == 1)

## Logistic regression to find empirical phase transition
fit.and.pred <- function(df) {
  q = .5
  model <- glm(success ~ k, data=df, family="binomial")
  ((log(q/(1-q)) - coef(model)["(Intercept)"]) / coef(model)["k"])[[1]]
}

logitFits <- selfcalib_single %>% ungroup() %>%
  group_by(L, N, n, M, generate_fn, snr, reg, outer_solver, inner_solver,
           solve_rank, relconvergetol, max_outer_iters, penconst_offset) %>%
  do(pred50 = fit.and.pred(.))
logitFit <- tidy(logitFits, pred50) %>% rename(pred50 = x)

## Join the logistic regression fit with selfcalib_single.bestsuccess
selfcalib_single.fitted <- selfcalib_single.bestsuccess %>% left_join(logitFit)
# only include data on k=2 (only want one phase transition series)
selfcalib_single.fitted <- filter(selfcalib_single.fitted, k==2)

## Set up labels for plotting
snr_labels <- function(value) {
  temp <- str_c(value, 'dB')
  str_replace_all(temp, 'InfdB', 'noiseless')
}
reg_labels <- function(value) {
  temp <- str_match(value, '.*:\\s(.*)')[,2]
  str_replace_all(temp, 'l', 'L')
}
solverank_labels <- function(value) {
  temp <- paste(value, 'dyads')
  temp <- str_replace_all(temp, 'NA dyads', 'convex')
  temp <- str_replace_all(temp, '1 dyads', '1 dyad')
}

## Generate Figure 7.1
selfcalib_single.bestsuccess %>%
  filter(outer_solver == 'mat') %>%
  ggplot(aes(x=n, y=k)) +
  geom_raster(hjust=0, vjust=0, aes(fill=success*100), interpolate=FALSE) +
  scale_fill_gradient(limits=c(0,100), low="black", high="white") +
  geom_line(data=data.frame(pts=0:15), aes(x=pts, y=35/pts), color="red") +
  geom_line(data=filter(selfcalib_single.fitted, outer_solver=='mat'), aes(y=pred50), color="yellow") +
  coord_cartesian(xlim=c(0,15), ylim=c(0,15)) +
  facet_grid(reg ~ snr, labeller = labeller(snr = as_labeller(snr_labels),
                                            reg = as_labeller(reg_labels))) +
  theme_bw() +
  labs(x = 's: nnz(y)', y = 'm: length of x', fill = 'success (%)') +
  theme(aspect.ratio=1, text = element_text(size=10))
ggsave('figures/single-snr-norm.pdf', width=5.5, height=2.0, units='in', scale=1.5, device=cairo_pdf)

## Generate Figure 7.2
selfcalib_single.bestsuccess %>%
  filter(snr == 15) %>%
  ggplot(aes(x=n, y=k)) +
  geom_raster(hjust=0, vjust=0, aes(fill=success*100), interpolate=FALSE) +
  scale_fill_gradient(limits=c(0,100), low="black", high="white") +
  geom_line(data=data.frame(pts=0:15), aes(x=pts, y=35/pts), color="red") +
  geom_line(data=filter(selfcalib_single.fitted, snr==15), aes(y=pred50), color="yellow") +
  coord_cartesian(xlim=c(0,15), ylim=c(0,15)) +
  facet_grid(reg ~ solve_rank, labeller = labeller(reg = as_labeller(reg_labels),
                                                   solve_rank = as_labeller(solverank_labels))) +
  theme_bw() +
  labs(x = 's: nnz(y)', y = 'm: length of x', fill = 'success (%)') +
  theme(aspect.ratio=1, text = element_text(size=10))
ggsave('figures/single-solverank-norm.pdf', width=5.5, height=2.5, units='in', scale=1.5, device=cairo_pdf)
