-------------------------- MODULE HistoryIsRealizable --------------------------
(* For a specification that includes history-determined variables, we prove that
it suffices to synthesize an implementation with the history variables unhidden.
More precisely


LET
    Spec(x, h) == Prop(x) /\ History(x, h)
    SpecH(x) == \EE h:  Spec(x, h)
IN
    IsRealizable(SpecH)  <=>  IsRealizable(Spec)


This result is useful for using temporal synthesis algorithms that do not
reason about \EE (for example GR(k) synthesis), and then hiding the
history variables, in order to obtain an implementation for properties that
contain temporal quantification of only history variables.


Author: Ioannis Filippidis


References
==========

[1] M. Abadi and L. Lamport
    "The existence of refinement mappings",
    TCS, 1991, 10.1016/0304-3975(91)90224-P

[2] M. Abadi and L. Lamport
    "An old-fashioned recipe for real time"
    TOPLAS, 1994, 10.1145/186025.186058

[3] N. Piterman and A. Pnueli and Y. Sa'ar
	"Synthesis of reactive(1) designs",
    VMCAI, 2006, 10.1007/11609773\_24

[4] L. Lamport and S. Merz
    "Auxiliary variables in TLA+",
    ArXiv, 2017, https://arxiv.org/pdf/1703.05121.pdf

--------------------------------------------------------------------------------
Copyright 2016-2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS TemporalLogic, TLAPS


------------------------- MODULE HistoryDeterminedVar --------------------------
VARIABLE v,
CONSTANT Init(_, _)  (* corresponds to f in [2, Eq.(4)] *)
CONSTANT Next(_, _, _)  (* corresponds to g in [2, Eq.(4)] *)

Hist(h, v) ==
    LET
        N == << h' = Next(h, v, v') >>_v
    IN
        /\ h = Init(v)
        /\ [][N]_<< h, v >>

THEOREM HistoryExists ==
    \AA v:  \EE h:  Hist(h, v)
    PROOF OMITTED
================================================================================


(* ATTENTION: Below this point, \EE stands for stutter-sensitive
quantification. For simplicity, a stutter-sensitive definition of
realizability is used, in raw TLA+. An analogous result can be shown in TLA+.

We also use a simplified definition of IsRealizable, with mu omitted.
Realizations that contain an initial condition are used.
*)


------------------------ MODULE RawHistoryDeterminedVar ------------------------
VARIABLE v,
CONSTANT Init(_, _)
CONSTANT Next(_, _, _)

Hist(h, v) ==
    LET
        N == h' = Next(h, v, v')
    IN
        /\ h = Init(v)
        /\ []N

THEOREM HistoryExists ==
    \AA v:  \EE h:  Hist(h, v)
    PROOF OMITTED
================================================================================


PROPOSITION ImplEE ==
    ASSUME
        TEMPORAL A(_), TEMPORAL B(_),
        \AA q:  A(q) => B(q)
    PROVE
        (\EE q:  A(q))  =>  (\EE q:  B(q))


PROPOSITION HidingHistoryPreservesRealizability ==
    ASSUME
        CONSTANT I(_, _, _),
        TEMPORAL Phi(_, _, _),
        IsRealizable(I, Phi)
    PROVE
        LET
            Init(x, y) == \E q:  I(x, y, q)
            PhiH(x, y) == \EE q:  Phi(x, y, q)
        IN
            IsRealizable(Init, PhiH)
PROOF
<1> DEFINE
    g(x, y) == CHOOSE q:  I(x, y, q)
    Init(x, y) == \E q:  I(x, y, q)
(* We cannot use CHOOSE to define fx, ... because CHOOSE cannot be applied
to a temporal-level expression. PICK can, but can occur only in proofs.
*)
<1>1. PICK fx1, fq, fm1, r:
        /\ IsAFunction(fx1)
        /\ IsAFunction(fq)
        /\ IsAFunction(fm1)
        /\ \AA x, y, q:
            \/ ~ \EE m:
                LET
                    args == << x, y, q, m >>
                IN
                    /\ I(x, y, q)
                    /\ m = r
                    /\ [] /\ x' = fx1[args]
                          /\ q' = fq[args]
                          /\ m' = fm1[args]
            \/ Phi(x, y, q)
    BY DEF IsRealizable
        (* and HidingHistoryPreservesRealizability!assumption *)
<1>2. \AA x, y:
        \/ ~ \EE q, m:
            LET
                args == << x, y, q, m >>
            IN
                /\ I(x, y, q)
                /\ m = r
                /\ [] /\ x' = fx1[args]
                      /\ q' = fq[args]
                      /\ m' = fm1[args]
        \/ \EE q:  Phi(x, y, q)
    BY <1>1, ImplEE
<1> DEFINE
    repack(t) ==
        LET
            x == t[1]
            y == t[2]
            q == t[3]
            m1 == t[4]
        IN
            << x, y, << m1, q >> >>
    Value(t, F(_)) ==
        LET
            x == t[1]
            y == t[2]
            m2 == t[3]
            (* initial arguments *)
            argsi == << x, y, g(x, y), r >>
            init == F(argsi)
            (* arguments when changes occur *)
            args == << x, y, m2[2], m2[1] >>
            later == F(args)
        IN
            IF m2 = << r >>
                THEN init
                ELSE later
    fx2 ==
        LET
            OldDom == DOMAIN fx1
            R == { repack(t):  t \in OldDom }
            S == R \cup { << x, y, << r >> >> }
            F(args) == fx1[args]
        IN
            [t \in S |-> Value(z, F)]
    fm2 ==
        LET
            OldDoms == (DOMAIN fm1) \cup DOMAIN fq
                (* here \cup is necessary *)
            R == { repack(t):  t \in OldDoms }
            S == R \cup { << x, y, << r >> >> }
            F(args) == << fm1[args], fq[args] >>
        IN
            [t \in S |-> Value(z, F)]
<1>3. \AA x, y:
        \/ ~ \EE m2:
            LET
                args == << x, y, m2 >>
            IN
                /\ \E q:  I(x, y, q)
                /\ m2 = << r >>
                /\ [] /\ x' = fx2[args]
                      /\ m2' = fm2[args]
        \/ \EE q:  Phi(x, y, q)
    <2>1. ASSUME VARIABLE x, VARIABLE y
          PROVE
              \/ ~ \EE m2:
                    LET
                        args == << x, y, m2 >>
                    IN
                        /\ \E q:  I(x, y, q)
                        /\ m2 = << r >>
                        /\ [] /\ x' = fx2[args]
                              /\ m2' = fm2[args]
              \/ \EE q, m:
                  LET
                      args == << x, y, q, m >>
                  IN
                      /\ I(x, y, q)
                      /\ m = r
                      /\ [] /\ x' = fx1[args]
                            /\ q' = fq[args]
                            /\ m' = fm1[args]
        <3> DEFINE A ==
                \EE m2:
                    LET
                        args == << x, y, m2 >>
                    IN
                        /\ \E q:  I(x, y, q)
                        /\ m2 = << r >>
                        /\ [] /\ x' = fx2[args]
                              /\ m2' = fm2[args]
        <3>1. \/ ~ A
              \/ \EE m2:
                        (* q is determined by history *)
                    /\ \EE q:  /\ q = g(x, y)
                               /\ [](q' = m2[2]')
                        (* m is determined by history *)
                    /\ \EE m:  /\ m = r
                               /\ [](m' = m2[1]')
                        (* from A *)
                    /\ LET
                            args == << x, y, m2 >>
                       IN
                            /\ \E q:  I(x, y, q)
                            /\ m2 = << r >>
                            /\ [] /\ x' = fx2[args]
                                  /\ m2' = fm2[args]
            BY RawHistoryDeterminedVar!HistoryExists
        <3>2. \/ ~ A
              \/ \EE m2, q, m:
                LET
                    args == << x, y, m2 >>
                IN
                    /\ \E z:  I(x, y, z)  (* avoid synonymy with q *)
                    /\ q = g(x, y)
                    /\ m = r
                    /\ [] /\ q' = m2[2]'
                          /\ m' = m2[1]'
                    /\ m2 = << r >>
                    /\ [] /\ x' = fx2[args]
                          /\ m2' = fm2[args]
            BY <3>1
        <3>3. \/ ~ A
              \/ \EE m2, q, m:
                LET
                    argsi == << x, y, g(x, y), r >>
                    args == << x, y, m2[2], m2[1] >>
                IN
                    /\ I(x, y, q)
                    /\ q = g(x, y)
                    /\ m = r
                    /\ [] /\ q' = m2[2]'
                          /\ m' = m2[1]'
                    /\ m2 = << r >>
                    /\ [] /\ x' = IF m2 = << r >>
                                THEN fx1[argsi]
                                ELSE fx1[args]

                          /\ m2' = IF m2 = << r >>
                                THEN << fm1[argsi], fq[argsi] >>
                                ELSE << fm1[args], fq[args] >>
            BY <3>2 DEF g, fx2, fm2
        <3>4. \/ ~ /\ m2 = << r >>
                   /\ [] \E a, b:  m2' = << a, b >>
              \/ [](m2' # << r >>)
            OBVIOUS
        <3>5. \/ ~ A
              \/ \EE m2, q, m:
                LET
                    argsi == << x, y, q, m >>
                    args == << x, y, q, m >>
                IN
                    /\ I(x, y, q)
                    /\ q = g(x, y)
                    /\ m = r
                    /\ [] /\ q' = m2[2]'
                          /\ m' = m2[1]'
                    /\ m2 = << r >>
                    /\ [] /\ x' = IF m2 = << r >>
                                THEN fx1[argsi]
                                ELSE fx1[args]
                          /\ m2' = IF m2 = << r >>
                                THEN << fm1[argsi], fq[argsi] >>
                                ELSE << fm1[args], fq[args] >>
            BY <3>3, <3>4
        <3>6. \/ ~ A
              \/ \EE m2, q, m:
                LET
                    args == << x, y, q, m >>
                IN
                    /\ I(x, y, q)
                    /\ m = r
                    /\ [] /\ q' = m2[2]'
                          /\ m' = m2[1]'
                    /\ [] /\ x' = fx1[args]
                          /\ m2' = << fm1[args], fq[args] >>
            BY <3>5
        <3>7. \/ ~ A
              \/ \EE q, m:
                LET
                    args == << x, y, q, m >>
                IN
                    /\ I(x, y, q)
                    /\ m = r
                    /\ [] /\ x' = fx1[args]
                          /\ q' = fq[args]
                          /\ m' = fm1[args]
            BY <3>6
        <3> QED
            BY <3>7 DEF A
    <2> QED
        BY <1>2, <2>1
<1>4. \E fx, fm, m0:
        /\ IsAFunction(fx)
        /\ IsAFunction(fm)
        /\ \AA x, y:
            \/ ~ \EE m:
                LET
                    args == << x, y, m >>
                IN
                    /\ Init(x, y)
                    /\ m = m0
                    /\ [] /\ x' = fx[args]
                          /\ m' = fm[args]
            \/ \EE q:  Phi(x, y, q)
    <2>1. /\ IsAFunction(fx2)
          /\ IsAFunction(fm2)
        BY DEF fx2, fm2
    <2> QED
        BY <2>1, <1>3 DEF Init
<1> QED
    BY <1>4 DEF IsRealizable

--------------------------------------------------------------------------------
(* Revealing history-determined variables leaves realizability unchanged. *)

(* Caution:

1. The next value y; of the environment variable y should not occur in
   fnext if we want a Moore implementation.

2. h should be history-determined by functions. Functions instead of
   operators are necessary for a straightforward proof that a function that
   controls the value of h does exist.

   Otherwise we would have to argue in terms of what values are relevant
   to realizability, which is complicated, and likely requires reasoning
   outside the object language (an independence-like proof).
*)


PROPOSITION UnhidingHistoryFuncPreservesRealizability ==
    ASSUME
        CONSTANT finit, CONSTANT fnext,
        CONSTANT Init(_, _),
        TEMPORAL Phi(_, _, _),
        /\ LET
               PhiH(x, y) == \EE h:  Phi(x, y, h)
           IN
               IsRealizable(Init, PhiH)
        /\ LET
               History(h, x, y) ==
                   /\ h = finit[x, y]
                   /\ [](h' = fnext[h, x, y, x'])
           IN
               \AA x, y, h:  Phi(x, y, h)  =>  History(h, x, y)
    PROVE
        LET I(x, y, h) == Init(x, y) /\ (h = finit[x, y])
        IN IsRealizable(I, Phi)
PROOF
<1> DEFINE
    I(x, y, h) == Init(x, y) /\ (h = finit[x, y])
    History(h, x, y) == /\ h = finit[x, y]
                        /\ [](h' = fnext[h, x, y, x'])
<1>1. PICK fx, fm, m0:
        /\ IsAFunction(fx)
        /\ IsAFunction(fm)
        /\ \AA x, y:
            \/ ~ \EE m:
                LET
                    args == << x, y, m >>
                IN
                    /\ Init(x, y)
                    /\ m = m0
                    /\ [] /\ x' = fx[args]
                          /\ m' = fm[args]
            \/ \EE h:  Phi(x, y, h)
    BY DEF IsRealizable
        (* and UnhidingHistoryFuncPreservesRealizability!assumption *)
<1>2. \AA x, y:
        \/ ~ \EE m:  Realization(Init, m0, fx, fm)
        \/ \EE q:  Phi(x, y, q)
    BY <1>1
<1>3. \AA x, y, h:
        \/ ~ /\ \EE m:  Realization(Init, m0, fx, fm)
             /\ History(h, x, y)
        \/ /\ \EE q:  Phi(x, y, q)
           /\ History(h, x, y)
    BY <1>2
<1>4. \AA x, y, h:
        \/ ~ \EE m:  /\ Realization(Init, m0, fx, fm)
                     /\ History(h, x, y)
        \/ Phi(x, y, h)
    <2>1. \AA x, y, q:  Phi(x, y, q)  =>  History(q, x, y)
        OBVIOUS  (* BY UnhidingHistoryFuncPreservesRealizability!assumption *)
    <2>2. \AA x, y, h:
            \/ ~ \EE m:  /\ Realization(Init, m0, fx, fm)
                         /\ History(h, x, y)
            \/ \EE q:  /\ Phi(x, y, q)
                       /\ History(q, x, y)
                       /\ History(h, x, y)
        BY <1>3, <2>1
    <2>3. \AA x, y, q, h:
            \/ ~ /\ History(q, x, y)
                 /\ History(h, x, y)
            \/ [](q = h)
        <3>1. ASSUME VARIABLE x, VARIABLE y, VARIABLE h, VARIABLE q
              PROVE
                    /\ History(q, x, y) <=>
                            /\ q = finit[x, y]
                            /\ [](q' = fnext[q, x, y, x']
                    /\ History(h, x, y) <=>
                            /\ h = finit[x, y]
                            /\ [](h' = fnext[h, x, y, x'])
            BY DEF History
        <3> DEFINE
            H == History(q, x, y) /\ History(h, x, y)
            Inv == q = h
            Next(u) == u' = fnext[u, x, y, x']
        <3>2. H  =>  Inv
            <4>1. H  =>  (q = finit[x, y])
                BY <3>1 DEF H
            <4>2. H  =>  (h = finit[x, y])
                BY <3>1 DEF H
            <4> QED
                BY <4>1, <4>2 DEF Inv
        <3>3. (Inv /\ Next(q) /\ Next(h))  =>  Inv'
            <4>1. Inv => (<< q, x, y, x' >> = << h, x, y, x' >>)
                BY DEF Inv
            <4>2. \/ ~ /\ << q, x, y, x' >> = << h, x, y, x' >>
                       /\ Next(q) /\ Next(h)
                  \/ q' = h'
                BY DEF Next
            <4>3. (q' = h')  <=> Inv'
                BY DEF Inv
            <4> QED
                BY <4>1, <4>2, <4>3
        <3> QED
            BY <3>1, <3>2, <3>3, RuleRawINV1
    <2>4. \AA x, y, h:
            \/ ~ \EE m:  /\ Realization(Init, m0, fx, fm)
                         /\ History(h, x, y)
            \/ \EE q:
                /\ Phi(x, y, q)
                /\ [](q = h)
                /\ History(q, x, y)
                /\ History(h, x, y)
        BY <2>2, <2>3
        (* in effect flexible substitution *)
    <2>5. \AA x, y, h:
            \/ ~ \EE m:  /\ Realization(Init, m0, fx, fm)
                         /\ History(h, x, y)
            \/ \EE q:
                /\ Phi(x, y, h)
                /\ [](q = h)
        BY <2>4
    <2> QED
        BY <2>5
<1>5. \AA x, y, h:
        (\EE m:  /\ Realization(Init, m0, fx, fm)
                 /\ History(h, x, y))
        <=> \EE m:
            LET
                args == << x, y, m >>
            IN
                /\ Init(x, y)
                /\ h = finit[x, y]
                /\ m = m0
                /\ [] /\ x' = fx[args]
                      /\ h' = fnext[h, x, y, x']
                      /\ m' = fm[args]
    BY <1>1 DEF Realization, History
<1> DEFINE
    Dom == (DOMAIN fx) \cup (DOMAIN fm)
    Proj(T, i) == {t[i]:  t \in T}
    DomX == Proj(Dom, 1) \cup Proj(DOMAIN fnext, 2)
    DomY == Proj(Dom, 2) \cup Proj(DOMAIN fnext, 3)
    DomM == Proj(Dom, 3)
    DomH == Proj(DOMAIN fnext, 1)
    (* repacking *)
    S == (DomX \X DomH) \X DomY \X DomM
    F(f, t) == LET x == t[1][1]  y == t[2]  m == t[3]
               IN f[x, y, m]
    G(f, t) == LET x == t[1][1]  h == t[1][2]  y == t[2]  m == t[3]
               IN f[h, x, y, fx[x, y, m]]
    fx2 == [t \in S |-> F(fx, t)]
    fm2 == [t \in S |-> F(fm, t)]
    fh2 == [t \in S |-> G(fnext, t)]
<1>6. \AA x, y, h:
        (\EE m:  /\ Realization(Init, m0, fx, fm)
                 /\ History(h, x, y))
        <=> \EE m:
            LET
                args == << << h, x >>, y, m >>
            IN
                /\ I(x, y, h)
                /\ m = m0
                /\ [] /\ x' = fx2[args]
                      /\ h' = fh2[args]
                      /\ m' = fm2[args]
    BY <1>5 DEF I, fx2, fm2, fh2
<1> QED
    BY <1>4, <1>6 DEF IsRealizable DEF I


--------------------------------------------------------------------------------
(* Combining the two previous directions into one theorem. *)

THEOREM RealizingHistory ==
    ASSUME
        CONSTANT finit, CONSTANT fnext,
        CONSTANT Init( _, _),
        TEMPORAL Phi(_, _, _),
        LET
               History(h, x, y) ==
                   /\ h = finit[x, y]
                   /\ [](h' = fnext[h, x, y, x'])
           IN
               \AA x, y, h:  Phi(x, y, h)  =>  History(h, x, y)
    PROVE
        LET
            I(x, y, h) == Init(x, y) /\ (h = finit[x, y])
            PhiH(x, y) == \EE h:  Phi(x, y, h)
        IN
            IsRealizable(I, Phi)  <=>  IsRealizable(Init, PhiH)
    PROOF
    BY HidingHistoryPreservesRealizability,
        UnhidingHistoryFuncPreservesRealizability

================================================================================
