---------------------------- MODULE Representation -----------------------------
(* A safety formula []Next in RTLA+ can be unsatisfiable even when Next is.
This cannot happen with the TLA+ formula [][Next]_v, because deadends cannot
form. Deadends return when conjoining a liveness formula to [][Next]_v.

In other words, there is no such thing as an unsatisfiable TLA+ formula of
the form [][Next]_v (or Init /\ [][Next]_v whenever Init is satisfiable).

Conjoining an initial condition Init to [][Next]_v preserves information
present in Init and Next (at least that information which is essential
when taking steps forward, which is what matters for RawWhilePlus).

Conjoining a livevess formula to the safety formula Init /\ [][Next]_v
destroys information, in the sense that the resulting property is
representable by multiple canonical formulas. Among these canonical formulas
are some whose subformulas Init, Next, Liveness lead to different
RawWhilePlus properties.


Author:  Ioannis Filippidis


References
==========

[1] L. Lamport, "Proving possibility properties", TCS, 1998
    10.1016/S0304-3975(98)00129-7

--------------------------------------------------------------------------------
Copyright 2016-2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS TemporalLogic, TLASemantics


(* Any safety property is machine-closed with respect to TRUE [1, Prop.3]. *)
PROPOSITION
    ASSUME
        STATE Init,
        ACTION Next
    PROVE
        Cl(Init /\ [][Next]_v /\ []<>TRUE)
        <=> Init /\ [][Next]_v
    PROOF
    <1> DEFINE
        A == Init /\ [][Next]_v /\ []<>TRUE
        B == Init /\ [][Next]_v
    <1>1. TRUE <=> []<>TRUE
        BY PTL
    <1>2. A <=> B
        BY <1>1
    <1>3. Cl(A) <=> Cl(B)
        BY <1>2
    <1>4. Cl(B) <=> B
        (* B is a safety property. *)
    <1> QED
        BY <1>3, <1>4


--------------------------------------------------------------------------------
(* RTLA+ results. *)


(* In RTLA+ a weaker action yields weaker safety. *)
PROPOSITION WeakerActionRTLA ==
    ASSUME
        ACTION A1, ACTION A2,
        A1 => A2
    PROVE
        ([] A1)  =>  ([] A2)
    PROOF
    <1>1. \A s1, s2:  (IsAState(s1) /\ IsAState(s2)) =>
            << s1, s2 >>[[ A1 ]]  =>  << s1, s2 >>[[ A2 ]]
    <1>2. SUFFICES
            ASSUME
                NEW sigma, IsABehavior(sigma),
                sigma |= [] A1
            PROVE
                sigma |= [] A2
    <1>3. ASSUME NEW n \in Nat
          PROVE << sigma[n], sigma[n + 1] >>[[ A1 ]]
        BY <1>2
    <1>4. ASSUME NEW n \in Nat
          PROVE << sigma[n], sigma[n + 1] >>[[ A2 ]]
        <2>1. /\ IsAState(sigma[n])
              /\ IsAState(sigma[n + 1])
            BY <1>2, <1>4 DEF IsABehavior
        <2> QED
            BY <1>1, <1>3, <2>1
    <1> QED
        BY <1>2, <1>4


(* In RTLA+ equal actions yield same safety. *)
COROLLARY EquivActionsRTLA ==
    ASSUME
        ACTION A1, ACTION A2,
        A1 <=> A2
    PROVE
        ([] A1)  <=>  ([] A2)
    PROOF
    <1>1. ([] A1)  =>  ([] A2)
        BY WeakerActionRTLA
    <1>2. ([] A2)  =>  ([] A1)
        BY WeakerActionRTLA
    <1> QED
        BY <1>1, <1>2


(* The converse of the previous proposition does not hold, due to deadends.
This RTLA+ fact corresponds in TLA+ to the multiplicity of representations
of a property as a conjunction of safety and liveness.
*)
PROPOSITION RawTails ==
    PROVE
        LET
            A1 == FALSE
            A2 == (x = 1) /\ (x' = 2)
        IN
            /\ ([] A1)  <=>  ([] A2)
            /\ ~ |= A1 <=> A2
    PROOF
    <1> DEFINE
        A1 == FALSE
        A2 == (x = 1) /\ (x' = 2)
    <1>1. ~ |= A1 <=> A2
        <2>1. SUFFICES
                \E s, t:  /\ IsAState(s)
                          /\ IsAState(t)
                          /\ << s, t >>[[ A2 /\ ~ A1 ]]
        <2>2. PICK s:  IsAState(s) /\ s[[x]] = 1
        <2>3. PICK t:  IsAState(t) /\ t[[t]] = 2
        <2> QED
            BY <2>2, <2>3 DEF A1, A2  (* goal from <2>1 *)
    <1>2. ([] A1)  <=>  ([] A2)
        <2>1. ([] A1)  =>  ([] A2)
            <3>1. ~ ([] A1)
                BY DEF A1
            <3> QED
                BY <3>1, PTL
        <2>2. ([] A2)  =>  ([] A1)
            <3>1. SUFFICES ~ [] A2  (* A2 leads to a deadend. *)
            <3>2. SUFFICES
                    ASSUME \E sigma:  /\ IsABehavior(sigma)
                                      /\ [] A2
                    PROVE FALSE
            <3>3. PICK sigma:  /\ IsABehavior(sigma)
                               /\ sigma |= [] A2
                BY <3>2
            <3>4. /\ sigma[0][[x]] = 1
                  /\ sigma[1][[x]] = 2
                <4>1. << sigma[0], sigma[1] >>[[ (x = 1) /\ (x' = 2) ]]
                    BY <3>3 DEF A2
                <4> QED
                    BY <4>1
            <3>5. ~ << sigma[1], sigma[2] >>[[ A2 ]]
                BY <3>4 DEF A2
            <3> QED
                BY <3>3, <3>5  (* goal from <3>2 *)
        <2> QED
            BY <2>1, <2>2
    <1> QED
        BY <1>1, <1>2


(* In RTLA+ equivalent initial conditions and actions yield the same
safety property.
*)
COROLLARY
    ASSUME
        STATE I1, STATE I2,
        ACTION A1, ACTION A2,
        I1 <=> I2,
        A1 <=> A2
    PROVE
        (I1 /\ [] A1)  <=>  (I2 /\ [] A2)
    PROOF
    <1>1. ([] A1)  <=>  ([] A2)
        BY EquivActionsRTLA
    <1>2. (I1 /\ [] A1)  <=>  (I1 /\ [] A2)
        BY <1>1
    <1>3. I1 <=> I2
        OBVIOUS
    <1> QED
        BY <1>2, <1>3

--------------------------------------------------------------------------------
(* TLA+ results. *)


(* Similar to the previous corollary, but in TLA+. *)
PROPOSITION
    ASSUME
        STATE I1, STATE I2, STATE v,
        ACTION A1, ACTION A2,
        I1 <=> I2,
        A1 <=> A2
    PROVE
        (I1 /\ [][A1]_v)  <=>  (I2 /\ [][A2]_v)
    OBVIOUS


(* Two equivalent tails are defined by actions with equivalent
nonstuttering parts.
*)
PROPOSITION InvertingTails ==
    ASSUME
        STATE v,
        ACTION A1, ACTION A2,
        ([][A1]_v)  <=>  ([][A2]_v)
    PROVE
        << A1 >>_v  <=>  << A2 >>_v
    PROOF
    <1>1. SUFFICES
            ASSUME
                NEW s1, NEW s2, IsAState(s1), IsAState(s2),
                /\ << s1, s2 >>[[ << A1 >>_v ]]
                /\ ~ << s1, s2 >>[[ << A2 >>_v ]]
            PROVE FALSE
        BY Semantics
    <1> DEFINE sigma == [n \in Nat |-> IF n = 0 THEN s1 ELSE s2]
    <1>2. IsABehavior(sigma)
        BY DEF sigma, IsABehavior
    <1>3. sigma |= [][A1]_v
        <2>1. << A >>_v  =>  [A]_v
            OBVIOUS
        <2> QED
            BY <1>1, <1>2, <2>1 DEF sigma
    <1>4. ~ (sigma |= [][A2]_v)
        <2>1. sigma[0] # sigma[1]
            <3>1. << sigma[0], sigma[1] >>[[ << A1 >>_v ]]
                BY <1>1 DEF sigma
            <3> QED
                BY <3>1
        <2>2. ~ << sigma[0], sigma[1] >>[[ v' = v ]]
            BY <1>3
        <2>3. ~ << sigma[0], sigma[1] >>[[ << A1 >>_v \/ (v' = v) ]]
            BY <2>2, <1>1 DEF sigma
        <2>4. ~ << sigma[0], sigma[1] >>[[ [A1]_v ]]
            <3>1. ((v' = v) /\ A1)  =>  (v' = v)
            <3>2. A1 <=> \/ A1 /\ (v' = v)
                         \/ << A1 >>_v
            <3>3. [A1]_v <=> \/ << A1 >>_v
                             \/ v' = v
                BY <3>1, <3>2
            <3> QED
                BY <2>3, <3>3
        <2> QED
            BY <2>4
    <1>5. (sigma |= [][A1]_v)  <=>  (sigma |= [][A2]_v)
        BY <1>2
    <1> QED
        <1>3, <1>4, <1>5  (* goal from <1>1 *)


LEMMA BoxActionEnabled ==
    ASSUME
        STATE v, ACTION A
    PROVE
        ENABLED [A]_v
    <1>1. [A]_v <=> (A \/ (v' = v))
        OBVIOUS
    <1>2. ENABLED (v' = v)
        <2>1. SUFFICES
                ASSUME NEW s1, IsAState(s1)
                PROVE \E s2:  /\ IsAState(s2)
                              /\ << s1, s2 >>[[ v' = v ]]
            OBVIOUS
        <2> DEFINE s2 == s1
        <2>3. IsAState(s2)
            BY <2>1 DEF s2
        <2>4. s2[[ v ]] = s1[[ v ]]
            BY <2>1, <2>3 DEF s2
        <2>5. << s1, s2 >>[[ v' = v ]]
            BY <2>4
        <2> QED
            BY <2>3, <2>5
    <1>3. ASSUME ACTION P, ACTION Q
          PROVE (ENABLED P) => ENABLED (P \/ Q)
        OBVIOUS
    <1>4. ENABLED (A \/ (v' = v))
        BY <1>2, <1>3
    <1> QED
        BY <1>1, <1>4


(* In the presence of an initial condition, the actions of two state machines
are equivalent only at reachable states, but may differ elsewhere. *)
PROPOSITION InvertingStateMachines ==
    ASSUME
        STATE I1, STATE I2, STATE v,
        ACTION A1, ACTION A2,
        LET
            SM1 == I1 /\ [][A1]_v
            SM2 == I2 /\ [][A2]_v
        IN
            SM1 <=> SM2
    PROVE
        LET
            SM1 == I1 /\ [][A1]_v
        IN
            /\ I1 <=> I2
            /\ SM1 => [][A1 /\ A2]_v
    <1> DEFINE
        SM1 == I1 /\ [][A1]_v
        SM2 == I2 /\ [][A2]_v
    <1>1. SM1 <=> SM2
        OBVIOUS  (* BY InvertingStateMachines *)
    <1>2. I1 <=> I2
        <2>1. I1 => I2
            <3>1. SUFFICES
                    ASSUME NEW s, IsAState(s), s[[ I1 ]]
                    PROVE s[[ I2 ]]
                (* BY STATE I1, STATE I2 *)
            <3> DEFINE sigma == Stutter(s)
            <3>2. IsABehavior(sigma)
                BY <3>1 DEF signa, Stutter, IsABehavior
            <3>3. sigma |= SM1
                <4>1. sigma |= I1
                    <5>1. sigma[0] = s
                        BY DEF sigma, Stutter
                    <5>2. s[[ I1 ]]
                        BY <3>1
                    <5> QED
                        BY <5>1, <5>2
                <4>2. sigma |= [][A1]_v
                    <5>1. sigma |= [][FALSE]_v
                        BY DEF sigma, Stutter
                    <5> QED
                        BY <4>3
                <4> QED
                    BY <4>1, <4>2 DEF SM1
            <3>4. sigma |= SM2
                BY <3>3, <1>1
            <3>5. sigma |= I2
                BY <3>4 DEF SM2
            <3> QED
                <4>1. sigma[0][[ I2 ]]
                    BY <3>5, <3>2  (* and STATE I2 *)
                <4>2. sigma[0] = s
                    BY DEF sigma, Stutter
                <4> QED
                    BY <4>1, <4>2
        <2>2. I2 => I1
            (* PROOF similar to that of <2>1. *)
        <2> QED
            BY <2>1, <2>2
    <1>3. SM1  =>  (SM1 /\ SM2)
        BY <1>1
    <1>4. SM1  =>  ([][A1]_v /\ [][A2]_v)
        BY <1>3 DEF SM1, SM2
    <1>5. SM1  =>  [][A1 /\ A2]_v
        BY <1>4
    <1> QED
        BY <1>2, <1>5

================================================================================
