---------------------------- MODULE TLASemantics -------------------------------
(* Some notions about TLA+ in the metatheory.

The TLA+ fragment of constant operators servers as the metatheory.
So the metatheory is ZF + CHOOSE + functions, similarly to [2, Chapter 16].


References
==========

[1] L. Lamport, The temporal logic of actions, TOPLAS, 1994
    10.1145/177492.177726

[2] L. Lamport, Specifying systems, Addison-Wesley, 2002

[3] M. Abadi and L. Lamport, "An old-fashioned recipe for real time",
    TOPLAS, 1994, 10.1145/186025.186058
*)
--------------------------------------------------------------------------------
EXTENDS Naturals, NaturalsInduction
CONSTANT VarNames  (* META *) (* Set of all variable names [2, p.311]. *)


--------------------------------------------------------------------------------
(* Axiomatic definition of states and behaviors. *)

IsAFunction(f) ==
    f = [ x \in DOMAIN f |-> f[x] ]

IsAState(s) ==
    /\ IsAFunction(s)
    /\ DOMAIN s = VarNames

IsABehavior(b) ==
    /\ IsAFunction(b)
    /\ DOMAIN b = Nat
    /\ \A n \in Nat:  IsAState(b[n])

NatGeq(n) == {r \in Nat:  r >= n}

(* The finite behavior made of the first n states of behavior sigma. *)
Prefix(sigma, n) == [i \in 0..(n - 1) |-> sigma[i]]
(* The infinite behavior that starts after the first n states of sigma. *)
Suffix(sigma, n) == [i \in NatGeq(n) |-> sigma[i]]

(* More formally, in this metatheoretic statement H should be
a string that is a TLA+ formula.
*)
PrefixSat(sigma, n, H) ==
    \E tau:  /\ IsABehavior(tau)
             /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
             /\ tau |= H

--------------------------------------------------------------------------------
(* If a behavior prefix can be extended to satisfy a propert P, then the
same prefix can be extended to satisfy any property Q weaker than P.
*)
THEOREM PrefixSatImp ==
    ASSUME
        NEW n,  (* The condition n \in Nat is unused, so not assumed. *)
        NEW sigma, IsABehavior(sigma),
        TEMPORAL P, TEMPORAL Q,
        P => Q
    PROVE
        PrefixSat(sigma, n, P)
            => PrefixSat(sigma, n, Q)
    PROOF
    <1>1. SUFFICES
            ASSUME PrefixSat(sigma, n, P)
            PROVE PrefixSat(sigma, n, Q)
        OBVIOUS
    <1>2. PICK tau:  /\ IsABehavior(tau)
                     /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                     /\ tau |= P
        BY <1>1 DEF PrefixSat
    <1>3. tau |= Q
        <2>1. P => Q
            OBVIOUS  (* BY PrefixSatImp!assumption *)
        <2> QED
            BY <1>2, <2>1
    <1>4. /\ IsABehavior(tau)
          /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
          /\ tau |= Q
        BY <1>2, <1>3
    <1> QED
        BY <1>4 DEF PrefixSat  (* goal from <1>1 *)


(* The first n states of tau and sigma match. *)
THEOREM PrefixSatAsSamePrefix ==
    ASSUME
        NEW sigma, IsABehavior(sigma),
        NEW n \in Nat,
        TEMPORAL H
    PROVE
        PrefixSat(sigma, n, H)
        <=> \E tau:  /\ IsABehavior(tau)
                     /\ Prefix(tau, n) = Prefix(sigma, n)
                     /\ tau |= H
    PROOF
    <1>1. SUFFICES
            ASSUME NEW tau, IsABehavior(tau)
            PROVE (Prefix(tau, n) = Prefix(sigma, n))
                    <=> \A i \in 0..(n - 1):  tau[i] = sigma[i]
        BY DEF PrefixSat
    <1> DEFINE
            SamePrefix == Prefix(tau, n) = Prefix(sigma, n)
            TauPrefix == [i \in 0..(n - 1) |-> tau[i]]
            SigmaPrefix == [i \in 0..(n - 1) |-> sigma[i]]
    <1>2. SamePrefix <=> (TauPrefix = SigmaPrefix)
        BY DEF Prefix, SamePrefix, TauPrefix, SigmaPrefix
    <1>3. SamePrefix <=> /\ DOMAIN TauPrefix = DOMAIN SigmaPrefix
                         /\ \A i \in DOMAIN TauPrefix:
                                TauPrefix[i] = SigmaPrefix[i]
        BY <1>2
    <1>4. DOMAIN TauPrefix = DOMAIN SigmaPrefix
        BY DEF TauPrefix, SigmaPrefix
    <1>5. \A i \in DOMAIN TauPrefix:  /\ TauPrefix[i] = tau[i]
                                      /\ SigmaPrefix[i] = sigma[i]
        BY DEF TauPrefix, SigmaPrefix
    <1>6. SamePrefix
            <=> \A i \in DOMAIN TauPrefix:  tau[i] = sigma[i]
        BY <1>3, <1>4, <1>5
    <1>7. SamePrefix
            <=> \A i \in 0..(n - 1):  tau[i] = sigma[i]
        BY <1>6
    <1> QED
        BY <1>7 DEF SamePrefix


PROPOSITION SamePrefixImpliesPrefixSatToo ==
    ASSUME
        TEMPORAL H,
        NEW n \in Nat,
        NEW sigma, IsABehavior(sigma),
        NEW eta, IsABehavior(eta),
        /\ Prefix(sigma, n) = Prefix(eta, n)
        /\ PrefixSat(sigma, n, H)
    PROVE
        PrefixSat(eta, n, H)
    <1>1. PICK tau:  /\ IsABehavior(tau)
                     /\ Prefix(tau, n) = Prefix(sigma, n)
                     /\ tau |= H
        BY PrefixSatAsSamePrefix
    <1>2. Prefix(sigma, n) = Prefix(eta, n)
        OBVIOUS  (* BY SamePrefixImpliesPrefixSatToo!assumption *)
    <1>3. /\ IsABehavior(tau)
          /\ Prefix(tau, n) = Prefix(eta, n)
          /\ tau |= H
        BY <1>1, <1>2
    <1> QED
        BY <1>3, PrefixSatAsSamePrefix


(* If the first n states of two behaviors are the same,
then PrefixSat for n, H has the same value for both behaviors.
*)
THEOREM EquivPrefixSatIfSamePrefix ==
    ASSUME
        TEMPORAL H,
        NEW n \in Nat,
        NEW sigma, IsABehavior(sigma),
        NEW eta, IsABehavior(eta),
        Prefix(sigma, n) = Prefix(eta, n)
    PROVE
        PrefixSat(sigma, n, H) <=> PrefixSat(eta, n, H)
    <1>1. PrefixSat(sigma, n, H) => PrefixSat(eta, n, H)
        BY SamePrefixImpliesPrefixSatToo
    <1>2. PrefixSat(eta, n, H) => PrefixSat(sigma, n, H)
        BY SamePrefixImpliesPrefixSatToo
    <1> QED
        BY <1>1, <1>2

--------------------------------------------------------------------------------
(* The "while" operator `^$-\hspace{-.18em}\triangleright$^'
[3, Sec. A4 on p. A-2]
*)
sigma |= While(A, G) ==
    /\ \A n \in Nat:  PrefixSat(sigma, n, A) => PrefixSat(sigma, n, G)
    /\ sigma |= A => G

--------------------------------------------------------------------------------
(* The "while plus" operator -+-> *)

(* Semantic form of stepwise implication.

For the safety properties A == []EnvNext and G == []SysNext, the semantic
operator corresponds to the syntactic operator as follows

sigma |= StepwiseImpl(EnvNext, SysNext)  <=>  PrefixPlusOne(sigma, A, G)
*)
PrefixPlusOne(sigma, A, G) ==
    \A n \in Nat:  PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)

(* The while-plus operator [2, p.316]. *)
sigma |= A -+-> G ==
    /\ PrefixPlusOne(sigma, A, G)
    /\ sigma |= A => G

(* This theorem expands the definition of -+->. *)
THEOREM WhilePlusProperties ==
    ASSUME
        NEW sigma, IsABehavior(sigma),
        TEMPORAL A, TEMPORAL G,
        A -+-> G
    PROVE
        /\ sigma |= A => G
        /\ \A n \in Nat:
            PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
    BY DEF -+->, PrefixPlusOne

(* We can view -+-> (other stepwise operators too) as an infinite conjunction:

sigma |= A -+-> G <=>
    /\ PrefixSat(sigma, 0, A)  =>  PrefixSat(sigma, 1, G)
    /\ PrefixSat(sigma, 1, A)  =>  PrefixSat(sigma, 2, G)
    ...
    ...
    /\ A => G
*)


(* Metatheoretic definition that means

{ var \in VarNames:  ~ |= F <=> (\AA var:  F) }

It seems that a semantic definition needs to mention all other
variables, thus an infinity of strings. A syntactic definition
can be given for any (finite length) formula by simply parsing it.
*)
VariablesOf(formula) == {
    var \in VarNames:  \E sigma, tau:
        /\ IsABehavior(sigma)
        /\ IsABehavior(tau)  (* tau is same as in \EE DEF *)
        /\ RefinesUpToVar(tau, sigma, var)
        /\ ~ ((sigma |= formula) <=> ~ (tau |= ~ formula)) }

--------------------------------------------------------------------------------
(* Stutter at state forever. *)
Stutter(state) == [n \in Nat |-> state]

(* Keep states 0..k and stutter state k indefinitely. *)
StutterAfter(sigma, n) == [i \in Nat |-> IF i < n THEN sigma[i]
                                                  ELSE sigma[n]]

THEOREM StutterAfterIsABehavior ==
    ASSUME
        NEW n \in Nat,
        NEW sigma,
        IsABehavior(sigma)
    PROVE
        LET
            eta == StutterAfter(sigma, n)
        IN
            IsABehavior(eta)
    <1> DEFINE eta == StutterAfter(sigma, n)
    <1>1. /\ IsAFunction(eta)
          /\ DOMAIN eta = Nat
        BY DEF eta, StutterAfter
    <1>2. ASSUME NEW i \in Nat
          PROVE IsAState(eta[i])
        <2>1. ASSUME NEW r \in Nat
              PROVE IsAState(sigma[r])
            <3>1. IsABehavior(sigma)
                OBVIOUS  (* BY ASSUME *)
            <3> QED
                BY <2>1 DEF IsABehavior
        <2>2. PICK r \in Nat:  sigma[r] = eta[i]
            <3>1. CASE i < n
                <4>1. sigma[i] = eta[i]
                    BY <3>1 DEF eta, StutterAfter
                <4> QED
                    BY <4>1, <1>2  (* The witness is i. *)
            <3>2. CASE i >= n
                <4>1. sigma[n] = eta[i]
                    BY <3>2 DEF eta, StutterAfter
                <4> QED
                    BY <4>1  (* The witness is n. *)
            <3> QED
                BY <3>1, <3>2, <1>2
        <2>3. IsAState(sigma[r])
            BY <2>1, <2>2
        <2> QED
            BY <2>2, <2>3
    <1> QED
        BY <1>1, <1>2 DEF IsABehavior


THEOREM StutterAfterHasSamePrefix ==
    ASSUME
        NEW n \in Nat,
        NEW k \in Nat,
        k < n,
        NEW sigma,
        IsABehavior(sigma)
    PROVE
        LET
            eta == StutterAfter(sigma, n)
        IN
            eta[k] = sigma[k]
    BY DEF StutterAfter


THEOREM StutteringTail ==
    ASSUME
        NEW n \in Nat,
        NEW k \in Nat,
        k >= n,
        NEW sigma,
        IsABehavior(sigma)
    PROVE
        LET
            eta == StutterAfter(sigma, n)
        IN
            eta[k] = sigma[n]
    BY DEF StutterAfter


THEOREM StutterAfterInit ==
    ASSUME
        NEW n \in Nat,
        NEW sigma,
        IsABehavior(sigma)
    PROVE
        LET
            eta == StutterAfter(sigma, n)
        IN
            eta[0] = sigma[0]
    <1>1. n \in Nat
        OBVIOUS  (* BY StutterAfterInit!assumption *)
    <1>2. CASE 0 < n
        BY StutterAfterHasSamePrefix
    <1>3. CASE 0 >= n
        BY StutteringTail
    <1> QED
        BY <1>1, <1>2, <1>3

--------------------------------------------------------------------------------
(* Metatheoretic statements asserting that P is a TLA+ expression of
a certain level. Used for bookkeeping during hand-written proofs.
*)
IsStateLevel(P) == TRUE
IsTemporalLevel(P) == TRUE

(* Metatheoretic statement asserting that P is a TLA+ formula.
Useful to distinguish from raw TLA+ formulas that aren't TLA+ formulas.
*)
IsATLAPlusFormula(P) == TRUE

--------------------------------------------------------------------------------
(* The stutter-free form `^$\natural \sigma$^' of `^$\sigma$,^' [2, p.316].

If the behavior sigma contains a finite number of nonstuttering steps,
then Natural(sigma) is a finite sequence.
A similar operator in [1] yields an infinite sequence with a stuttering tail.
Either of these definitions can be used to define temporal quantification.
*)
Natural(sigma) ==
    LET
        f[n \in Nat] == IF n = 0 THEN 0
                                 ELSE IF sigma[n] = sigma[n - 1]
                                      THEN f[n - 1]
                                      ELSE f[n - 1] + 1
        S == { f[n]:  n \in Nat }
    IN
        [ n \in S |->
          sigma[CHOOSE i \in Nat:  f[i] = n ] ]

(* `^
The behaviors s and t differ only in the values of variable var.
This is $s =_{var} t$ in [1].
^' *)
EqualUpToVar(s, t, var) ==
    \A n \in Nat:  \A v \in VarNames:  (v # var) => (s[n][v] = t[n][v])

(* `^ $\rho \sim \sigma$ asserts that the sequences $\natural \rho$ and
$\natural \sigma$ are equal [1].
^' *)
Sim(rho, sigma) == Natural(rho) = Natural(sigma)

(* A useful definition, based on [1].
RefinesUpToVar is noncommutative
(though both EqualUpToVar and Sim are commutative).
*)
RefinesUpToVar(tau, sigma, var) ==
    \E rho:  /\ IsABehavior(rho)
             /\ Sim(rho, sigma)
             /\ EqualUpToVar(rho, tau, var)

(* The similarity of sutter-free forms after overwriting the variable `x' in
the behavior `^$\tau$.^' [2, p.316].
*)
SimUpToVar(sigma, tau, var) ==
    LET
        s == Natural(sigma)
        t == Natural(tau)
    IN
        s = [n \in DOMAIN t |-> [t[n] EXCEPT ![var] = s[n][var] ] ]

--------------------------------------------------------------------------------
(* Temporal existential quantification, based on [1]. *)
AXIOM
    sigma |= \EE x:  F <=>
        \E tau:  /\ IsABehavior(tau)
                 /\ RefinesUpToVar(tau, sigma, "x")
                 /\ tau |= F

(* Temporal existential quantification, based on [2] and [3, p. A-2]. *)
AXIOM
    sigma |= \EE x:  F <=>
        \E tau:  /\ IsABehavior(tau)
                 /\ SimUpToVar(sigma, tau, "x")
                 /\ tau |= F

================================================================================
