----------------------------- MODULE TemporalLogic -----------------------------
(* Some definitions about temporal properties.

Author:  Ioannis Filippidis


References
==========

[1] L. Lamport, "Miscellany", 21 April 1991

[2] M. Abadi and S. Merz,
    "On TLA as a logic", Deductive Program Design, 1996

[3] B. Alpern and F.B. Schneider
    "Defining liveness", IPL, 1985
    10.1016/0020-0190(85)90056-0

[4] B. Jonsson and Y.-K. Tsay,
    "Assumption/guarantee specifications in linear-time temporal logic",
    TCS, 1996, 10.1016/0304-3975(96)00069-2

[5] M. Abadi and L. Lamport,
    "Conjoining specifications", TOPLAS, 1995
    10.1145/203095.201069

[6] L. Lamport, "Proving possibility properties", TCS, 1998
    10.1016/S0304-3975(98)00129-7

[7] M. Abadi and L. Lamport, "An old-fashioned recipe for real time",
    TOPLAS, 1994, 10.1145/186025.186058

[8] U. Klein and N. Piterman and A. Pnueli,
    "Effective synthesis of asynchronous systems from GR(1) specifications",
    VMCAI, 2012, 10.1007/978-3-642-27940-9_19
    (Technical report 2011-944 of Courant Inst. of Math. Sciences)

[9] A. Pnueli and R. Rosner,
    "On the synthesis of an asynchronous reactive module",
    ICALP, 1989, 10.1007/BFb0035790

--------------------------------------------------------------------------------
Copyright 2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS TLASemantics, NaturalsInduction

--------------------------------------------------------------------------------
(* Safety and liveness. *)

MustUnstep(b) == /\ b = TRUE
                 /\ [][b' = FALSE]_b
                 /\ <>(b = FALSE)
SamePrefix(b, u, x) == [](b => (u = x))
Front(P(_, _), x, b) == \EE u:  P(u) /\ SamePrefix(b, u, x)

(* A syntactic definition of closure [1].
See also [2, Sec. 5.3] and [4, Sec. 2.1 on p. 52].
*)
Cl(P(_), x) == \AA b:  MustUnstep(b) => Front(P, x, b)
(* A semantic definition of closure [6, Eq. (1) on p. 342] and [7, p. A-2].
The syntactic and semantic definitions of closure are equivalent.
*)
(* sigma |= Cl(P) == \A n \in Nat:  PrefixSat(sigma, n, P) *)


(* Using closure we can define safety and liveness [6, p. 343].
These definitions are equivalent to those that mention violating behaviors
[6, Eq.(2) on p.343].
*)
IsSafety(P(_)) == \AA x:  P(x) <=> Cl(P, x)
IsLiveness(P(_)) == \AA x:  Cl(P, x)

(* Each property is decomposable into safety and liveness [3]. *)
SafetyPart(P(_), x) == Cl(P, x)
LivenessPart(P(_), x) == SafetyPart(P, x) => P(x)  (* [4, Sec. 2.3 on p.54] *)


(* Conjoining the safety and liveness parts yields the property P. *)
THEOREM
    ASSUME
        TEMPORAL P(_), VARIABLE x
    PROVE
        P(x) <=> /\ SafetyPart(P, x)
                 /\ LivenessPart(P, x)
    PROOF
    <1>1. LivenessPart(P, x) <=> (SafetyPart(P, x) => P(x))
        BY DEF LivenessPart
    <1> QED
        BY <1>1


(* For any temporal property P, the safety part is a safety property and
the liveness part is a liveness property.
*)
THEOREM
    ASSUME
        TEMPORAL P(_), VARIABLE x
    PROVE
        LET
            S(u) == SafetyPart(P, u)
            L(u) == LivenessPart(P, u)
        IN
            /\ IsSafety(S, x)
            /\ IsLiveness(L, x)
    PROOF OMITTED


IsMachineClosed(S(_), L(_), x) ==
    LET
        SL(u) == S(u) /\ L(u)
    IN
        S(x) <=> Cl(SL, x)

IsConstant(P) == \E c:  [](P = c)
Canonical(Init, Next, L, v) == Init /\ [][Next]_v /\ L
(* The "state machine" form. *)
SM(Init, Next, v) == Init /\ [][Next]_v


(* Any action comprises of a nonstuttering and a stuttering part. *)
StutteringPart(A, v) == A /\ (v = v')
NonStutteringPart(A, v) == << A >>_v  (* alternative name:  ChangingPart *)
THEOREM
    ASSUME STATE v, ACTION A
    PROVE A <=> \/ StutteringPart(A, v)
                \/ << A >>_v
    OMITTED

THEOREM
    ASSUME STATE v, ACTION A
    PROVE << A >>_v  =>  [A]_v
    OMITTED


(* trick for handling other arities:

    LET P(x) == L(x.p, x.q)
    IN IsLiveness(P)
*)

--------------------------------------------------------------------------------
(* Temporal quantification in raw TLA+ with past. *)


(* Teamporal quantification that preserves stutter-invariance [8, Sec. 2.1].
See also [9] (where behavior indices are not used though).
*)
sigma, i |= \EE x:  F <=>
    \E tau, k:
        /\ IsABehavior(tau)
        /\ k \in Nat
        /\ tau, k |= F
        /\ \E rho:
            LET
                Start(r) == 0..(r - 1)
                End(r) == Nat \ Start(r)
                RhoFront == [n \in 0..Start(k) |-> rho[n]]
                TauFront == [n \in 0..Start(i) |-> tau[n]]
                RhoTail == [n \in End(k) |-> rho[n]]
                TauTail == [n \in End(i) |-> tau[n]]
            IN
                /\ IsABehavior(rho)
                /\ Sim(RhoFront, TauFront)
                /\ Sim(RhoTail, TauTail)
                /\ EqualUpToVar(rho, tau, "x")


(* Temporal quantification that breaks stutter-invariance [8, Sec. 2.1]. *)
sigma, i |= EEE x:  F <=>
    \E tau:  /\ IsABehavior(tau)
             /\ EqualUpToVar(sigma, tau, "x")
             /\ tau, i |= F

--------------------------------------------------------------------------------
(* Properties of closure *)

LEMMA ClosureProperties ==
    ASSUME
        TEMPORAL P, NEW sigma, NEW n \in Nat,
        /\ IsABehavior(sigma)
        /\ sigma |= Cl(P)
    PROVE
        \E tau:  /\ IsABehavior(tau)
                 /\ \A i \in 0..n:  tau[i] = sigma[i]
                 /\ tau |= P
    OMITTED


(* [6, Prop. 1/item 2] and [7, Sec. B3 on p. A-4] *)
LEMMA ClosureIsMonotonic ==
    ASSUME
        VARIABLE x,
        TEMPORAL A(_), TEMPORAL B(_),
        \AA u:  A(u) => B(u)
    PROVE
        Cl(A, x) => Cl(B, x)
    PROOF
    <1>1. Cl(A, x) <=>
            \AA b:  \/ ~ MustUnstep(b)
                    \/ \EE u:
                        /\ A(u)
                        /\ [](b => (u = x))
        BY DEF Cl
    <1> DEFINE
            H == \EE u:  /\ A(u)
                         /\ [](b => (u = x))

            G == \EE u:  /\ B(u)
                         /\ [](b => (u = x))
    <1>2. H <=> \EE u:  /\ A(u)
                        /\ \AA x:  A(x) => B(x)
                        /\ [](b => (u = x))
        BY DEF H  (* and ClosureIsMonotonic!assumption *)
    <1>3. H <=> \EE u:  /\ A(u)
                        /\ A(u) => B(u)
                        /\ [](b => (u = x))
        BY <1>2
    <1>4. H => G
        BY <1>3 DEF G
    <1>5. Cl(A, x) =>
            \AA b:  \/ ~ MustUnstep(b)
                    \/ G
        BY <1>1, <1>4 DEF H, G
    <1>6. Cl(B, x) <=>
            \AA b:  \/ ~ MustUnstep(b)
                    \/ \EE u:
                        /\ B(u)
                        /\ [](b => (u = x))
        BY DEF Cl
    <1> QED
        BY <1>5, <1>6 DEF G


(* If the closure of property P is satisfiable, so is P. *)
LEMMA SATClosureInit ==
    ASSUME
        TEMPORAL P,
        STATE Init, STATE v, ACTION Next,
        NEW sigma,
        /\ |= Cl(P, x) <=> (Init /\ [][Next]_v)
        /\ IsABehavior(sigma)
        /\ sigma |= Init
    PROVE
        \E tau:  /\ IsABehavior(tau)
                 /\ tau[0] = sigma[0]
                 /\ sigma |= P
    PROOF
    <1> DEFINE eta == [n \in Nat |-> sigma[0]]
    <1>2. IsABehavior(eta)
    <1>3. eta |= [][FALSE]_v
        BY DEF eta
    <1>4. eta |= Init
        <2>1. sigma|= Init
            OBVIOUS
        <2> QED
            BY <2>1 DEF eta
    <1>5. eta |= Cl(P)
        <2>1. eta |= Init /\ [][Next]_v
            BY <1>3, <1>4
        <2>2. |= Cl(P) <=> (Init /\ [][Next]_v)
            OBVIOUS
        <2> QED
            BY <2>1, <2>2
    <1>6. PICK beta:  /\ IsABehavior(beta)
                      /\ beta[0] = eta[0]
                      /\ beta |= P
        BY <1>5, ClosureProperties
    <1> QED
        <2>1. eta[0] = sigma[0]
            BY DEF eta
        <2> QED
            BY <1>6, <2>1


LEMMA ClosureOfSafety ==
    ASSUME
        TEMPORAL P,
        IsSafety(P)
    PROVE
        Cl(P) <=> P
    OMITTED


PROPOSITION ClosureAndLiveness ==
    ASSUME
        VARIABLE x, VARIABLE y,
        STATE Init, ACTION Next,
        TEMPORAL L,
        LET
            v == << x, y >>
            S == Init /\ [][Next]_v
        IN
            IsMachineClosed(S, L)
    PROVE
        LET
            v == << x, y >>
            S == Init /\ [][Next]_v
            P == S /\ L
        IN
            P  <=>  (L /\ Cl(P))
    PROOF
    <1> DEFINE
        v == << x, y >>
        S == Init /\ [][Next]_v
        P == S /\ L
    <1>1. S <=> Cl(S /\ L)
        BY DEF IsMachineClosed
    <1>2. (S /\ L) <=> (L /\ Cl(S /\ L))
        BY <1>1
    <1> QED
        BY <1>2


(* If a property P implies a safety property Q,
then the closure of P implies Q.
*)
LEMMA ClosureIsTightestSafety ==
    ASSUME
        TEMPORAL P, TEMPORAL Q,
        /\ IsSafety(Q)
        /\ P => Q
    PROVE
        Cl(P) => Q
    OMITTED

(* The closure of a property P is the tightest safety property that P implies
[6, Prop.1/item 1]. Also, Extensivity among Kuratowski's closure axioms.
*)
LEMMA ClosureImplied ==
    ASSUME
        (* symbols u and b are undeclared in the current context *)
        TEMPORAL P(_),
        VARIABLE x
    PROVE
        P(x) => Cl(P, x)
    PROOF
    <1>1. P(x) => \EE u:  P(x) /\ [](u = x)
        OBVIOUS
        (* The bound variable u is a history-determined variable.
        u is undeclared in the current context,
        so u does not occur in the expression P(x). *)
    <1>2. (\EE u:  P(x) /\ [](u = x))
            => \AA b:  \/ ~ MustUnstep(b)
                       \/ \EE u:  /\ P(u)
                                  /\ [](b => (u = x))
        <2>1. (\EE u:  P(x) /\ [](u = x))
                => \EE u:  P(x) /\ [](u = x) /\ P(u)
            OMITTED  (* a proof of this step should argue about all
            possible temporal-level expressions P(_), thus in terms
            of all the production rules of the grammar, and the semantics. *)
        <2>2. (\EE u:  P(x) /\ [](u = x) /\ P(u))
                => \EE u:  [](u = x) /\ P(u)
            OBVIOUS
        <2>3. (\EE u:  [](u = x) /\ P(u))
                => \AA b:  \EE u:  [](u = x) /\ P(u)
            OBVIOUS
            (* The identifier b is undeclared in the current context,
            so b does not occur in the expression P(u). *)
        <2>4. (\AA b:  \EE u:  [](u = x) /\ P(u))
                => \AA b:  \/ ~ MustUnstep(b)
                           \/ \EE u:  P(u) /\ [](b => (u = x))
            OBVIOUS
        <2> QED
            BY <2>1, <2>2, <2>3, <2>4
    <1>3. Cl(P, x) <=> \AA b:  \/ ~ MustUnstep(b)
                               \/ \EE u:  /\ P(u)
                                          /\ [](b => (u = x))
        BY DEF Cl
    <1> QED
        BY <1>1, <1>2, <1>3


COROLLARY ClosureIdempotent ==
    ASSUME
        TEMPORAL P
    PROVE
        LET C == Cl(P)
        IN Cl(C) <=> C
    <1>1. Cl(P) => Cl(Cl(P))
        <2>1. P => Cl(P)
            BY ClosureImplied
        <2> QED
            BY <2>1, ClosureIsMonotonic
    <1>2. Cl(Cl(P)) => Cl(P)
        OMITTED  (* Sketch: for any n-prefix of sigma, pick an extension tau,
            with tau |= Cl(P). BY DEF of Cl, every prefix of tau is
            extensible to a behavior that satisfies P. For the n-prefix
            of tau pick such an extension eta, with eta |= P.

            The behaviors sigma and tau have common n-prefix. Thus, eta is
            an extension of sigma[0..n] that satisfies P. BY DEF of Cl,
            sigma |= Cl(P). *)
    <1> QED
        BY <1>1, <1>2


LEMMA ClosureOfImpl ==
    ASSUME
        TEMPORAL E, TEMPORAL M
    PROVE
        (Cl(E) => Cl(M))  =>  Cl(E => M)
    PROOF
    <1>1. Cl(M) => Cl(E => M)
        <2>1. M => (E => M)
            OBVIOUS
        <2> QED
            BY <2>1, ClosureIsMonotonic
    <1>2. (~ Cl(E)) => Cl(E => M)
        <2>1. (~ Cl(E)) => ~ E
            BY ClosureImplied
        <2>2. (~ E) => (E => M)
            OBVIOUS
        <2>3. (E => M)  =>  Cl(E => M)
            BY ClosureImplied
        <2> QED
            BY <2>1, <2>2, <2>3
    <1> QED
        BY <1>1, <1>2


PROPOSITION ConjClosureInsideClosure ==
    ASSUME
        TEMPORAL A, TEMPORAL B,
        A => Cl(B)
    PROVE
        Cl(A)  <=>  Cl(A /\ Cl(B))
    PROOF
    <1> DEFINE
        Q == A /\ Cl(B)
    <1>1. Cl(Q) => Cl(A)
        <2>1. Q => A
            BY DEF Q
        <2> QED
            BY <2>1, ClosureIsMonotonic
    <1>2. Cl(A) => Cl(Q)
        <2>1. A => Cl(B)
            OBVIOUS  (* BY ConjClosureInsideClosure!assumption *)
        <2>2. A => (A /\ Cl(B))
            BY <2>1
        <2>3. A => Q
            BY <2>2 DEF Q
        <2> QED
            BY <2>2, ClosureIsMonotonic
    <1> QED
        BY <1>1, <1>2


PROPOSITION ClosureSample ==
    ASSUME
        (* these operators may depend on variables declared in the context
        where this theorem is used. So the bound identifiers declared within
        the theorem and its proof are assumed to stand for identifiers that
        are selected to be different from all previously declared identifiers.

        This is required by the rules of TLA+, which doesn't allow
        redeclaration of an identifier, even a bounded identifier.
        *)
        VARIABLE x,
        CONSTANT R(_, _),
        TEMPORAL P(_)
    PROVE
        \/ ~ \EE u:  R(u, x) /\ Cl(P, u)
        \/ \EE u:  R(u, x) /\ P(u)
    PROOF
    <1>1. (\EE u:  R(u, x) /\ Cl(P, u))
        <=> \EE u:  /\ R(u, x)
                    /\ \AA b:
                        \/ ~ MustUnstep(b)
                        \/ \EE r:  /\ P(r)
                                   /\ [](b => (r = u))
        BY DEF Cl
    <1>2. \EE q:  MustUnstep(q)
        BY DEF MustUnstep
    <1>3. (\EE u:  R(u, x) /\ Cl(P, u))
        <=> \EE u:  /\ R(u, x)
                    /\ \EE q:  MustUnstep(q)
                    /\ \AA b:
                        \/ ~ MustUnstep(b)
                        \/ \EE r:  /\ P(r)
                                   /\ [](b => (r = u))
        BY <1>1, <1>2
    <1>4. (\EE u:  R(u, x) /\ Cl(P, u))
        <=> \EE u, q:
            /\ R(u, x) /\ MustUnstep(q)
            /\ \AA b:
                \/ ~ MustUnstep(b)
                \/ \EE r:  /\ P(r)
                           /\ [](b => (r = u))
        BY <1>3  (* pull \EE q outside *)
    <1>5. (\EE u:  R(u, x) /\ Cl(P, u))
        => \EE u, q:
            /\ R(u, x) /\ MustUnstep(q)
            /\ \/ ~ MustUnstep(q)
               \/ \EE r:  /\ P(r)
                          /\ [](q => (r = u))
        BY <1>4 DEF \AA  (* substitute STATE q for b *)
    <1>6. (\EE u:  R(u, x) /\ Cl(P, u))
        => \EE u, q:
            /\ R(u, x) /\ MustUnstep(q)
            /\ \EE r:  /\ P(r)
                       /\ [](q => (r = u))
        BY <1>5
    <1>7. ASSUME VARIABLE q, VARIABLE u
          PROVE \/ ~ /\ MustUnstep(q)
                     /\ [](q => (r = u))
                \/ r = u
        <2>1. ASSUME VARIABLE q, VARIABLE u
              PROVE MustUnstep(q) => (q = TRUE)
            BY DEF MustUnstep
        <2> QED
            BY <2>1
    <1>8. (\EE u:  R(u, x) /\ Cl(P, u))
        => \EE u, q, r:
            /\ R(u, x) /\ P(r)
            /\ (r = u)
        BY <1>6, <1>7
    <1>9. (\EE u:  R(u, x) /\ Cl(P, u))
        => \EE u, q, r:
            R(r, x) /\ P(r)
    <1>10. \/ ~ \EE u:  R(u, x) /\ Cl(P, u)
           \/ \EE r:  R(r, x) /\ P(r)
        BY <1>9
    <1> QED
        BY <1>10


(* A property is equisatisfiable with its closure.
See also SATClosureInit
*)
LEMMA ClosureEquiSAT ==
    ASSUME
        TEMPORAL P(_)
    PROVE
        (\EE u:  P(u))  <=>  \EE u:  Cl(P, u)
    PROOF
    <1>1. (\EE u:  P(u))  =>  \EE u:  Cl(P, u)
        <2>1. ASSUME VARIABLE u
              PROVE P(u) => Cl(P, u)
            BY ClosureImplied
        <2> QED
            BY <2>1
    <1>2. (\EE u:  Cl(P, u))  =>  \EE u:  P(u)
        (* can also use:  BY ClosureSample *)
        <2>1. (\EE u:  Cl(P, u))
            <=> \EE u:  \AA b:  \/ ~ MustUnstep(b)
                                \/ \EE r:  /\ P(r)
                                           /\ [](b => (r = u))
            BY DEF Cl
        <2>2. (\EE u:  Cl(P, u))
            => \EE u:  \AA b:  \/ ~ MustUnstep(b)
                               \/ \EE r:  P(r)
            BY <2>1
        <2>3. \EE q:  MustUnstep(q)
            BY DEF MustUnstep
        <2>4. (\EE u:  Cl(P, u))
            => \EE u:  /\ \EE q:  MustUnstep(q)
                       /\ \AA b:  \/ ~ MustUnstep(b)
                                  \/ \EE r:  P(r)
            BY <2>2, <2>3
        <2>5. (\EE u:  Cl(P, u))
            => \EE u, q:
                /\ MustUnstep(q)
                /\ \AA b:  \/ ~ MustUnstep(b)
                           \/ \EE r:  P(r)
            BY <2>4
        <2>6. (\EE u:  Cl(P, u))
            => \EE u, q:
                /\ MustUnstep(q)
                /\ \/ ~ MustUnstep(q)
                   \/ \EE r:  P(r)
            BY <2>5 DEF \AA
        <2>7. (\EE u:  Cl(P, u))
            => \EE u, q:
                \EE r:  P(r)
            BY <2>6
        <2> QED
            BY <2>7
    <1> QED
        BY <1>1, <1>2

--------------------------------------------------------------------------------
(* Properties that relate PrefixSat and PrefixPlusOne to closure. *)


LEMMA PrefixSatOfClosure ==
    ASSUME
        NEW sigma, IsABehavior(sigma),
        NEW n \in Nat,
        TEMPORAL P
    PROVE
        PrefixSat(sigma, n, P) => PrefixSat(sigma, n, Cl(P))
    PROOF
    <1>1. P => Cl(P)
        BY ClosureImplied
    <1> QED
        BY <1>1, PrefixSatImp


LEMMA PrefixPlusOneEquivWhilePlusOfClosures ==
    ASSUME
        TEMPORAL E,
        TEMPORAL M
    PROVE
        PrefixPlusOne(E, M)    <=>    (Cl(E) -+-> Cl(M))
    OMITTED  (* TODO *)


LEMMA WhilePlusOfClosures ==  (* I think this proof holds also in RTLA+ *)
    ASSUME
        TEMPORAL A,
        TEMPORAL G
    PROVE
        (Cl(E) -+-> Cl(M))    <=>    PrefixPlusOne(Cl(E), Cl(M))
    <1> DEFINE
        CE == Cl(E)
        CM == Cl(M)
    <1>1. (Cl(Ec) -+-> Cl(Mc))    <=>    PrefixPlusOne(Ec, Mc)
        BY PrefixPlusOneEquivWhilePlusOfClosures
    <1>2. /\ Cl(Ec) <=> Ec
          /\ Cl(Mc) <=> Mc
        BY ClosureIdempotent
    <1> QED
        BY <1>1, <1>2

--------------------------------------------------------------------------------
(* Properties of stepwise operators.

For brevity, this section uses the semantic closure operator Cl(P(_)),
instead of the syntactic operator Cl(P(_), x). These two operators yield
the same result whenever property P depends on no variables other than
its argument. Similar adaptations apply to other operators in this section.

In order for these conclusions to hold for other similar operators
(e.g., WhilePlusHalf), those operators should have the same basic properties,
in particular WhilePlusOfClosuresIsSafety and WhilePlusSafetyLivenessDecomp.
*)


(* The stepwise implication of safety properties is a safety property.
Equivalently, the stepwise implication of closures is a safety property.

[5, Lemma 1 on p. A-3]
*)
PROPOSITION WhilePlusOfClosuresIsSafety ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        LET C == Cl(A) -+-> Cl(G)
        IN IsSafety(C)
    PROOF
    <1> DEFINE
        ClA == Cl(A)
        ClG == Cl(G)
        C == Cl(A) -+-> Cl(G)
    <1>1. SUFFICES Cl(C) <=> C
        BY DEF IsSafety, C
    <1>2. SUFFICES Cl(C) => C
        <2>1. C => Cl(C)
            BY ClosureImplied
        <2> QED
            BY <2>1, <1>2
    <1>3. SUFFICES
            ASSUME
                NEW sigma, IsABehavior(sigma),
                sigma |= Cl(C)
            PROVE sigma |= C
        OBVIOUS
    <1>4. SUFFICES
            ASSUME ~ (sigma |= C)
            PROVE FALSE
        OBVIOUS
    <1>5. ~ \A n \in Nat:
            PrefixSat(sigma, n, ClA) => PrefixSat(sigma, n + 1, ClG)
        BY <1>4 DEF -+->
    <1>6. PICK n \in Nat:
            PrefixSat(sigma, n, ClA) /\ ~ PrefixSat(sigma, n + 1, ClG)
        BY <1>5
    <1>7. (* Any extension of sigma's n-prefix satisfies ClA. *)
        ASSUME
            NEW eta, IsABehavior(eta),
            Prefix(sigma, n) = Prefix(eta, n)
        PROVE
            PrefixSat(sigma, n, ClA) <=> PrefixSat(eta, n, ClA)
        BY EquivPrefixSatIfSamePrefix
    <1>8. (* No extension of sigma's (n + 1)-prefix can satisfy ClG. *)
        ASSUME
            NEW eta, IsABehavior(eta),
            Prefix(sigma, n + 1) = Prefix(eta, n + 1)
        PROVE
            PrefixSat(sigma, n + 1, ClG) <=> PrefixSat(eta, n + 1, ClG)
        BY EquivPrefixSatIfSamePrefix
    <1>9.
        ASSUME
            NEW eta, IsABehavior(eta),
            Prefix(sigma, n + 1) = Prefix(eta, n + 1)
        PROVE
            /\ PrefixSat(eta, n, ClA)
            /\ ~ PrefixSat(eta, n + 1, ClG)
        <2>1. Prefix(sigma, n) = Prefix(eta, n)
            BY <1>9 DEF Prefix
        <2>2. PrefixSat(sigma, n, ClA) <=> PrefixSat(eta, n, ClA)
            BY <1>9, <2>1, <1>7
        <2>3. PrefixSat(sigma, n + 1, ClG) <=> PrefixSat(eta, n + 1, ClG)
            BY <1>9, <1>8
        <2>4. PrefixSat(eta, n, ClA)
            BY <2>2, <1>6
        <2>5. ~ PrefixSat(eta, n + 1, ClG)
            BY <2>3, <1>6
        <2> QED
            BY <2>4, <2>5
    <1>10. \A eta:
            \/ ~ /\ IsABehavior(eta)
                 /\ Prefix(sigma, n + 1) = Prefix(eta, n + 1)
            \/ /\ PrefixSat(eta, n, ClA)
               /\ ~ PrefixSat(eta, n + 1, ClG)
        BY <1>9
    <1>11. \A eta:
            \/ ~ /\ IsABehavior(eta)
                 /\ Prefix(sigma, n + 1) = Prefix(eta, n + 1)
            \/ ~ \A k \in Nat:
                PrefixSat(eta, k, ClA) => PrefixSat(eta, k + 1, ClG)
        BY <1>10
    <1>12. \A eta:
            \/ ~ /\ IsABehavior(eta)
                 /\ Prefix(sigma, n + 1) = Prefix(eta, n + 1)
            \/ ~ (eta |= C)
        BY <1>11 DEF C
    <1>13. ~ \E eta:  /\ IsABehavior(eta)
                      /\ Prefix(sigma, n + 1) = Prefix(eta, n + 1)
                      /\ eta |= C
        BY <1>12
    <1>14. ~ PrefixSat(sigma, n, C)
        BY <1>13, PrefixSatAsSamePrefix
    <1>15. ~ (sigma |= Cl(C))
        BY <1>14 DEF Cl  (* The semantic definition of closure. *)
    <1> QED
        BY <1>3, <1>15  (* goal from <1>4 *)


(* The open-system propery A -+-> G is the conjunction of a safety and
a liveness part. The safety part involves only closures, which is useful.
The liveness part relates stepwise to logical implication (-+-> to =>).

That Cl(A) -+-> Cl(G) is the safety part and A => G the liveness part does
not follow from this theorem, but from WhilePlusSafetyLivenessDecomp.

[5, Lemma 2 on p. A-3]
*)
THEOREM WhilePlusAsConj ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        A -+-> G  <=>  /\ Cl(A) -+-> Cl(G)
                       /\ A => G
    OMITTED


(* [5, Lemma 3 on p. A-3] *)
PROPOSITION StepwiseAntecedent ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        (A /\ (A -+-> G))  =>  G
    OMITTED


PROPOSITION StepwiseConsequent ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        G  =>  (A -+-> G)
    OMITTED


(* Closure distributes over stepwise implication. *)
THEOREM WhilePlusMachineClosedRepr ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        Cl(A -+-> G)  <=>  (Cl(A) -+-> Cl(G))
    PROOF
    <1> DEFINE
        P == A -+-> G
        C == Cl(A) -+-> Cl(G)
    <1>1. Cl(P) => C
        <2>1. P => C
            <3>1. A -+-> G <=> /\ Cl(A) -+-> Cl(G)
                               /\ A => G
                BY WhilePlusAsConj
            <3>2. A -+-> G => (Cl(A) -+-> Cl(G))
                BY <3>1
            <3> QED
                BY <3>2 DEF P, C
        <2>2. Cl(P) => Cl(C)
            BY <2>1, ClosureIsMonotonic
        <2>3. Cl(C) <=> C
            <3>1. IsSafety(C)
                BY WhilePlusOfClosuresIsSafety DEF C
            <3> QED
                BY <3>1, ClosureOfSafety
        <2> QED
            BY <2>2, <2>3
    <1>2. C => Cl(P)
        <2>1. SUFFICES
                ASSUME
                    NEW sigma, IsABehavior(sigma),
                    sigma |= C
                PROVE
                    sigma |= Cl(P)
            OBVIOUS
        <2>2. CASE sigma |= Cl(A)
            <3>1. sigma |= Cl(G)
                BY <2>1, <2>2, StepwiseAntecedent DEF C
            <3>2. SUFFICES
                    ASSUME NEW n \in Nat
                    PROVE PrefixSat(sigma, n, P)
                BY DEF Cl  (* goal from <2>1 *)
            <3>3. PICK tau:  /\ IsABehavior(tau)
                             /\ Prefix(tau, n) = Prefix(sigma, n)
                             /\ tau |= G
                BY <3>1, PrefixSatAsSamePrefix
            <3>4. tau |= A -+-> G
                BY <3>3, StepwiseConsequent
            <3>5. \E tau:  /\ IsABehavior(tau)
                           /\ Prefix(tau, n) = Prefix(sigma, n)
                           /\ tau |= P
                BY <3>3, <3>4 DEF P
            <3> QED
                BY <3>5, PrefixSatAsSamePrefix  (* goal from <2>1 *)
        <2>3. CASE ~ sigma |= Cl(A)
            <3>1. sigma |= Cl(A) -+-> Cl(G)
                BY <2>1 DEF C
            <3>2. sigma |= A => G
                <4>1. sigma |= ~ Cl(A)
                    BY <2>3
                <4>2. (~ Cl(A)) => ~ A
                    BY ClosureImplied
                <4>3. sigma |= ~ A
                    BY <4>1, <4>2
                <4> QED
                    BY <4>3
            <3>3. sigma |= A -+-> G
                BY <3>1, <3>2, WhilePlusAsConj
            <3>4. sigma |= P
                BY <3>3 DEF P
            <3> QED
                BY <3>4, ClosureImplied
        <2> QED
            BY <2>2, <2>3
    <1> QED
        BY <1>1, <1>2


(* A representation theorem. *)
THEOREM WhilePlusSafetyLivenessDecomp ==
    ASSUME
        TEMPORAL A, TEMPORAL G
    PROVE
        LET
            AG == A -+-> G
            C == Cl(A) -+-> Cl(G)
        IN
            /\ SafetyPart(AG)  <=>  C
            /\ LivenessPart(AG)  <=>  (C => AG)
    PROOF
    <1>1. SafetyPart(A -+-> G)  <=>  Cl(A) -+-> Cl(G)
        BY WhilePlusMachineClosedRepr DEF SafetyPart
    <1>2. LivenessPart(A -+-> G)
            <=> (Cl(A) -+-> Cl(G))  =>  (A -+-> G)
        BY <1>1 DEF LivenessPart
    <1> QED
        BY <1>1, <1>2


THEOREM
    ASSUME
        TEMPORAL E, TEMPORAL M
    PROVE
        Cl(E -+-> M)  =>  Cl(E => M)
    PROOF
    <1>1. Cl(E -+-> M)  =>  (Cl(E) => Cl(M))
        BY WhilePlusSafetyLivenessDecomp
    <1>2. (Cl(E) => Cl(M))  =>  Cl(E => M)
        BY ClosureOfImpl
    <1> QED
        BY <1>1, <1>2


(* Feedback sustains M. *)
THEOREM  WhilePlusFeedback ==
    ASSUME
        TEMPORAL M,
        ~ |= ~ M  (* M is satisfiable *)
    PROVE
        LET C == Cl(M)
        IN (C -+-> C)  <=>  C
    PROOF
    <1> DEFINE C == Cl(M)
    <1>1. C  =>  (C -+-> C)
        BY StepwiseConsequent
    <1>2. (C -+-> C)  =>  C
        <2>1. SUFFICES
                ASSUME
                    NEW sigma, IsABehavior(sigma),
                    sigma |= C -+-> C
                PROVE
                    sigma |= C
            OBVIOUS
        <2>2. \A n \in Nat:
                PrefixSat(sigma, n, C)
                    => PrefixSat(sigma, n + 1, C)
            BY <2>1, WhilePlusProperties
        <2>3. PrefixSat(sigma, 0, C)
            <3>1. ~ |= ~ M
                OBVIOUS  (* BY WhilePlusFeedback!assumption *)
            <3>2. \E tau:  /\ IsABehavior(tau)
                           /\ tau |= M
                BY <3>1
            <3> QED
                BY <3>2 DEF PrefixSat
        <2>4. \A n \in Nat:  PrefixSat(sigma, n, C)
            BY <2>2, <2>3, NatInduction
        <2>5. sigma |= Cl(C)
            BY <2>4 DEF Cl  (* semantic DEF of closure *)
        <2> QED
            BY <2>5, ClosureIdempotent DEF C
    <1> QED
        BY <1>1, <1>2 DEF C


(* Feeding the same temporal property as both arguments of the while-plus
operator cancels out the liveness part of that property.
*)
LEMMA ErasingLiveness ==
    ASSUME
        TEMPORAL M
    PROVE
        (M -+-> M)  <=>  (Cl(M) -+-> Cl(M))
    PROOF
    <1>1. M -+-> M  <=>  /\ Cl(M) -+-> Cl(M)
                         /\ M => M
        BY WhilePlusAsConj
    <1>2. M => M
        OBVIOUS
    <1> QED
        BY <1>1, <1>2


(* If M is satisfiable, then the while-plus property M -+-> M is the closure
Cl(M) of M.
*)
COROLLARY ClosureViaWhilePlus ==
    ASSUME
        TEMPORAL M,
        ~ |= ~ M
    PROVE
        (M -+-> M)  <=>  Cl(M)
    PROOF
    <1>1. (M -+-> M)  <=>  (Cl(M) -+-> Cl(M))
        BY ErasingLiveness
    <1>2. (Cl(M) -+-> Cl(M))  <=>  Cl(M)
        BY WhilePlusFeedback
    <1> QED
        BY <1>1, <1>2


(* An instance of the schema from [5, Lemma 5]. *)
LEMMA ConjoiningSafety ==
    ASSUME
        TEMPORAL A, TEMPORAL B, TEMPORAL C, TEMPORAL D,
        IsSafety(A), IsSafety(B), IsSafety(C), IsSafety(D)
    PROVE
        \/ ~ /\ A -+-> B
             /\ C -+-> D
        \/ (A /\ B) -+-> (C /\ D)
    OMITTED


THEOREM
    ASSUME
        TEMPORAL P, TEMPORAL Q,
        IsSafety(P), IsSafety(Q),
        ~ |= ~ (P /\ Q)
    PROVE
            \/ ~ /\ P -+-> Q
                 /\ Q -+-> P
            \/ P /\ Q
    PROOF
    <1>1. \/ ~ /\ P -+-> Q
               /\ Q -+-> P
          \/ (P /\ Q) -+-> (Q /\ P)
        BY ConjoiningSafety
    <1>2. ((P /\ Q) -+-> (Q /\ P))
            <=> (P /\ Q)
        <2>1. ~ |= ~ (P /\ Q)
            OBVIOUS
        <2>2. IsSafety(P /\ Q)
            OMITTED  (* The conjunction of safety properties is safety. *)
        <2> QED
            BY <2>1, <2>2, WhilePlusFeedback
    <1> QED
        BY <1>1, <1>2


(* If we weaken the first argument and strengthen the second argument of -+->,
then the resulting open-system refines the open-system we started with.
*)
THEOREM RefinementOfWhilePlus ==
    ASSUME
        TEMPORAL A, TEMPORAL G,
        TEMPORAL P, TEMPORAL R,
        /\ P => A
        /\ G => R
        (* if each of A, G, P, R contains recurrence,
        then these are GR(1) problems (via Klein-Pnueli). If each has
        GR(1) liveness, then these are GR(2) problems. *)
    PROVE
        (A -+-> G)  =>  (P -+-> R)
    PROOF
    <1>1. /\ A -+-> G <=> /\ PrefixPlusOne(A, G)
                          /\ A => G

          /\ P -+-> R <=> /\ PrefixPlusOne(P, R)
                          /\ P => R
        BY DEF -+->
    <1>2. (A => G)  =>  (P => R)
        <2>1. SUFFICES (P /\ (A => G)) => R
            OBVIOUS
        <2>2. (P /\ (A => G))  =>  G
            <3>1. P => A
                OBVIOUS  (* BY RefinementOfWhilePlus!assumption *)
            <3> QED
                BY <3>1
        <2>3. G => R
            OBVIOUS  (* BY RefinementOfWhilePlus! assumption *)
        <2> QED
            BY <2>2, <2>3  (* goal from <2>1 *)
    <1>3. PrefixPlusOne(A, G)  =>  PrefixPlusOne(P, R)
        <2>1. SUFFICES
                ASSUME
                    NEW n \in Nat,
                    NEW sigma, IsABehavior(sigma)
                PROVE
                    \/ ~ (PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
                    \/ PrefixSat(sigma, n, P) => PrefixSat(sigma, n + 1, R)
            BY DEF PrefixPlusOne
        <2>2. SUFFICES
                ASSUME
                    /\ PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
                    /\ PrefixSat(sigma, n, P)
                PROVE
                    PrefixSat(sigma, n + 1, R)
            OBVIOUS  (* goal from <2>1 *)
        <2>3. PrefixSat(sigma, n, A)
            <3>1. PrefixSat(sigma, n, P)
                BY <2>2
            <3>2. P => A
                OBVIOUS  (* BY RefinementOfWhilePlus! assumption *)
            <3>3. PrefixSat(sigma, n, P) => PrefixSat(sigma, n, A)
                <4>1. IsABehavior(sigma)
                    BY <2>1
                <4> QED
                    BY <4>1, <3>2, PrefixSatImp
            <3> QED
                BY <3>1, <3>3
        <2>4. PrefixSat(sigma, n + 1, G)
            BY <2>3, <2>2
        <2> QED
            <3>1. G => R
                OBVIOUS  (* BY RefinementOfWhilePlus! assumption *)
            <3>2. PrefixSat(sigma, n + 1, G) => PrefixSat(sigma, n + 1, R)
                <4>1. IsABehavior(sigma)
                    BY <2>1
                <4> QED
                    BY <4>1, <3>1, PrefixSatImp
            <3> QED
                BY <2>4, <3>2  (* goal from <2>2 *)
    <1> QED
        BY <1>1, <1>2, <1>3

--------------------------------------------------------------------------------
(* Proof of rewriting -+-> with safety as first argument. *)


PROPOSITION WeakeningLivenessPreservesMachineClosure ==
    ASSUME
        TEMPORAL S, TEMPORAL L, TEMPORAL R,
        Cl(S /\ L) <=> S
    PROVE
        Cl(S /\ (L \/ R))  <=>  S
    PROOF
    <1>1. DEFINE
        Z == Cl(S /\ (L \/ R))
    <1>2. S => Z
        <2>1. (S /\ L)  =>  (S /\ (L \/ R))
            OBVIOUS
        <2>2. Cl(S /\ L)  =>  Cl(S /\ (L \/ R))
            BY <2>1, ClosureIsMonotonic
        <2>3. S => Cl(S /\ (L \/ R))
            BY <2>2
                (* and WeakeningLivenessPreservesMachineClosure!assumption *)
        <2> QED
            BY <2>3 DEF Z
    <1>3. Z => S
        <2>1. (S /\ (L \/ R)) => S
            OBVIOUS
        <2>2. Cl(S /\ (L \/ R))  =>  Cl(S)
            BY <2>1, ClosureIsMonotonic
        <2>3. Z => Cl(S)
            BY <2>2 DEF Z
        <2>4. Cl(S) <=> S
            <3>1. Cl(L /\ S) <=> S
                OBVIOUS
                (* BY WeakeningLivenessPreservesMachineClosure!assumption *)
            <3>2. Cl(Cl(L /\ S)) <=> Cl(S)
                BY <3>1
            <3>3. Cl(Cl(L /\ S)) <=> Cl(L /\ S)
                BY ClosureIdempotent
            <3>4. Cl(L /\ S) <=> Cl(S)
                BY <3>2, <3>3
            <3> QED
                BY <3>1, <3>4
        <2> QED
            BY <2>3, <2>4
    <1> QED
        BY <1>1, <1>2 DEF Z


(* A claim on p. 528 in [5]. *)
THEOREM RewritingWhilePlusWithSafetyArg1 ==
    ASSUME
        TEMPORAL E, TEMPORAL M
    PROVE
        LET
            ENew == Cl(E)
            MNew == Cl(M) /\ (E => M)
        IN
            /\ (E -+-> M)  <=>  (ENew -+-> MNew)
            /\ IsSafety(ENew)
    PROOF
    <1> DEFINE
        EM == E -+-> M
        ENew == Cl(E)
        MNew == Cl(M) /\ (E => M)
    <1>1. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ E => M
        BY WhilePlusAsConj
    <1>2. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ Cl(E) => Cl(M)
                 /\ E => M
        <2>1. (Cl(E) -+-> Cl(M))
                <=>  /\ Cl(Cl(E)) -+-> Cl(Cl(M))
                     /\ Cl(E) => Cl(M)
            BY WhilePlusAsConj
        <2>2. \/ ~ (Cl(E) -+-> Cl(M))
              \/ Cl(E) => Cl(M)
            BY <2>1
        <2> QED
            BY <1>1, <2>2
    <1>3. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ Cl(E) => Cl(M)
                 /\ (Cl(E) /\ E) => M
        <2>1. E => Cl(E)
            BY ClosureImplied
        <2>2. E <=> (E /\ Cl(E))
            BY <2>1
        <2> QED
            BY <1>2, <2>2
    <1>4. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ Cl(E) => Cl(M)
                 /\ Cl(E) => (E => M)
        BY <1>3
    <1>5. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ Cl(E) => /\ Cl(M)
                             /\ E => M
        BY <1>4
    <1>6. EM <=> /\ Cl(Cl(E)) -+-> Cl(Cl(M) /\ (E => M))
                 /\ Cl(E) => /\ Cl(M)
                             /\ E => M
        <2>1. Cl(E) <=> Cl(Cl(E))
            BY ClosureIdempotent
        <2>2. Cl(M) <=> Cl(Cl(M) /\ (E => M))
            (* In words: The pair M, Cl(M) is machine-closed. *)
            <3>1. M => Cl(M)
                BY ClosureImplied
            <3>2. M <=> (Cl(M) /\ M)
                BY <3>1
            <3>3. Cl(M) <=> Cl(Cl(M) /\ M)
                BY <3>2
            <3>4. Cl(M) <=> Cl(Cl(M) /\ (M \/ ~ E))
                BY <3>3, WeakeningLivenessPreservesMachineClosure
                (* with S == Cl(M), L == M, R == ~ E *)
            <3> QED
                BY <3>4
        <2> QED
            BY <1>5, <2>1, <2>2
    <1>7. EM <=> /\ Cl(ENew) -+-> Cl(MNew)
                 /\ ENew => MNew
        BY <1>6 DEF ENew, MNew
    <1>8. IsSafety(ENew)
        <2>1. Cl(ENew) <=> Cl(Cl(E))
            BY DEF Enew
        <2>2. Cl(Cl(E)) <=> Cl(E)
            BY ClosureIdempotent
        <2>3. Cl(E) <=> ENew
            BY DEF ENew
        <2>4. Cl(ENew) <=> ENew
            BY <2>1, <2>2, <2>3
        <2> QED
            BY <2>4 DEF IsSafety
    <1> QED
        BY <1>7, <1>8 DEF EM, ENew, MNew


(* The liveness part is shifted to the second argument of -+->.
A form of "saturation".
*)
PROPOSITION
    ASSUME
        TEMPORAL E, TEMPORAL M
    PROVE
        E -+-> M  <=>  /\ Cl(E) -+-> Cl(M)
                       /\ Cl(E) => (LivenessPart(E) => M)
    PROOF
    <1> DEFINE
        EM == E -+-> M
    <1>1. EM <=> /\ Cl(E) -+-> Cl(M)
                 /\ E => M
        BY WhilePlusAsConj
    <1>2. (E => M)
            <=>  ((E /\ Cl(E)) => M)
        BY ClosureImplied
    <1>3. ((E /\ Cl(E)) => M)
            <=> (Cl(E) => (E => M))
        OBVIOUS
    <1>4. (Cl(E) => (E => M))
            <=> (Cl(E) => ((E \/ ~ Cl(E)) => M))
        OBVIOUS
    <1>5. LivenessPart(E) <=> (Cl(E) => E)
        BY DEF LivenessPart
    <1> QED
        BY <1>1, <1>2, <1>3, <1>4, <1>5


--------------------------------------------------------------------------------
(* The raw version of RuleINV1 from the module TLAPS. *)
THEOREM RuleRawINV1 ==
    ASSUME
        STATE I, ACTION N,
        (I /\ N)  =>  I'
    PROVE
        (I /\ []N)  =>  []I
    OMITTED

================================================================================
