----------------------------- MODULE UnzipTheorems -----------------------------
(* Theorems related to the operator Unzip.

Obvious proofs below are so for TLAPS v1.4.3.


Author:  Ioannis Filippidis

--------------------------------------------------------------------------------
Copyright 2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS WhilePlusHalfTheorems


ExistsUnique(P(_)) ==
    /\ \E u:  P(u)
    /\ \A u, v:  (P(u) /\ P(v))  =>  (u = v)


THEOREM InessentialNoninterleaving ==
    ASSUME
        VARIABLE x, VARIABLE y, CONSTANT Inv(_, _),
        CONSTANT SysTurn(_, _), CONSTANT Next(_, _, _, _),
        LET SysNext(p, q, v) == \E u:  Next(p, q, u, v)
            EnvNext(p, q, u) == \E v:  Next(p, q, u, v)
        IN
            /\ \/ ~ (SysTurn(x, y) /\ Inv(x, y))
               \/ ExistsUnique(LAMBDA r:  EnvNext(x, y, r))
            /\ SysNext(x, y, y') /\ EnvNext(x, y, x')
    PROVE
        \/ ~ /\ SysTurn(x, y)
             /\ Inv(x, y)
        \/ Next(x, y, x', y')
PROOF
<1> DEFINE
    SysNext(p, q, v) == \E u:  Next(p, q, u, v)
    EnvNext(p, q, u) == \E v:  Next(p, q, u, v)
<1>1. SUFFICES
        ASSUME SysTurn(x, y) /\ Inv(x, y)
        PROVE Next(x, y, x', y')
    OBVIOUS
<1>3. SUFFICES
        ASSUME ~ Next(x, y, x', y')
        PROVE FALSE
    OBVIOUS  (* goal from <1>1 *)
<1>2. /\ \E u:  Next(x, y, u, y')
      /\ \E v:  Next(x, y, x', v)
    <2>1. /\ SysNext(x, y, y')
          /\ EnvNext(x, y, x')
        OBVIOUS
    <2> QED
        BY <2>1 DEF SysNext, EnvNext
<1>4. PICK u:  Next(x, y, u, y')
    BY <1>2
<1>10. PICK v:  Next(x, y, x', v)
    BY <1>2
<1>5. u # x'
    <2>1. SUFFICES ASSUME u = x'
                   PROVE FALSE
    <2>2. Next(x, y, u, y')
        BY <1>4
    <2>3. Next(x, y, x', y')
        BY <2>1, <2>2
    <2>4. ~ Next(x, y, x', y')
        BY <1>3
    <2> QED
        BY <2>3, <2>4  (* goal from <2>1 *)
<1>6. SUFFICES u = x'
    BY <1>5  (* goal from <1>3 *)
<1>7. /\ \E a:  Next(x, y, u, a)
      /\ \E b:  Next(x, y, x', b)
    BY <1>4, <1>10
<1>8. /\ EnvNext(x, y, u)
      /\ EnvNext(x, y, x')
    BY <1>7 DEF EnvNext
<1> QED
    <2>1. ExistsUnique(LAMBDA r:  EnvNext(x, y, r))
        <3>1. SysTurn(x, y) /\ Inv(x, y)
            BY <1>1
        <3>2. \/ ~ (SysTurn(x, y) /\ Inv(x, y))
              \/ ExistsUnique(LAMBDA r:  EnvNext(x, y, r))
            OBVIOUS
        <3> QED
            BY <3>1, <3>2
    <2> QED
        BY <1>8, <2>1 DEF ExistsUnique


THEOREM CPreSimplerByConjunctivity ==
    ASSUME
        NEW Next, NEW SysNext, NEW EnvNext, NEW Target,
        Next <=> (SysNext /\ EnvNext)  (* Conjunctivity *)
    PROVE
        ( /\ SysNext
          /\ EnvNext => Target )
        <=>
        ( /\ SysNext
          /\ EnvNext => /\ Next
                        /\ Target )
PROOF OBVIOUS
(*
    /\ SysNext
    /\ EnvNext => Target
    <=>
    /\ SysNext
    /\ EnvNext => SysNext
    /\ EnvNext => Target
    <=>
    /\ SysNext
    /\ EnvNext => /\ SysNext
                  /\ Target
    <=>
    /\ SysNext
    /\ EnvNext => /\ SysNext /\ EnvNext
                  /\ Target
    <=>
    /\ SysNext
    /\ EnvNext => /\ Next
                  /\ Target
*)


THEOREM EquienablednessImpliesCartesianity ==
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT EnvNext(_, _, _),
        CONSTANT SysNext(_, _, _),
        (\E u:  EnvNext(x, y, u)) <=> \E v:  SysNext(x, y, v)
    PROVE
        (* The proof goal says that NewNext is Cartesian. *)
        LET
            NewNext(p, q, u, v) == /\ EnvNext(x, y, u)
                                   /\ SysNext(x, y, v)
        IN
            /\ SysNext(x, y, y') <=> \E u:  NewNext(x, y, u, y')
            /\ EnvNext(x, y, x') <=> \E v:  NewNext(x, y, x', v)
PROOF OBVIOUS


(* Actions $EnvNext, SysNext$ that result from $Unzip$ are enabled at the
same states. *)
PROPOSITION EquiEnablednessFromUnzip ==
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT Next(_, _, _, _),
        CONSTANT SysNext(_, _, _),
        CONSTANT EnvNext(_, _, _),
        /\ \A v:  SysNext(x, y, v) <=> \E u:  Next(x, y, u, v)
        /\ \A u:  EnvNext(x, y, u) <=> \E v:  Next(x, y, u, v)
    PROVE
        (\E u:  EnvNext(x, y, u)) <=> \E v:  SysNext(x, y, v)
PROOF OBVIOUS
(*
<1>1. (ENABLED EnvNext(x, y, x')) <=> \E u:  EnvNext(x, y, u)
<1>2. (\E u:  EnvNext(x, y, u)) <=> \E u:  \E v:  Next(x, y, u, v)
<1>3. (\E u:  \E v:  Next(x, y, u, v)) <=> \E v:  \E u:  Next(x, y, u, v)
<1>4. (\E v:  \E u:  Next(x, y, u, v)) <=> \E v:  SysNext(x, y, v)
<1>5. (\E v:  SysNext(x, y, v)) <=> ENABLED SysNext(x, y, y')
<1> QED
    BY <1>1, <1>2, <1>3, <1>4, <1>5
*)


COROLLARY
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT Next(_, _, _, _),
        CONSTANT SysNext(_, _, _),
        CONSTANT EnvNext(_, _, _),
        /\ \A v:  SysNext(x, y, v) <=> \E u:  Next(x, y, u, v)
        /\ \A u:  EnvNext(x, y, u) <=> \E v:  Next(x, y, u, v)
    PROVE
        LET
            NewNext(p, q, u, v) == /\ EnvNext(x, y, u)
                                   /\ SysNext(x, y, v)
        IN
            /\ SysNext(x, y, y') <=> \E u:  NewNext(x, y, u, y')
            /\ EnvNext(x, y, x') <=> \E v:  NewNext(x, y, x', v)
PROOF OBVIOUS
(*
<1>1. (\E u:  EnvNext(x, y, u)) <=>  \E v:  SysNext(x, y, v)
    BY EquiEnablednessFromUnzip
<1> QED
    BY <1>1, EquienablednessImpliesCartesianity
*)



COROLLARY
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT Next(_, _, _, _)
    PROVE
        LET
            (* The operators SysNext and EnvNext are already
               ``balanced'', but may not imply
               Next when conjoined. This is why we have to do
               the factorization as the next theorem below. *)
            SysNext(p, q, v) == \E u:  Next(p, q, u, v)
            EnvNext(p, q, u) == \E v:  Next(p, q, u, v)
            NewNext(p, q, u, v) ==
                /\ SysNext(x, y, v)
                /\ EnvNext(x, y, u)
            (* NewNext is conjunctive and Cartesian,
            so the controllable step operator is simpler when we apply
            Unzip to a property defined using NewNext. *)
        IN
            /\ SysNext(x, y, y') = \E u:  NewNext(x, y, u, y')
            /\ EnvNext(x, y, x') = \E v:  NewNext(x, y, x', v)
PROOF OBVIOUS

--------------------------------------------------------------------------------

PROPOSITION PoofTheAntecedent ==
    ASSUME
        CONSTANT A, CONSTANT B,
        CONSTANT C, CONSTANT D,
        A => D
    PROVE
        ( /\ A
          /\ (B => C) )
        <=>
        ( /\ A
          /\ (D /\ B) => C )
PROOF OBVIOUS


(* Even though TLAPS proves the above, below is a proof by hand. *)
PROPOSITION
    ASSUME
        CONSTANT A, CONSTANT B,
        CONSTANT C, CONSTANT D,
        A => D
    PROVE
        ( /\ A
          /\ (B => C) )
        <=>
        ( /\ A
          /\ (D /\ B) => C )
PROOF
<1>1. ASSUME /\ A
             /\ B => C
      PROVE /\ A
            /\ (D /\ B) => C
    <2>1. /\ A
          /\ C \/ ~ B
        BY <1>1
    <2>2. (C \/ ~ B) => (C \/ ~ B \/ ~ D)
        OBVIOUS
    <2>3. /\ A
          /\ C \/ ~ B \/ ~ D
        BY <2>1, <2>2
    <2> QED
        BY <2>3
<1>2. ASSUME /\ A
             /\ (D /\ B) => C
      PROVE /\ A
            /\ B => C
    <2>1. /\ A
          /\ \/ ~ (D /\ B)
             \/ C
        BY <1>2
    <2>2. /\ A
          /\ \/ ~ D \/ ~ B
             \/ C
        BY <2>1
    <2>3. \/ /\ A
             /\ \/ ~ B
                \/ C
          \/ /\ A
             /\ ~ D
        BY <2>2
    <2>4. ~ (A /\ ~ D)
        <3>1. A => D
            OBVIOUS
        <3> QED
            BY <3>1
    <2>5. \/ /\ A
             /\ B => C
          \/ FALSE
        BY <2>3, <2>4
    <2> QED
        BY <2>5
<1> QED
    BY <1>1, <1>2

--------------------------------------------------------------------------------


THEOREM SeparatingTheRealizablePart ==
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT Next(_, _, _, _),
        CONSTANT Target(_, _),
        CONSTANT EnvNext(_, _, _),
        CONSTANT SysNext(_, _, _),
        (ENABLED SysNext(x, y, y')) => ENABLED EnvNext(x, y, x')
    PROVE
        LET
            NewNext(u, v) ==
                /\ SysNext(x, y, v) /\ EnvNext(x, y, u)
                /\ \A w:  EnvNext(x, y, w) => Next(x, y, w, v)
                    (* The second conjunct shrinks the first in order
                    to ensure receptivity at those states. *)
            NewSysNext(v) == \E u:  NewNext(u, v)
            NewEnvNext(u) == \E v:  NewNext(u, v)
            A == \E v:
                /\ SysNext(x, y, v)
                /\ \A u:  EnvNext(x, y, u) => /\ Next(x, y, u, v)
                                              /\ Target(u, v)
            B == \E v:
                /\ NewSysNext(v)
                /\ \A u:  NewEnvNext(u) => /\ NewNext(u, v)
                                           /\ Target(x', v)
            C == \E v:
                /\ NewSysNext(v)
                /\ \A u:  NewEnvNext(u) => Target(u, v)
        IN
            /\ NewNext(x', y') => Next(x, y, x', y')
            /\ A <=> B
            /\ A <=> C
            /\ NewNext(x', y') <=> (NewSysNext(y') /\ NewEnvNext(x'))
PROOF
<1> DEFINE
    A == \E v:
        /\ SysNext(x, y, v)
        /\ \A u:  EnvNext(x, y, u) => /\ Next(x, y, u, v)
                                      /\ Target(u, v)
<1>1. A <=>
    \E v:  /\ SysNext(x, y, v)
           /\ \A u:  EnvNext(x, y, u) => Next(x, y, u, v)
           /\ \A u:  EnvNext(x, y, u) => Target(u, v)

<1>2. DEFINE NewSysNext(p, q, v) ==
        /\ SysNext(p, q, v)
        /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
    (* This definition of NewSysNext differs from that in the
    theorem statement. Nevertheless, we show their equivalence below. *)
<1>3. A <=>
    \E v:  /\ NewSysNext(x, y, v)
           /\ \A u:  EnvNext(x, y, u) => Target(u, v)
    BY <1>1 DEF NewSysNext
<1>4. A <=>
    \E v:  \A u:
        /\ NewSysNext(x, y, v)
        /\ \/ ~ /\ EnvNext(x, y, u)
                /\ ENABLED NewSysNext(x, y, y')
           \/ Target(u, v)
    <2>1. ASSUME NEW v
          PROVE NewSysNext(x, y, v) <=> \A u:  NewSysNext(x, y, v)
        BY DEF NewSysNext
    <2>2. ASSUME NEW v
          PROVE NewSysNext(x, y, v) => ENABLED NewSysNext(x, y, y')
        BY DEF NewSysNext
    <2>3. (\E v:  /\ NewSysNext(x, y, v)
                  /\ \A u:  EnvNext(x, y, u) => Target(u, v))
            <=>
            (\E v:
                /\ \A u:  NewSysNext(x, y, v)
                /\ \A u:  EnvNext(x, y, u) => Target(u, v))
        BY <2>1
    <2>4. A <=>
            \E v:  \A u:
                /\ NewSysNext(x, y, v)
                /\ EnvNext(x, y, u) => Target(u, v)
        BY <1>3, <2>3
    <2> QED
        BY <2>4, <2>2
<1>5. DEFINE NewNext(p, q, u, v) ==
    /\ SysNext(p, q, v) /\ EnvNext(p, q, u)
    /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
<1>6. ASSUME NEW p, NEW q, NEW u, NEW v
      PROVE NewNext(p, q, u, v)  <=>  /\ NewSysNext(p, q, v)
                                      /\ EnvNext(p, q, u)
    BY DEF NewNext, NewSysNext
<1>7. ASSUME NEW p, NEW q, NEW v
      PROVE NewSysNext(p, q, v)  <=>  \E u:  NewNext(p, q, u, v)
    <2>1. (\E u:  NewNext(p, q, u, v))
        <=> \E u:  /\ SysNext(p, q, v) /\ EnvNext(p, q, u)
                   /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
        BY DEF NewNext
    <2>2. (\E u:  NewNext(p, q, u, v))
        <=> /\ SysNext(p, q, v)
            /\ \E u:  EnvNext(p, q, u)
            /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
        BY <2>1
    <2>3. SysNext(p, q, v) => \E u:  EnvNext(p, q, u)
        <3>1. SysNext(p, q, v) => \E s:  SysNext(p, q, s)
            OBVIOUS
        <3>2. (\E s:  SysNext(p, q, s))  =>  \E u:  EnvNext(p, q, u)
            <4>1. (ENABLED SysNext(x, y, y')) => ENABLED EnvNext(x, y, x')
                OBVIOUS  (* BY SeparatingTheRealizablePart!assumption *)
            <4> QED
                BY <4>1
        <3> QED
            BY <3>1, <3>2
    <2>4. (\E u:  NewNext(p, q, u, v))
        <=> /\ SysNext(p, q, v)
            /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
        BY <2>2, <2>3
    <2> QED
        BY <2>4 DEF NewSysNext
<1>8. DEFINE NewEnvNext(p, q, u) == \E v:  NewNext(p, q, u, v)
<1>9. ASSUME NEW p, NEW q, NEW u
      PROVE NewEnvNext(p, q, u) <=> /\ EnvNext(p, q, u)
                                    /\ ENABLED NewSysNext(p, q, y')
    <2> DEFINE F == NewEnvNext(p, q, u)
    <2>1. F
        <=> \E v:  /\ SysNext(p, q, v) /\ EnvNext(p, q, u)
                   /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
    <2>2. F
        <=> \E v:  /\ NewSysNext(p, q, v)
                   /\ EnvNext(p, q, u)
    <2>3. F  <=>  EnvNext(p, q, u) /\ \E v:  NewSysNext(p, q, v)
    <2>4. F  <=>  EnvNext(p, q, u) /\ ENABLED NewSysNext(p, q, y')
    <2> QED
        BY <2>4 DEF F
<1>10. A <=>
        \E v:  \A u:
            /\ NewSysNext(x, y, v)
            /\ NewEnvNext(x, y, u) => Target(u, v)
    BY <1>4, <1>9
<1>11. ASSUME NEW p, NEW q, NEW u, NEW v
       PROVE NewNext(p, q, u, v) <=> /\ NewSysNext(p, q, v)
                                     /\ NewEnvNext(p, q, u)
    <2>1. NewNext(p, q, u, v)  <=>  /\ NewSysNext(p, q, v)
                                    /\ EnvNext(p, q, u)
        BY <1>6
    <2>2. NewSysNext(p, q, v) => ENABLED NewSysNext(p, q, y')
        OBVIOUS
    <2>3. NewNext(p, q, u, v)  <=>  /\ NewSysNext(p, q, v)
                                    /\ EnvNext(p, q, u)
                                    /\ ENABLED NewSysNext(p, q, y')
        BY <2>1, <2>2
    <2> QED
        BY <2>3, <1>9
<1>12. A <=>
        \E v:  \A u:
            /\ NewSysNext(x, y, v)
            /\ NewEnvNext(x, y, u) => /\ NewNext(x, y, u, v)
                                      /\ Target(u, v)
    BY <1>10, <1>11, CPreSimplerByConjunctivity
<1>13. ASSUME NEW p, NEW q, NEW u, NEW v
       PROVE NewNext(p, q, u, v) => Next(p, q, u, v)
    <2>1. SUFFICES ASSUME NewNext(p, q, u, v)
                   PROVE Next(p, q, u, v)
        OBVIOUS
    <2>2. /\ SysNext(p, q, v) /\ EnvNext(p, q, u)
          /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
        BY <2>1 DEF NewNext
    <2>3. /\ EnvNext(p, q, u)
          /\ \A r:  EnvNext(p, q, r) => Next(p, q, r, v)
        BY <2>2
    <2> QED
        BY <2>3  (* goal from <2>1 *)
<1> QED
    BY <1>7, <1>10, <1>11, <1>12, <1>13 DEF NewNext, NewEnvNext


COROLLARY
    ASSUME
        VARIABLE p, VARIABLE q,
        CONSTANT Next(_, _, _, _),
        CONSTANT Target(_, _)
    PROVE
        LET
            SysNext(x, y, v) == \E u:  Next(x, y, u, v)
            EnvNext(x, y, u) == \E v:  Next(x, y, u, v)
            NewNext(x, y, u, v) ==
                /\ SysNext(x, y, v) /\ EnvNext(x, y, u)
                /\ \A w:  EnvNext(x, y, w) => Next(x, y, w, v)
            NewSysNext(x, y, v) == \E u:  NewNext(x, y, u, v)
            NewEnvNext(x, y, u) == \E v:  NewNext(x, y, u, v)
            A(x, y) == \E v:  \A u:
                /\ SysNext(x, y, v)
                /\ EnvNext(x, y, u) => /\ Next(x, y, u, v)
                                       /\ Target(u, v)
        IN
            /\ NewNext(p, q, p', q') => Next(p, q, p', q')
                (* Conjunctivity and Cartesianity *)
            /\ NewNext(p, q, p', q')
                <=> /\ NewSysNext(p, q, q')
                    /\ NewEnvNext(p, q, p')
            /\ A(p, q) <=> \E v:  \A u:
                /\ NewSysNext(p, q, v)
                /\ NewEnvNext(p, q, u) => /\ NewNext(p, q, u, v)
                                          /\ Target(u, v)
            /\ A(p, q) <=> \E v:  \A u:
                /\ NewSysNext(p, q, v)
                /\ NewEnvNext(p, q, u) => Target(u, v)
PROOF
    BY EquiEnablednessFromUnzip, SeparatingTheRealizablePart

--------------------------------------------------------------------------------
(* Unzip has desirable properties:

1. the assumption is by construction safety, and
2. the assumption is well-separated.

Recall that:
    Unzip(P) <=> WPH(WPH(P, P), P)
*)


(* The assumption in the WhilePlusHalf that defines Unzip is a safety property.
That this property, namely WPH(C, C, y, x), is safety follows similarly to
the proof of WhilePlusMachineClosedRepr.
*)
PROPOSITION
    ASSUME
        TEMPORAL P(_, _),
        VARIABLE x, VARIABLE y
    PROVE
        LET C == Cl(P, x, y)
        IN WPH(P, P, y, x) <=> WPH(C, C, y, x)
PROOF
<1> DEFINE
    C == Cl(P, x, y)
<1>1. WPH(P, P, y, x) <=> /\ WPH(C, C, y, x)
                          /\ P(y, x) => P(y, x)
    BY WhilePlusHalfAsConj
<1>2. P(y, x) => P(y, x)
    OBVIOUS
<1> QED
    BY <1>1, <1>2


(* This proposition ensures well-separation of the first and second argument
given to WhilePlusHalf for defining Unzip.
*)
PROPOSITION
    ASSUME
        TEMPORAL P(_, _),
        VARIABLE x, VARIABLE y
    PROVE
        LET
            Q(u, v) == P(v, u)
            E(u, v) == WPH(Q, Q, v, u)
        IN
            /\ Cl(P, x, y) => Cl(E, x, y)
            /\ P(x, y) => Cl(E, x, y)
PROOF
<1> DEFINE
    Q(u, v) == P(v, u)
    E(u, v) == WPH(Q, Q, v, u)
<1>1. P(x, y) => WPH(Q, Q, y, x)
    <2>1. WPH(Q, Q, y, x) <=>
        \AA b:  \/ ~ /\ MayUnstep(b)
                     /\ Front(Q, y, x, b)
                \/ FrontPlusHalf(Q, y, x, b)
        BY DEF WPH, WhilePlusHalf
    <2>2. ASSUME VARIABLE b
          PROVE P(x, y)  =>  FrontPlusHalf(Q, y, x, b)
        <3>1. FrontPlusHalf(Q, y, x, b)
                <=> \EE u, v:
                        /\ Q(u, v)
                        /\ SamePrefix(b, u, v, y, x)
                        /\ PlusHalf(b, v, x)
            BY DEF FrontPlusHalf
        <3>2. ASSUME VARIABLE u, VARIABLE v
              PROVE
                SamePrefix(b, u, v, y, x)
                <=> SamePrefix(b, v, u, x, y)
            BY SwapInSamePrefix
        <3>3. FrontPlusHalf(Q, y, x, b)
                <=> \EE v, u:
                        /\ P(v, u)
                        /\ SamePrefix(b, v, u, x, y)
                        /\ PlusHalf(b, v, x)
            BY <3>1, <3>2
        <3>4. FrontPlusHalf(Q, y, x, b)
                <=> \EE u, v:
                        /\ P(u, v)
                        /\ SamePrefix(b, u, v, x, y)
                        /\ PlusHalf(b, u, x)
            BY <3>3
        <3>5. P(x, y) => \EE u, v:
                /\ [](<< u, v >> = << x, y >>)
                /\ P(x, y)
            OBVIOUS
        <3>6. P(x, y) => \EE u, v:
                /\ [](<< u, v >> = << x, y >>)
                /\ u = x
                /\ P(u, v)
            BY <3>5
        <3>7. P(x, y) => \EE u, v:
                /\ [](<< u, v >> = << x, y >>)
                /\ u = x
                /\ [][u' = x']_<< b, u, x >>
                /\ P(u, v)
            BY <3>6  (* TLA rule *)
        <3>8. P(x, y) => \EE u, v:
                /\ [](b => (<< u, v >> = << x, y >>))
                /\ u = x
                /\ [][b => (u' = x')]_<< b, u, x >>
                /\ P(u, v)
            BY <3>7
        <3> QED
            BY <3>4, <3>8
    <2> QED
        BY <2>1, <2>2
<1>2. P(x, y) => E(x, y)
    BY <1>1 DEF E
<1>3. P(x, y) => Cl(E, x, y)
    <2>1. E(x, y) => Cl(E, x, y)
        BY ClosureImplied
    <2> QED
        BY <1>2, <2>1
<1>4. Cl(P, x, y) => Cl(E, x, y)
    <2>1. Cl(P, x, y) => Cl(Cl(E, x, y), x, y)
        BY <1>3, ClosureIsMonotonic
    <2>2. Cl(E, x, y) <=> Cl(Cl(E, x, y), x, y)
        BY ClosureIdempotent
    <2> QED
        BY <2>1, <2>2
<1> QED
    BY <1>3, <1>4 DEF E


(* Expand an expression that occurs in the first argument of WhilePlusHalf
within Unzip.
*)
PROPOSITION
    ASSUME
        TEMPORAL P(_, _),
        VARIABLE x, VARIABLE y, VARIABLE b
    PROVE
        LET
            Q(u, v) == P(v, u)
        IN
            Front(Q, y, x, b) <=> Front(P, x, y, b)
PROOF
<1> DEFINE
    Q(u, v) == P(v, u)
<1>1. Front(Q, y, x, b)
        <=> \EE u, v:  /\ Q(u, v)
                       /\ SamePrefix(b, u, v, y, x)
    BY DEF Front
<1>2. ASSUME VARIABLE u, VARIABLE v
      PROVE
        SamePrefix(b, u, v, y, x)
        <=> SamePrefix(b, v, u, x, y)
    BY SwapInSamePrefix
<1>3. ASSUME VARIABLE u, VARIABLE v
      PROVE Q(u, v) <=> P(v, u)
    BY DEF Q
<1>4. Front(Q, y, x, b)
        <=> \EE u, v:  /\ P(v, u)
                       /\ SamePrefix(b, v, u, x, y)
    BY <1>1, <1>2, <1>3
<1>5. Front(Q, y, x, b)
        <=> \EE v, u:  /\ P(v, u)
                       /\ SamePrefix(b, v, u, x, y)
    BY <1>4
<1>6. Front(P, x, y, b)
        <=> \EE v, u:  /\ P(v, u)
                       /\ SamePrefix(b, v, u, x, y)
    BY DEF Front
<1> QED
    BY <1>5, <1>6


--------------------------------------------------------------------------------
THEOREM NotExtensible ==
    ASSUME
           \E tau:  /\ IsABehavior(tau)
                    /\ tau |= B
                    /\ tau[0].a = 1
                    /\ tau[1].a = 20
                    /\ tau[0].b = 2
    PROVE FALSE
PROOF
<1>3. PICK tau:
        /\ IsABehavior(tau)
        /\ tau |= B
        /\ LET
               s0 == tau[0]
               s1 == tau[1]
           IN
               /\ s0.a = 1
               /\ s1.a = 20
               /\ s0.b = 2
<1> DEFINE
    s0 == tau[0]
    s1 == tau[1]
    IsNonstuttering(step) == step[1] # step[2]
<1>4. tau |= []<>(b = 2)
    BY <1>3 DEF B
<1>1. IsNonstuttering(<< s0, s1 >>)
    <2>1. s0.a # s1.a
        BY DEF s0, s1
    <2> QED
        BY <2>1 DEF IsNonstuttering
<1>2. s1.b = 1
    BY <1>1 DEF B, s0
<1>5. \E i \in Nat:  tau[i].b # tau[i + 1].b
    (* A step that changes b eventually occurs. *)
    BY <1>2, <1>4 DEF s1
<1>6. \A n \in Nat:
        \/ tau[n] = tau[n + 1]
        \/ tau[n] # s1
        \/ << tau[n], tau[n + 1] >>[[ b' # b ]]
    (* Any nonstuttering step from $s_1$ must change $b$. *)
    BY <1>2, <1>3 DEF s1, B
<1>7. \E j \in Nat:
        /\ \A k \in 1..j:  tau[k] = s1
        /\ << tau[j], tau[j + 1] >>[[ b' # b ]]
    (* The earliest nonstutering step after $tau[1]$ does change $b$. *)
    BY <1>5, <1>6, LeastNumberPrinciple
<1>8. tau[j + 1].b = 20
    <2>1. tau[j].a = 20
        <3>1. tau[j] = s1
            BY <1>7
        <3>2. s1.a = 20
            BY <1>3 DEF s1
        <3> QED
            BY <3>1, <3>2
    <2>2. << tau[j], tau[j + 1] >>[[ b' = a ]]
        <3>1. << tau[j], tau[j + 1] >>[[ b' # b ]]
            BY <1>7
        <3>2. tau |= B
            BY <1>3
        <3> QED
            BY <3>1, <3>2 DEF B
    <2> QED
        BY <2>1, <2>2
<1>9. tau[j + 1].b \in 1..2
    <2>1. tau |= B
        BY <1>3
    <2> QED
        BY <2>1 DEF B
<1> QED
    BY <1>8, <1>9
================================================================================
