----------------------- MODULE WhilePlusHalfTheorems ---------------------------
(* Properties of the operator WhilePlusHalf, a variant of -+->.

Below is a proof of the stepwise form of WhilePlusHalf in raw TLA+ with past
temporal operators. This module includes definitions that are relevant to
the family of stepwise implication operators. These definitions include
syntactic and semantic definitions of these operators.

Author:  Ioannis Filippidis

--------------------------------------------------------------------------------
Copyright 2016-2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS
    TLASemantics, TemporalLogic, TemporalQuantification,
    NaturalsInduction, TLAPS


PROPOSITION ShorterPrefixSat ==
    ASSUME
        NEW n \in Nat,
        NEW sigma, IsABehavior(sigma),
        TEMPORAL G
    PROVE
        PrefixSat(sigma, n + 1, G) => PrefixSat(sigma, n, G)
    PROOF
    <1>1. SUFFICES ASSUME PrefixSat(sigma, n + 1, G)
                   PROVE PrefixSat(sigma, n, G)
        OBVIOUS
    <1>2. \E tau:
            /\ IsABehavior(tau)
            /\ \A i \in 0..((n + 1) - 1):  tau[i] = sigma[i]
            /\ tau |= G
        BY <1>1 DEF PrefixSat
    <1>3. \E tau:
            /\ IsABehavior(tau)
            /\ \A i \in 0..n:  tau[i] = sigma[i]
            /\ tau |= G
        BY <1>2
    <1>4. ASSUME
            NEW tau,
            \A i \in 0..n:  tau[i] = sigma[i]
          PROVE
            \A i \in 0..(n - 1):  tau[i] = sigma[i]
        <2>4. n \in Nat
            OBVIOUS  (* BY ShorterPrefixSat *)
        <2>1. SUFFICES ASSUME NEW i \in 0..(n - 1)
                       PROVE tau[i] = sigma[i]
            OBVIOUS
        <2>2. ASSUME n = 0
              PROVE FALSE
            <3>1. (n - 1) = -1
                BY <2>2
            <3>2. 0..(n - 1) = {}
                BY <3>1
            <3>3. i \in {}
                BY <3>2, <2>1
            <3> QED
                BY <3>3
        <2>3. CASE n > 0
            <3>1. (n - 1) \in 0..n
                BY <2>4, <2>3
            <3>2. 0..(n - 1) \subseteq 0..n
                BY <3>1
            <3>3. i \in 0..n
                BY <2>1, <3>2
            <3> QED
                BY <3>3, <1>4
        <2> QED
            BY <2>2, <2>3, <2>4
    <1>5. \E tau:
            /\ IsABehavior(tau)
            /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
            /\ tau |= G
        BY <1>3, <1>4
    <1> QED
        BY <1>5 DEF PrefixSat


PROPOSITION
    ASSUME
        NEW n \in Nat,
        NEW sigma, IsABehavior(sigma),
        TEMPORAL A, TEMPORAL G
    PROVE
        PrefixPlusOne(sigma, A, G) <=>
            \A n \in Nat:
                PrefixSat(sigma, n, A) => /\ PrefixSat(sigma, n, G)
                                          /\ PrefixSat(sigma, n + 1, G)
    PROOF
    <1>1. PrefixPlusOne(sigma, A, G) <=>
            \A n \in Nat:
                PrefixSat(sigma, n, A)
                => PrefixSat(sigma, n + 1, G)
        BY DEF PrefixPlusOne
    <1>2. PrefixSat(sigma, n + 1, G) => PrefixSat(sigma, n, G)
        BY ShorterPrefixSat
    <1> QED
        BY <1>1, <1>2


--------------------------------------------------------------------------------
(* Semantic definition of "while" operators. *)

(* The semantic and syntactic definitions of -+-> and WPH are equivalent,
despite the semantic ones omitting stutter-equivalence. This ows to the fact
that temporal quantification serves for only replacing the behavior's tail,
not for step-refinement.
*)


(* The While operator from the module TLASemantics.
Copied here for comparison.
*)
(*
sigma |= While(A, G) ==
    /\ \A n \in Nat:  PrefixSat(sigma, n, A) => PrefixSat(sigma, n, G)
    /\ sigma |= A => G
*)


(* The WhilePlus operator. Compied here for comparison. *)
(*
PrefixPlusOne(sigma, A, G) ==
    \A n \in Nat:  PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)

sigma |= A -+-> G ==
    /\ PrefixPlusOne(sigma, A, G)
    /\ sigma |= A => G
*)


(* Attention: the signature of the operator is in the object language (TLA+),
but the definition is in the metatheory. Thus, x and y need delicate handling.
*)
sigma |= WhilePlusHalf(A, G, x, y) <=>  (* notice this is <=>, not == *)
    LET
        SamePrefixSatXY(tau, n, H) ==
            /\ IsABehavior(tau)
            /\ tau |= H
            /\ \A i \in 0..(n - 1):
                /\ tau[i].x = sigma[i].x
                /\ tau[i].y = sigma[i].y
        PrefixSatVar(n, H) ==
            \E tau:  SamePrefixSatXY(tau, n, H)
        PrefixSatVarPlusHalf(n, H) ==
            \E tau:  /\ SamePrefixSatXY(tau, n, H)
                     /\ tau[n].y = sigma[n].y
    IN
        /\ sigma |= F => G
        /\ \A n \in Nat:
            PrefixSatVar(n, F) => PrefixSatVarPlusHalf(n, G)

(*
The semantic definitions of $WPH$ and -+-> both are meaningful in
raw \tlaplus. The syntactic definitions are equivalent to the semantic
definitions within TLA+. The syntactic definitions are equivalent to the
semantic ones also within raw TLA+, even after replacing temporal
quantification by its stutter-sensitive version.

The reason is the same as mentioned above: the definitions utilize temporal
quantification for only hiding the behavior's tail; not for step-refinement.
*)

(* Each operator could be defined in roughly three ways:
within TLA+ (e.g., WhilePlus), which is also sensible within raw TLA+,
within raw TLA+ (e.g., RawWhilePlus), and
in the metatheory, with the below definition as a demonstration:

MetaWhilePlusHalf(sigma, A, G, x, y) ==
    LET
        SamePrefixSatXY ...
*)

--------------------------------------------------------------------------------
(* Contents of module OpenSystems (refactored).

These are syntactic definitions for the family of "while" operators.
*)


(* Variable b starts in BOOLEAN and changes at most once to FALSE. *)
MayUnstep(b) == /\ b \in BOOLEAN
                /\ [][b' = FALSE]_b
(* Variable b starts in BOOLEAN and becomes FALSE with at most one change. *)
Unstep(b) == /\ MayUnstep(b)
             /\ <>(b = FALSE)
(* Variable b starts TRUE and changes once to FALSE. *)
MustUnstep(b) == /\ b = TRUE
                 /\ Unstep(b)

(* Redefined from module TemporalLogic to change arity. *)
SamePrefix(b, u, v, x, y) == [](b => (<< u, v >> = << x, y >>))
PlusHalf(b, v, y) == /\ v = y
                     /\ [][ b => (v' = y') ]_<< b, v, y >>
(* Redefined from module TemporalLogic to change arity. *)
Front(P(_, _), x, y, b) ==
    \EE u, v:
        /\ P(u, v)
        /\ SamePrefix(b, u, v, x, y)
FrontPlusHalf(P(_, _), x, y, b) ==
    \EE u, v:
        /\ P(u, v)
        /\ SamePrefix(b, u, v, x, y)
        /\ PlusHalf(b, v, y)
FrontPlus(P(_, _), x, y, b) == \EE u, v:
    LET
        vars == << b, x, y, u, v >>
        Init == << u, v >> = << x, y >>
        Next == b => (<< u', v' >> = << x', y' >>)
        Plus == [][ Next ]_vars
    IN
        /\ P(u, v)
        /\ Init /\ Plus

(* An additional definition (not in the module OpenSystems).
This is a syntactic definition of the While operator.
*)
While(A(_, _), G(_, _), x, y) ==
    \AA b:
        (MayUnstep(b) /\ Front(A, x, y, b)) => Front(G, x, y, b)

(* The TLA+ operator -+-> expressed within the logic [1, p.337].
[1] Leslie Lamport, "Specifying systems", Addison-Wesley, 2002
*)
WhilePlus(A(_, _), G(_, _), x, y) ==
    \AA b:
        (MayUnstep(b) /\ Front(A, x, y, b)) => FrontPlus(G, x, y, b)

(* A variant of the WhilePlus operator. *)
WhilePlusHalf(A(_, _), G(_, _), x, y) ==
    \AA b:
        (MayUnstep(b) /\ Front(A, x, y, b)) => FrontPlusHalf(G, x, y, b)

(* An operator that forms an open system from the closed system that the
temporal property P(x, y) describes.
*)
Unzip(P(_, _), x, y) ==
    LET
        Q(u, v) == P(v, u)  (* swap back to x, y *)
        A(u, v) == WhilePlusHalf(Q, Q, v, u)  (* swap to y, x *)
    IN
        WhilePlusHalf(A, P, x, y)


PROPOSITION SwapInSamePrefix ==
    ASSUME
        VARIABLE u, VARIABLE v, VARIABLE x, VARIABLE y
    PROVE
        SamePrefix(b, u, v, y, x)
        <=> SamePrefix(b, v, u, x, y)
    PROOF
    <1>1. ASSUME VARIABLE u, VARIABLE v
          PROVE
            SamePrefix(b, u, v, y, x)
            <=> [](b => (<< u, v >> = << y, x >>))
        BY DEF SamePrefix
    <1>2. ASSUME VARIABLE u, VARIABLE v
          PROVE
            (<< u, v >> = << y, x >>)
            <=> (<< v, u >> == << x, y >>)
        OBVIOUS
    <1> QED
        BY <1>1, <1>2
--------------------------------------------------------------------------------

(* How quantification of initial conditions is handled distinguishes between
a disjoint-state specification (\E \A) and a shared-state specification
(\E \E or \A \A). *)


--------------------------------------------------------------------------------

(* Below we use a definition of closure that takes three arguments. *)
Cl(P(_), x, y) == \AA b:  MustUnstep(b) => Front(P, x, y, b)

WPH(A, G, x, y) == WhilePlusHalf(A, G, x, y)


(* analogous to ClosureEquiSAT *)
PROPOSITION ClosureEquiSATHalf ==
    ASSUME
        VARIABLE y,
        TEMPORAL P(_, _)
    PROVE
        (\EE u, v:  (v = y) /\ P(u, v))
        <=>  \EE u, v:  (v = y) /\ Cl(P, u, v)
    PROOF
    <1> DEFINE
        ClP(u, v) == Cl(P, u, v)
    <1>1. \/ ~ \EE u, v:  /\ v = y
                          /\ P(u, v)
          \/ \EE u, v:  /\ v = y
                        /\ Cl(P, u, v)
        BY ClosureImplied
    <1>2. \/ ~ \EE u, v:  /\ v = y
                          /\ Cl(P, u, v)
          \/ \EE u, v:  /\ v = y
                        /\ P(u, v)
        <2> DEFINE R(v, y) == v = y
        <2>1. SUFFICES
                \/ ~ \EE u, v:  R(v, y) /\ Cl(P, u, v))
                \/ \EE u, v:  R(v, y) /\ P(u, v)
            BY DEF R
        <2> QED
            BY ClosureSample  (* goal from <2>1 *)
    <1> QED
        BY <1>1, <1>2


PROPOSITION ReplaceWithClosureWithinFront ==
    ASSUME
        VARIABLE x, VARIABLE y, VARIABLE b,
        TEMPORAL P(_, _)
    PROVE
        LET
            Fr(P(_, _), b) == Front(P, x, y, b)
            ClP(u, v) == Cl(P, u, v)
        IN
            \/ ~ MustUnstep(b)
            \/ Fr(P, b) <=> Fr(ClP, b)
    PROOF
    <1> DEFINE
        Fr(P(_, _), b) == Front(P, x, y, b)
        ClP(u, v) == Cl(P, u, v)
    <1>1. Fr(P, b)
            <=> \EE u, v:  /\ P(u, v)
                           /\ SamePrefix(b, u, v, x, y)
        BY DEF Fr, Front
    <1>2. Fr(ClP, b)
            <=> \EE u, v:  /\ ClP(u, v)
                           /\ SamePrefix(b, u, v, x, y)
        BY DEF Fr, Front
    <1>3. Fr(P, b) => Fr(ClP, b)
        <2>1. ASSUME VARIABLE u, VARIABLE v
              PROVE P(u, v) => ClP(u, v)
            BY ClosureImplied
        <2>2. Fr(P, b)
                <=> \EE u, v:  /\ P(u, v) /\ ClP(u, v)
                               /\ SamePrefix(b, u, v, x, y)
            BY <1>1, <2>1
        <2>3. Fr(P, b)
                => \EE u, v:  /\ ClP(u, v)
                              /\ SamePrefix(b, u, v, x, y)
            BY <2>2
        <2> QED
            BY <2>3, <1>2
    <1>4. \/ ~ MustUnstep(b)
          \/ Fr(ClP, b) => Fr(P, b)
        <2>1. Fr(ClP, b)
                <=> \EE u, v:  /\ \AA r:  \/ ~ MustUnstep(r)
                                          \/ Front(P, u, v, r)
                               /\ SamePrefix(b, u, v, x, y)
            BY <1>2 DEF ClP, Cl
        <2>2. Fr(ClP, b)
                => \EE u, v:  /\ \/ ~ MustUnstep(b)
                                 \/ Front(P, u, v, b)
                              /\ SamePrefix(b, u, v, x, y)
            BY <2>1, InstantiateAA
        <2>3. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE u, v:  /\ Front(P, u, v, b)
                            /\ SamePrefix(b, u, v, x, y)
            BY <2>2
        <2>4. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE u, v:  /\ \EE p, q:  /\ P(p, q)
                                          /\ SamePrefix(b, p, q, u, v)
                            /\ SamePrefix(b, u, v, x, y)
            BY <2>3 DEF SamePrefix
        <2>5. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE u, v, p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, u, v)
                    /\ SamePrefix(b, u, v, x, y)
            BY <2>4
        <2>6. ASSUME
                VARIABLE u, VARIABLE v, VARIABLE p, VARIABLE q
              PROVE
                \/ ~ /\ SamePrefix(b, p, q, u, v)
                     /\ SamePrefix(b, u, v, x, y)
                \/ SamePrefix(b, p, q, x, y)
            <3>1. SUFFICES
                  \/ ~ /\ [](b => (<< p, q >> = << u, v >>))
                       /\ [](b => (<< u, v >> = << x, y >>))
                  \/ [](b => (<< p, q >> = << x, y >>))
                BY DEF SamePrefix
            <3>2. ( /\ [](b => (<< p, q >> = << u, v >>))
                    /\ [](b => (<< u, v >> = << x, y >>))
                  ) <=> (
                    [](b => /\ << p, q >> = << u, v >>
                            /\ << u, v >> = << x, y >>)
                  )
                BY PTL
            <3>3. \/ ~ [](b => /\ << p, q >> = << u, v >>
                               /\ << u, v >> = << x, y >>)
                  \/ [](b => (<< p, q >> = << x, y >>))
                BY PTL
            <3> QED
                BY <3>2, <3>3  (* goal from <3>1 *)
        <2>7. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE u, v, p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, x, y)
            BY <2>5, <2>6
        <2>8. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, x, y)
            BY <2>7
        <2>9. \/ ~ MustUnstep(b)
              \/ ~ Fr(ClP, b)
              \/ \EE u, v:
                    /\ P(u, v)
                    /\ SamePrefix(b, u, v, x, y)
            BY <2>8  (* rename the bound variables p, q to u, v *)
        <2>10. \/ ~ MustUnstep(b)
               \/ ~ Fr(ClP, b)
               \/ Fr(P, b)
            BY <2>9, <1>1
        <2> QED
            BY <2>10
    <1> QED
        BY <1>3, <1>4


PROPOSITION ReplaceWithClosureWithinFrontPlusHalf ==
    ASSUME
        VARIABLE x, VARIABLE y, VARIABLE b,
        TEMPORAL P(_, _)
    PROVE
        LET
            FPH(P(_, _), b) == FrontPlusHalf(P, x, y, b)
            ClP(u, v) == Cl(P, u, v)
        IN
            \/ ~ MustUnstep(b)
            \/ FPH(P, b) <=> FPH(ClP, b)
    PROOF
    <1> DEFINE
        FPH(P(_, _), b) == FrontPlusHalf(P, x, y, b)
        ClP(u, v) == Cl(P, u, v)
    <1>1. FPH(P, b)
            <=> \EE u, v:  /\ P(u, v)
                           /\ SamePrefix(b, u, v, x, y)
                           /\ PlusHalf(b, v, y)
        BY DEF FPH, FrontPlusHalf
    <1>2. FPH(ClP, b)
            <=> \EE u, v:  /\ ClP(u, v)
                           /\ SamePrefix(b, u, v, x, y)
                           /\ PlusHalf(b, v, y)
        BY DEF FPH, FrontPlusHalf
    <1>3. FPH(P, b) => FPH(ClP, b)
        <2>1. ASSUME VARIABLE u, VARIABLE v
              PROVE P(u, v) => ClP(u, v)
            BY ClosureImplied
        <2> QED
            BY <1>1, <2>1, <1>2
    <1>4. \/ ~ MustUnstep(b)
          \/ ~ FPH(ClP, b)
          \/ FPH(P, b)
        <2>1. FPH(ClP, b)
                <=> \EE u, v:  /\ \AA r:  \/ ~ MustUnstep(r)
                                          \/ Front(P, u, v, r)
                               /\ SamePrefix(b, u, v, x, y)
                               /\ PlusHalf(b, v, y)
            BY <1>2 DEF ClP, Cl
        <2>2. \/ ~ MustUnstep(b)
              \/ \EE z:  /\ MustUnstep(z)
                         /\ [](b => z)
                         /\ [][z' = b]_<< b, z >>
            <3>1. SUFFICES
                    ASSUME
                        NEW sigma, IsABehavior(sigma),
                        sigma |= MustUnstep(b)
                    PROVE sigma |= \EE z:  /\ MustUnstep(z)
                                           /\ [](b => z)
                                           /\ [][z' = b]_<< b, z >>
                OBVIOUS
            <3>2. tau == [n \in Nat |->
                    [sigma[n] EXCEPT !["z"] =
                        IF n = 0 THEN TRUE
                        ELSE sigma[n - 1]["b"]]
                    (* a one-step delay *)
            <3>3. IsABehavior(tau)
                BY <3>1 DEF tau
            <3>4. tau |= MustUnstep(b)
                <4>1. sigma |= MustUnstep(b)
                    BY <3>1
                <4>2. \A n \in Nat:  tau[n]["b"] = sigma[n]["b"]
                    BY DEF tau
                <4> QED
                    BY <4>1, <4>2
            <3>5. EqualUpToVar(tau, sigma, "z")
                BY DEF EqualUpToVar, tau
            <3>6. Sim(sigma, sigma)
                BY DEF Sim
            <3>7. CHOOSE k \in Nat:
                    /\ \A n \in 0..k:  tau[n]["b"] = TRUE
                    /\ \A n \in Nat:  (n > k)  =>  (tau[n]["b"] = FALSE)
                BY <3>4 DEF MustUnstep, Unstep, MayUnstep
            <3>8. LET m == k + 1
                  IN /\ \A n \in 0..m:  tau[n]["z"] = TRUE
                     /\ \A n \in Nat:  (n > m)  =>  (tau[n]["z"] = FALSE)
                BY <3>7 DEF tau
            <3>9. tau |= MustUnstep(z)
                BY <3>8 DEF MustUnstep, Unstep, MayUnstep
            <3>10. tau |= /\ [](b => z)
                          /\ [][z' = b]_<< b, z >>
                <4>1. /\ \A n \in 0..k:  tau[n]["b"] = TRUE
                      /\ \A n \in 0..k:  tau[n]["z"] = TRUE
                      /\ \A n \in Nat:  (n > k)  =>  (tau[n]["b"] = FALSE)
                    BY <3>7, <3>8
                <4>2. \A n \in Nat:  (tau[n]["b"] => tau[n]["z"])
                        (* Writing
                            (tau[n]["b"] = TRUE) => (tau[n]["z"] = TRUE)
                        would not lead to the desired conclusion below,
                        unless we invoked the type invariant. *)
                    BY <4>1
                <4>3. \A n \in Nat:  tau[n] |= (b => z)
                    BY <4>2
                <4>4. \A n \in Nat:  << tau[n], tau[n + 1] >> |= z' = b
                    BY DEF tau
                <4>5. \A n \in Nat:
                        << tau[n], tau[n + 1] >> |= [z' = b]_<< b, z >>
                    BY <4>4
                <4> QED
                    BY <4>3, <4>5
            <3>11. tau |= /\ MustUnstep(z)
                          /\ [](b => z)
                          /\ [][z' = b]_<< b, z >>
                BY <3>9, <3>10
            <3>12. /\ IsABehavior(tau)
                   /\ /\ IsABehavior(sigma)  (* RefinesUpToVar *)
                      /\ Sim(sigma, sigma)
                      /\ EqualUpToVar(sigma, tau, "z")
                   /\ tau |= MustUnstep(z) /\ [](b => z)
                <4>1. IsABehavior(tau)
                    BY <3>3
                <4>2. IsABehavior(sigma)
                    BY <3>1
                <4>3. EqualUpToVar(tau, sigma, "z")
                    BY <3>5
                <4>4. Sim(sigma, sigma)
                    BY <3>6
                <4> QED
                    BY <4>1, <4>2, <4>3, <4>4, <3>11
            <3> QED
                BY <3>12 DEF RefinesUpToVar, \EE
        <2>3. \/ ~ MustUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v:
                    /\ \EE z:  /\ MustUnstep(z)
                               /\ [](b => z)
                               /\ [][z' = b]_<< b, z >>
                    /\ \AA r:  \/ ~ MustUnstep(r)
                               \/ Front(P, u, v, r)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>1, <2>2
        <2>4. \/ ~ MustUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v, z:
                    /\ MustUnstep(z)
                    /\ [](b => z)
                    /\ [][z' = b]_<< b, z >>
                    /\ \AA r:  \/ ~ MustUnstep(r)
                               \/ Front(P, u, v, r)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>3
        <2>5. \/ ~ MusUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v, z:
                    /\ MustUnstep(z)
                    /\ [](b => z)
                    /\ [][z' = b]_<< b, z >>
                    /\ \/ ~ MustUnstep(z)
                       \/ Front(P, u, v, z)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>4
        <2>6. \/ ~ MusUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v, z:
                    /\ MustUnstep(z)
                    /\ [](b => z)
                    /\ [][z' = b]_<< b, z >>
                    /\ Front(P, u, v, z)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>5, InstantiateAA
        <2>7. \/ ~ MusUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v, z:
                    /\ MustUnstep(z)
                    /\ [](b => z)
                    /\ [][z' = b]_<< b, z >>
                    /\ \EE p, q:
                        /\ P(p, q)
                        /\ SamePrefix(z, p, q, u, v)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>6 DEF Front
        <2>8. \/ ~ MusUnstep(b)
              \/ ~ FPH(ClP, b)
              \/ \EE u, v, z, p, q:
                    /\ MustUnstep(z)
                    /\ [](b => z)
                    /\ [][z' = b]_<< b, z >>
                    /\ P(p, q)
                    /\ SamePrefix(z, p, q, u, v)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
            BY <2>7
        <2>9. ASSUME
                  VARIABLE z, VARIABLE p, VARIABLE q,
                  VARIABLE u, VARIABLE v
              PROVE
                  \/ ~ /\ SamePrefix(z, p, q, u, v)
                       /\ SamePrefix(b, u, v, x, y)
                       /\ [](b => z)
                  \/ SamePrefix(b, p, q, x, y)
            BY DEF SamePrefix
        <2>10. ASSUME
                  VARIABLE z, VARIABLE p, VARIABLE q,
                  VARIABLE u, VARIABLE v
               PROVE
                    \/ ~ /\ MustUnstep(z)
                         /\ SamePrefix(z, p, q, u, v)
                         /\ PlusHalf(b, v, y)
                    \/ q = y
            <3>1. PlusHalf(b, v, y) => (v = y)
                BY DEF PlusHalf
            <3>2. \/ ~ SamePrefix(z, p, q, u, v)
                  \/ z => (q = v)
                BY DEF SamePrefix
            <3>3. MustUnstep(z) => (z = TRUE)
                BY DEF MustUnstep
            <3> QED
                BY <3>1, <3>2, <3>3
        <2>11. ASSUME
                  VARIABLE z, VARIABLE p, VARIABLE q,
                  VARIABLE u, VARIABLE v
               PROVE
                   \/ ~ /\ [](b => z)
                        /\ [][z' = b]_<< b, z >>
                        /\ SamePrefix(z, p, q, u, v)
                        /\ PlusHalf(b, v, y)
                   \/ [][b => (y' = q')]_<< b, y, q >>

            <3>1. \/ ~ /\ [](b => z)
                       /\ [][z' = b]_<< b, z >>
                  \/ [][b => z']_<< b, v, y, q >>
                <4>1. SUFFICES
                        ASSUME
                            (b => z) /\ [z' = b]_<< b, z >>
                        PROVE
                            [b => z']_<< b, v, y, q >>
                    BY PTL
                <4>2. CASE UNCHANGED << b, z >>
                    <5>1. (b => z')  <=>  (b => z)
                        BY <4>2
                    <5>2. b => z'
                        BY <4>1, <5>1
                    <5> QED
                        BY <5>2  (* goal from <4>1 *)
                <4>3. CASE ~ UNCHANGED << b, z >>
                    <5>1. z' = b
                        BY <4>1, <4>3
                    <5>2. b => z'
                        BY <5>1
                    <5> QED
                        BY <5>2
                <4> QED
                    BY <4>2, <4>3

            <3>2. \/ ~ SamePrefix(z, p, q, u, v)
                  \/ [][z' => (v' = q')]_<< b, v, y, q, z >>
                <4>1. \/ ~ SamePrefix(z, p, q, u, v)
                      \/ [](z => (v = q))
                    BY DEF SamePrefix
                <4> QED
                    BY <4>1, PTL

            <3>3. \/ ~ /\ [][b => z']_<< b, v, y, q >>
                       /\ [][z' => (v' = q')]_<< b, v, y, q, z >>
                  \/ [][b => (v' = q')]_<< b, v, y, q >>
                <4>1. SUFFICES
                        ASSUME
                            /\ ~ UNCHANGED << b, v, y, q >>
                            /\ [b => z']_<< b, v, y, q >>
                            /\ [z' => (v' = q')]_<< b, v, y, q, z >>
                        PROVE
                            b => (v' = q')
                    BY PTL
                <4>2. /\ b => z'
                      /\ z' => (v' = q')
                    BY <4>1
                <4> QED
                    BY <4>2

            <3>4. \/ ~ /\ [](b => z)
                       /\ [][z' = b]_<< b, z >>
                       /\ SamePrefix(z, p, q, u, v)
                  \/ /\ [][z' => (v' = q')]_<< b, v, y, q, z >>
                     /\ [][b => z']_<< b, v, y, q >>
                BY <3>1, <3>2

            <3>5. \/ ~ /\ [](b => z)
                       /\ [][z' = b]_<< b, z >>
                       /\ SamePrefix(z, p, q, u, v)
                  \/ [][b => (v' = q')]_<< b, v, y, q >>
                BY <3>3, <3>4

            <3>6. \/ ~ /\ [](b => z)
                       /\ [][z' = b]_<< b, z >>
                       /\ SamePrefix(z, p, q, u, v)
                       /\ PlusHalf(b, v, y)
                  \/ /\ [][b => (v' = q')]_<< b, v, y, q >>
                     /\ [][b => (v' = y')]_<< b, v, y >>
                BY <3>5 DEF PlusHalf

            <3>7. \/ ~ /\ [][b => (v' = q')]_<< b, v, y, q >>
                       /\ [][b => (v' = y')]_<< b, v, y >>
                       /\ SamePrefix(z, p, q, u, v)
                       /\ [](b => z)
                  \/ [][b => (y' = q')]_<< b, y, q >>
                <4>1. SUFFICES
                        ASSUME
                            /\ [b => (v' = q')]_<< b, v, y, q >>
                            /\ [b => (v' = y')]_<< b, v, y >>
                            /\ z => (<< p, q >> = << u, v >>)
                            /\ b => z
                        PROVE
                            [b => (y' = q')]_<< b, y, q >>
                    BY PTL
                <4>2. SUFFICES
                        ASSUME b /\ ~ UNCHANGED << b, y, q >>
                        PROVE y' = q'
                    OBVIOUS  (* goal from <4>1 *)
                <4>3. CASE UNCHANGED q
                    <5>1. ~ UNCHANGED << b, y >>
                        BY <4>2, <4>3
                    <5>2. /\ b => (v' = q')
                          /\ b => (v' = y')
                        BY <5>1, <4>1
                    <5> QED
                        <6>1. b
                            BY <4>2
                        <6> QED
                            BY <5>1, <5>2, <6>1  (* goal from <4>2 *)
                <4>4. CASE ~ UNCHANGED q
                    <5>1. v' # v
                        <6>1. b => (v' = q')
                            BY <4>1, <4>4
                        <6>2. v' = q'
                            BY <6>1, <4>2
                        <6>3. v' # q
                            BY <4>4, <6>2
                        <6>4. q = v
                            BY <4>1, <4>2
                        <6> QED
                            <6>3, <6>4
                    <5>2. /\ b => (v' = q')
                          /\ b => (v' = y')
                        BY <5>1, <4>1
                    <5> QED
                        <6>1. b
                            BY <4>2
                        <6> QED
                            BY <5>1, <5>2, <6>1  (* goal from <4>2 *)
                <4> QED
                    BY <4>3, <4>4

            <3> QED
                BY <3>6, <3>7

        <2>12. \/ ~ MusUnstep(b)
               \/ ~ FPH(ClP, b)
               \/ \EE u, v, z, p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, x, y)
                    /\ q = y
                    /\ [][b => (y' = q')]_<< b, y, q >>
            BY <2>8, <2>9, <2>10, <2>11
        <2>13. \/ ~ MusUnstep(b)
               \/ ~ FPH(ClP, b)
               \/ \EE p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, x, y)
                    /\ q = y
                    /\ [][b => (y' = q')]_<< b, y, q >>
            BY <2>12
        <2>14. \/ ~ MusUnstep(b)
               \/ ~ FPH(ClP, b)
               \/ \EE p, q:
                    /\ P(p, q)
                    /\ SamePrefix(b, p, q, x, y)
                    /\ PlusHalf(b, q, y)
            BY <2>13 DEF PlusHalf
        <2> QED
            BY <2>14, <1>1
    <1> QED
        BY <1>3, <1>4


(* This decomposition is of the same form as that of -+-> .

This fact can be used to prove a safety-liveness decomposition analogous
to the theorem WhilePlusMachineClosedRepr.
*)
THEOREM WhilePlusHalfAsConj ==
    ASSUME
        VARIABLE x, VARIABLE y,
        TEMPORAL A(_, _), TEMPORAL G(_, _)
    PROVE
        LET
            ClA(u, v) == Cl(A, u, v)
            ClG(u, v) == Cl(G, u, v)
        IN
            WPH(A, G, x, y) <=> /\ WPH(ClA, ClG, x, y)
                                /\ A(x, y) => G(x, y)
    PROOF
    <1> DEFINE
        ClA(u, v) == Cl(A, u, v)
        ClG(u, v) == Cl(G, u, v)
        Fr(P(_, _), b) == Front(P, x, y, b)
        FPH(P(_, _), b) == FrontPlusHalf(P, x, y, b)
    <1> USE DEF WPH, WhilePlusHalf, Fr, FPH, Front, FrontPlusHalf,
            SamePrefix, PlusHalf, MustUnstep, MustUnstep

    <1>3. ASSUME
            TEMPORAL Q(_, _), TEMPORAL R(_, _)
          PROVE
            WPH(Q, R, x, y) <=>
                /\ \AA b:  (Fr(Q, b) /\ [](b = TRUE)) => FPH(R, b)
                /\ \AA b:  (Fr(Q, b) /\ MustUnstep(b)) => FPH(R, b)
                /\ \AA b:  (Fr(Q, b) /\ [](b = FALSE)) => FPH(R, b)
        <2>1. MayUnstep(b) <=> \/ [](b = TRUE)
                               \/ MustUnstep(b)
                               \/ [](b = FALSE)
            BY DEF MayUnstep
        <2>2. WPH(Q, R, x, y) <=>
            \AA b:
                /\ (Fr(Q, b) /\ [](b = TRUE)) => FPH(R, b)
                /\ (Fr(Q, b) /\ MustUnstep(b)) => FPH(R, b)
                /\ (Fr(Q, b) /\ [](b = FALSE)) => FPH(R, b)
            BY <2>1 DEF WPH
        <2> QED
            BY <2>2 DEF \AA

    <1>4. ASSUME VARIABLE b
          PROVE  (* The first conjunct of <1>3 is A => G *)
            ( \/ ~ /\ Fr(A, b)
                   /\ [](b = TRUE)
              \/ FPH(G, b))
            <=> (A(x, y) => G(x, y))
        <2>1. ASSUME
                TEMPORAL P(_, _)
              PROVE
                \/ ~ [](b = TRUE)
                \/ P(x, y) <=> \EE u, v:
                    /\ P(u, v)
                    /\ SamePrefix(b, u, v, x, y))
            <3>1. ASSUME VARIABLE u, VARIABLE v
                  PROVE \/ ~ [](b = TRUE)
                        \/ SamePrefix(b, u, v, x, y)
                           <=> [](<< u, v >> = << x, y >>)
                BY DEF SamePrefix
            <3> QED
                BY <3>1 DEF \EE
        <2>2. \/ ~ [](b = TRUE)
              \/ Fr(A, b) <=> A(x, y)
            BY <2>1 DEF Fr
        <2>3. \/ ~ [](b = TRUE)
              \/ FPH(G, b) <=> G(x, y)
            <3>1. FPH(G, b) <=> \EE u, v:
                    /\ G(u, v)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
                BY DEF FPH
            <3>2. ASSUME VARIABLE u, VARIABLE v
                  PROVE
                    \/ ~ [](b = TRUE)
                    \/ SamePrefix(b, u, v, x, y) => PlusHalf(b, v, y)
                BY DEF SamePrefix, PlusHalf
            <3>3. \/ ~ [](b = TRUE)
                  \/ FPH(G, b) <=> \EE u, v:
                        /\ G(u, v)
                        /\ SamePrefix(b, u, v, x, y)
                BY <3>1, <3>2
            <3> QED
                BY <2>1, <3>3
        <2> QED
            <3>1. (( \/ ~ /\ Fr(A, b)
                         /\ [](b = TRUE)
                     \/ FPH(G, b))
                    <=> (A(x, y) => G(x, y)))
                <=>
                    \/ ~ [](b = TRUE)
                    \/ (Fr(A, b) => FPH(G, b))
                        <=> (A(x, y) => G(x, y)))
                OBVIOUS
            <3>2. \/ ~ [](b = TRUE)
                  \/ (Fr(A, b) => FPH(G, b))
                      <=> (A(x, y) => G(x, y)))
                BY <2>2, <2>3
            <3> QED
                BY <3>1, <3>2

    <1>5. ASSUME VARIABLE b
          PROVE (Fr(A, b) /\ MustUnstep(b)) => FPH(G, b)
            <=> (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)
        <2>1. \/ ~ MustUnstep(b)
              \/ Fr(A, b) <=> Fr(ClA, b)
            BY ReplaceWithClosureWithinFront
        <2>2. \/ ~ MustUnstep(b)
              \/ FPH(G, b) <=> FPH(ClG, b)
            BY ReplaceWithClosureWithinFrontPlusHalf
        <2> QED
            BY <2>1, <2>2

    <1>6. ASSUME VARIABLE b
          PROVE (Fr(A, b) /\ [](b = FALSE)) => FPH(G, b)
            <=> (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
        <2>1. \/ ~ [](b = FALSE)
              \/ Fr(A, b) <=> Fr(ClA, b)
            <3>1. ASSUME TEMPORAL P(_, _)
                  PROVE
                    \/ ~ [](b = FALSE)
                    \/ Fr(P, b) <=> \EE u, v:  P(u, v)
                BY DEF Fr, SamePrefix
            <3>2. (\EE u, v:  A(u, v))  <=>  \EE u, v:  ClA(u, v)
                BY ClosureEquiSAT
            <3>3. \/ ~ [](b = FALSE)
                  \/ Fr(A, b) <=> \EE u, v:  A(u, v)
                BY <3>1
            <3>4. \/ ~ [](b = FALSE)
                  \/ Fr(ClA, b) <=> \EE u, v:  ClA(u, v)
                BY <3>1
            <3> QED
                BY <3>2, <3>3, <3>4
        <2>2. \/ ~ [](b = FALSE)
              \/ FPH(G, b) <=> FPH(ClG, b)
            <3>1. ASSUME TEMPORAL P(_, _)
                  PROVE
                    \/ ~ [](b = FALSE)
                    \/ FPH(G, b) <=> \EE u, v:  (v = y) /\ P(u, v)
                BY DEF FPH, SamePrefix, PlusHalf
            <3>2. (\EE u, v:  (v = y) /\ G(u, v))
                    <=> \EE u, v:  (v = y) /\ ClG(u, v)
                BY ClosureEquiSATHalf
            <3>3. \/ ~ [](b = FALSE)
                  \/ FPH(G, b) <=> \EE u, v:  (v = y) /\ G(u, v)
                BY <3>1
            <3>4. \/ ~ [](b = FALSE)
                  \/ FPH(ClG, b) <=> \EE u, v:  (v = y) /\ ClG(u, v)
                BY <3>1
            <3> QED
                BY <3>2, <3>3, <3>4
        <2> QED
            BY <2>1, <2>2

    <1>7. WPH(A, G, x, y) <=>
            /\ A(x, y) => G(x, y)
            /\ \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)
            /\ \AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
        <2>1. (A(x, y) => G(x, y))
                <=> (\AA b:  A(x, y) => G(x, y))
            OBVIOUS
        <2> QED
            BY <1>3, <2>1, <1>5, <1>6

    <1>8. \/ ~ (A(x, y) => G(x, y))
          \/ \AA b:  (Fr(ClA, b) /\ [](b = TRUE)) => FPH(ClG, b)
        <2>1. (A(x, y) => G(x, y))  =>  (Cl(A, x, y) => Cl(G, x, y))
            BY ClosureIsMonotonic
        <2>2. (Cl(A, x, y) => Cl(G, x, y))
            <=> \AA b:  \/ ~ [](b = TRUE)
                        \/ Fr(ClA, b) => FPH(ClG, b)
            (* proof similar to that of <1>4 *)
        <2> QED
            BY <2>1, <2>2

    <1>9. WPH(A, G, x, y) <=>
            /\ A(x, y) => G(x, y)
            /\ \AA b:  (Fr(ClA, b) /\ [](b = TRUE)) => FPH(ClG, b)
            /\ \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)
            /\ \AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
        BY <1>7, <1>8

    <1> QED
        <2>1. WPH(ClA, ClG, x, y) <=>
                /\ \AA b:  (Fr(ClA, b) /\ [](b = TRUE)) => FPH(ClG, b)
                /\ \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)
                /\ \AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
            BY <1>3 DEF ClA, ClG
        <2> QED
            BY <1>9, <2>1


THEOREM WhilePlusHalfSafetyLivenessDecomposition ==
    ASSUME
        VARIABLE x, VARIABLE y,
        TEMPORAL A, TEMPORAL G
    PROVE
        LET
            W == WhilePlusHalf(A, G, x, y)
            C == Cl(A, x, y) -+-> Cl(G, x, y)
        IN
            /\ SafetyPart(W) <=> C
            /\ LivenessPart(W) <=> (C => W)
    PROOF OMITTED  (* similar to WhilePlusSafetyLivenessDecomp. *)


--------------------------------------------------------------------------------
(* Expressing WhilePlusHalf in raw TLA+ with past. *)

(* An operator used to describe WhilePlusHalf in raw TLA+. The arguments
InitA and InitB are not environment and component initial conditions; they
are just appropriately defined predicates.
*)
RawWhilePlusHalf(
        InitA, InitB,
        EnvNext, Next, SysNext,
        Le, Ls) ==
    InitA => /\ InitB
             /\ [](Earlier(EnvNext)  =>  /\ Earlier(Next)
                                         /\ SysNext)
             /\ (ILe /\ []EnvNext)  =>  Ls

(* The conjunctive form of the operator has the advantage of making
reasoning about closure easier, for the particular form of InitA that
arises by translating WPH to raw TLA+.
*)

(* Expanded form after the intended substitutions (see below): *)

RawWhilePlusHalfFull(
        IeP(_, _), JeP(_, _), IsP(_, _),
        EnvNext, Next, SysNext, Le, Ls) ==
    \/ ~ \E p, q:  IeP(p, q) => JeP(p, q)
    \/ /\ \E p:  IsP(p, y)
       /\ \/ ~ \/ ~ IeP(x, y)
               \/ JeP(x, y)
          \/ /\ IsP(x, y)
             /\ IeP(x, y) \/ [](Next /\ SysNext)
             /\ \/ ~ IeP(x, y)
                \/ [](Earlier(EnvNext) => /\ Earlier(Next)
                                          /\ SysNext)
             /\ \/ ~ \/ ~ IeP(x, y)
                     \/ JeP(x, y) /\ Le /\ []EnvNext
                \/ Ls


(* This is the "shallow" case. *)
PROPOSITION
    ASSUME
        CONSTANT JeP(_, _), CONSTANT IsP(_, _),
        ACTION EnvNext, ACTION Next, ACTION SysNext,
        TEMPORAL Le, TEMPORAL Ls
    PROVE
        RawWhilePlusHalfFull(
            TRUE, JeP, IsP, EnvNext, Next, SysNext, Le, Ls)
        <=>
        \/ ~ \E p, q:  JeP(p, q)
        \/ /\ \E p:  IsP(p, y)
           /\ \/ ~  JeP(x, y)
              \/ /\ IsP(x, y)
                 /\ [](Earlier(EnvNext) => /\ Earlier(Next)
                                           /\ SysNext)
                 /\ (Le /\ []EnvNext) => Ls
    PROOF OBVIOUS

(*
If |= IeP(x, y), then the above becomes
    \/ ~ \E p, q:  JeP(p, q)
    \/ /\ \E p:  IsP(p, y)
       /\ \/ ~ JeP(x, y)
          \/ /\ IsP(x, y)
             /\ [](Earlier(EnvNext) => /\ Earlier(Next)
                                       /\ SysNext)
             /\ (LeP(x, y) /\ []EnvNext)  =>  LsP(x, y)
*)


SIH(EnvNext, Next, SysNext) ==
    [](Earlier(EnvNext)  =>  /\ Earlier(Next)
                             /\ SysNext)


PROPOSITION
    ASSUME ACTION EnvNext, ACTION Next, ACTION SysNext
    PROVE WeakStepwiseImpl(EnvNext, SysNext)
        <=> SIH(EnvNext, SysNext, TRUE)
    PROOF OBVIOUS


PROPOSITION
    ASSUME ACTION EnvNext, ACTION Next, ACTION SysNext
    PROVE StepwiseImpl(EnvNext, SysNext)
        <=> SIH(EnvNext, TRUE, SysNext)
        <=> SIH(EnvNext, SysNext, SysNext)
    PROOF OBVIOUS



(* Stepwise form of WhilePlusHalf. *)
THEOREM WhilePlusHalfStepwiseForm ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma, IsABehavior(sigma),
        CONSTANT IeP(_, _),
        CONSTANT JeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL LeP, TEMPORAL LsP,
        (* The constants are predicates.
        TEMPORAL level expressions can only be formulas,
        so LeP and LsP are certainly Boolean-valued.
        *)
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  JeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            xy == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Je == JeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            Le == LeP(x, y)
            Ls == LsP(x, y)

            A(u, v) ==
                LET
                    I == IeP(u, v)
                    J == JeP(u, v)
                    N == NeP(u, v, u', v')
                    vrs = << u, v >>
                    L == LeP(u, v)
                IN
                    I => (J /\ [][N]_vrs /\ L)

            Q(u, v) ==
                \/ ~ IeP(u, v)
                \/ /\ JeP(u, v)
                   /\ [][NeP(u, v, u', v')]_<< u, v >>

            R(u, v) == /\ IsP(u, v)
                       /\ [][NsP(u, v, u', v')]_<< u, v >>
        IN
            (* for our intended usage, the term IeP => LeP never arises;
            either IeP is TRUE, or LeP is TRUE. So this assumption
            reduces to either:
                Cl(I => (J /\ [][N]_vrs))  <=>  (I => (J /\ [][N]_vrs))
            or
                Cl(J /\ [][N]_vrs /\ L)  <=>  (J /\ [][N]_vrs)

            The first case is easy to prove, because it is a safety
            property (alternatively, we can invoke the safety-liveness
            decomposition of WhilePlusHalf.

            The second case is a typical machine-closure condition.
            *)
            /\ \AA u, v:  Cl(A, u, v) <=> Q(u, v)
            /\ \AA u, v:  IsMachineClosed(R, LsP, u, v)
    PROVE
        LET
            A(u, v) ==
                LET
                    I == IeP(u, v)
                    J == JeP(u, v)
                    N == NeP(u, v, u', v')
                    vrs = << u, v >>
                    L == LeP(u, v)
                IN
                    I => (J /\ [][N]_vrs /\ L)
            G(u, v) ==
                LET
                    I == IsP(u, v)
                    N == NsP(u, v, u', v')
                    vrs == << u, v >>
                    L == LsP(u, v)
                IN
                    I /\ [][N]_vrs /\ L
            Phi == WhilePlusHalf(A, G, x, y)

            xy == << x, y >>
            Ie == IeP(x, y)
            Is == IsP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            Le == LeP(x, y)
            Ls == LsP(x, y)

            EnvNext == [Ne]_<< x, y >>
            Next == [Ns]_<< x, y >>
            SysNext == [ \E r:  NsP(x, y, r, y') ]_y

            RawPhi == RawWhilePlusHalfFull(
                IeP, JeP, IsP, EnvNext, SysNext, Le, Ls)
        IN
            (sigma, 0 |= RawPhi)  <=>  (sigma |= Phi)
    PROOF
    <1> DEFINE
        Is == IsP(x, y)
        Ie == IeP(x, y)
        Je == JeP(x, y)
        Ne == NeP(x, y, x', y')
        Ns == NsP(x, y, x', y')
        Le == LeP(x, y)
        Ls == LsP(x, y)

        A(u, v) ==
            LET
                I == IeP(u, v)
                J == JeP(u, v)
                N == NeP(u, v, u', v')
                vrs = << u, v >>
                L == LeP(u, v)
            IN
                I => (J /\ [][N]_vrs /\ L)
        G(u, v) ==
            LET
                I == IsP(u, v)
                N == NsP(u, v, u', v')
                vrs == << u, v >>
                L == LsP(u, v)
            IN
                I /\ [][N]_vrs /\ L
        ClA(u, v) == Cl(A, u, v)
        ClG(u, v) == Cl(G, u, v)

        Fr(P(_, _), b) == Front(P, x, y, b)
        FPH(P(_, _), b) == FrontPlusHalf(P, x, y, b)

        Q(u, v) == IeP(u, v) => /\ JeP(u, v)
                                /\ [][NeP(u, v, u', v')]_<< u, v >>
        R(u, v) == IsP(u, v) /\ [][NsP(u, v, u', v')]_<< u, v >>

        EnvNext == [Ne]_<< x, y >>
        Next == [Ns]_<< x, y >>
        SysNext == [\E r:  NsP(x, y, r, y')]_y

    (* Not using bound variables via \AA u, v here would be a mistake.
    To understand why, consider the operator Foo(u) == x = u
    and what the assertion Foo(x) tells us (nothing).
    *)
    <1>5. \AA u, v:  ClA(u, v) <=> Q(u, v)
        <2>1. \AA u, v:  ClA(u, v) <=> Cl(A, u, v)
            BY DEF ClA
        <2>2. \AA u, v:  Cl(A, u, v) <=>
                \/ ~ IeP(u, v)
                \/ /\ JeP(u, v)
                   /\ [][NeP(u, v, u', v')]_<< u, v >>
            BY DEF A
                (* and WhilePlusHalfStepwiseForm!assumption *)
        <2>3. \AA u, v:  Cl(A, u, v) <=> Q(u, v)
            BY <2>2 DEF Q
        <2> QED
            BY <2>1, <2>3

    <1>6. \AA u, v:  ClG(u, v) <=> R(u, v)
        <2>1. \AA u, v:  ClG(u, v) <=> Cl(G, u, v)
            BY DEF ClG
        <2>2. ASSUME VARIABLE u, VARIABLE v
              PROVE R(u, v) <=> Cl(G, u, v)
            <3>1. LET F(u, v) == R(u, v) /\ LsP(u, v)
                  IN R(u, v) <=> Cl(F, u, v)
                BY DEF R, IsMachineClosed
                    (* and WhilePlusHalfStepwiseForm!assumption *)
            <3>2. G(u, v)  <=>  (R(u, v) /\ LsP(u, v))
                BY DEF G, R
            <3> QED
                BY <3>1, <3>2
        <2> QED
            BY <2>1, <2>2

    <1>7. ASSUME VARIABLE b
          PROVE Fr(ClA, b) <=> Fr(Q, b)
        <2>1. Fr(ClA, b) <=> \EE u, v:
                /\ ClA(u, v)
                /\ SamePrefix(b, u, v, x, y)
            BY DEF Fr, Front, ClA
        <2>2. Fr(ClA, b) <=> \EE u, v:
                /\ Q(u, v)
                /\ SamePrefix(b, u, v, x, y)
            BY <2>1, <1>5
        <2> QED
            BY <2>2 DEF Front, Fr
    <1>8. ASSUME VARIABLE b
          PROVE FPH(ClG, b) <=> FPH(R, b)
        <2>1. FPH(ClG, b) <=> \EE u, v:
                /\ ClG(u, v)
                /\ SamePrefix(b, u, v, x, y)
                /\ PlusHalf(b, v, y)
            BY DEF FPH, FrontPlusHalf
        <2>2. FPH(ClG, b) <=> \EE u, v:
                /\ R(u, v)
                /\ SamePrefix(b, u, v, x, y)
                /\ PlusHalf(b, v, y)
            BY <2>1, <1>6
        <2> QED
            BY <2>2 DEF FrontPlusHalf, FPH

    <1>1. WPH(A, G, x, y) <=>
            (* liveness part *)
            /\ A(x, y) => G(x, y)
            (* initial condition *)
            /\ \AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
            (* stepwise implication *)
            /\ \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)

        (* This expansion combines the theorem
        WhilePlusHalfAsConj
        with reversal of some of its final steps.
        *)
        <2>1. WPH(A, G, x, y) <=>
                /\ WPH(ClA, ClG, x, y)
                /\ A(x, y) => G(x, y)
            BY WhilePlusHalfAsConj

        <2>2. WPH(A, G, x, y) <=>
                /\ \AA b:  (Fr(ClA, b) /\ [](b = TRUE)) => FPH(ClG, b)
                /\ \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b)
                /\ \AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)
                /\ A(x, y) => G(x, y)
            OMITTED

        <2>3. \/ ~ (A(x, y) => G(x, y))
              \/ \AA b:  (Fr(ClA, b) /\ [](b = TRUE)) => FPH(ClG, b)
            OMITTED
        <2> QED
            BY <2>2, <2>3

    (* The liveness part. *)
    <1>2. (A(x, y) => G(x, y))
            <=> \/ ~ \/ ~ IeP(x, y)
                     \/ /\ JeP(x, y)
                        /\ [][NeP(x, y, x', y')]_<< x, y >>
                        /\ LeP(x, y)
                \/ /\ IsP(x, y)
                   /\ [][NsP(x, y, x', y')]_<< x, y >>
                   /\ LsP(x, y)
        (* This assertion is expressed in TLA+. Any TLA+ formula is also
        a formula of raw TLA+ with past, so we can transfer this equivalence
        to the raw logic. The same observation applies to the assertion of
        step <1>6 below.
        *)
        BY DEF A, G

    (* The initial condition. *)
    <1>3. (\AA b:  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b))
            <=> \/ ~ (\E p, q:  IeP(p, q) => JeP(p, q))
                \/ \E p:  Is(p, y)

        <2>1. ASSUME VARIABLE b
              PROVE (  (Fr(ClA, b) /\ [](b = FALSE)) => FPH(ClG, b)  )
                    <=> (  (Fr(Q, b) /\ [](b = FALSE)) => FPH(R, b)  )
            BY <1>6, <1>7
        <2>2. ASSUME VARIABLE b
              PROVE (  (Fr(Q, b) /\ [](b = FALSE)) => FPH(R, b)  )
                    <=> \/ ~ (Fr(Q, b) /\ [](b = FALSE))
                        \/ FPH(R, b) /\ [](b = FALSE)
            OBVIOUS
        <2>3. ASSUME VARIABLE b
              PROVE \/ ~ [](b = FALSE)
                    \/ Fr(Q, b)  <=>  \E p, q:  IeP(p, q) => JeP(p, q)
            <3>1. \/ ~ [](b = FALSE)
                  \/ Fr(Q, b) <=> \EE u, v:  /\ Q(u, v)
                                             /\ SamePrefix(b, u, v, x, y)
                BY DEF Fr, Front
            <3>2. \/ ~ [](b = FALSE)
                  \/ Fr(Q, b) <=> \EE u, v:  Q(u, v)
                <4>1. \/ ~ [](b = FALSE)
                      \/ SamePrefix(b, u, v, x, y)
                    BY DEF SamePrefix
                <4> QED
                    BY <3>1, <4>1
            <3>3. (\EE u, v:  Q(u, v))
                    <=> \E p, q:  IeP(p, q) => JeP(p, q)
                <4>1. (\EE u, v:  Q(u, v))
                        <=> \EE u, v:  \/ ~ IeP(u, v)
                                       \/ /\ JeP(u, v)
                                          /\ [][NeP(u, v, u', v')]_<< u, v >>
                    BY DEF Q
                <4>2. (\EE u, v:  Q(u, v))
                        <=> \/ \EE u, v:  ~ IeP(u, v)
                            \/ \EE u, v:  /\ JeP(u, v)
                                          /\ [][NeP(u, v, u', v')]_<< u, v >>
                    BY <4>1
                <4>3. (\EE u, v:  Q(u, v))
                        <=> \/ \E p, q:  ~ IeP(p, q)
                            \/ \E p, q:  JeP(p, q)
                    BY <4>2  (* just stutter forever the initial state *)
                <4> QED
                    BY <4>3
            <3> QED
                BY <3>2, <3>3
        <2>4. ASSUME VARIABLE b
              PROVE \/ ~ [](b = FALSE)
                    \/ FPH(R, b)  <=>  \E p:  IsP(p, y)
            <3>1. \/ ~ [](b = FALSE)
                  \/ FPH(R, b) <=> \EE u, v:
                        /\ R(u, v)
                        /\ SamePrefix(b, u, v, x, y)
                        /\ PlusHalf(b, v, y)
                BY DEF FPH, FrontPlusHalf
            <3>2. \/ ~ [](b = FALSE)
                  \/ FPH(R, b) <=> \EE u, v:  R(u, v) /\ (v = y)
                <4>1. \/ ~ [](b = FALSE)
                      \/ SamePrefix(b, u, v, x, y)
                    BY DEF SamePrefix
                <4>2. \/ ~ [](b = FALSE)
                      \/ PlusHalf(b, v, y) <=> (v = y)
                    BY DEF PlusHalf
                <4> QED
                    BY <3>1, <4>1, <4>2
            <3>3. (\EE u, v:  R(u, v) /\ (v = y))
                    <=> \E p:  IsP(p, y)
                <4>1. (\EE u, v:  R(u, v) /\ (v = y))
                        <=> \EE u, v:  /\ IsP(u, v) /\ (v = y)
                                       /\ [][NsP(u, v, u', v')]_<< u, v >>
                    BY DEF R
                <4>2. (\EE u, v:  R(u, v) /\ (v = y))
                        <=> \EE u, v:  IsP(u, v) /\ (v = y)
                    BY <4>1  (* just stutter forever the initial state *)
                <4>3. (\EE u, v:  IsP(u, v) /\ (v = y))
                        <=> \E p:  IsP(p, y)
                    <5>1. (\EE u, v:  IsP(u, v) /\ (v = y))
                            <=> \EE u, v:  IsP(u, y) /\ (v = y)
                        OBVIOUS
                    <5>2. (\EE u, v:  IsP(u, y) /\ (v = y))
                            <=> \EE u:  IsP(u, y) /\ \EE v:  v = y
                        OBVIOUS
                    <5>3. (\EE u:  IsP(u, y) /\ \EE v:  v = y)
                            <=> \EE u:  IsP(u, y)
                        OBVIOUS
                    <5>4. (\EE u:  IsP(u, y))  <=>  \E p:  IsP(p, y)
                        OBVIOUS
                    <5> QED
                        BY <5>1, <5>2, <5>3, <5>4
                <4> QED
                    BY <4>2, <4>3
            <3> QED
                BY <3>2, <3>3
        <2> QED
            BY <2>1, <2>2, <2>3, <2>4

    (* The stepwise implication (part of safety). *)
    <1>4. (sigma |= \AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b))
            <=> sigma, 0 |=  (* at this point we have to use past raw TLA+
                    to accommodate for the operator Earlier. *)
                \/ ~ \/ ~ Ie
                     \/ Je
                \/ /\ Is
                   /\ Ie \/ [](Next /\ SysNext)
                   /\ Ie => [](Earlier(EnvNext) => /\ Earlier(Next)
                                                   /\ SysNext)
        <2>7. ASSUME VARIABLE u, VARIABLE v
              PROVE
                  \/ ~ /\ b = TRUE
                       /\ [][b' = FALSE]_b
                  \/ [][b' => b]_<< u, v, x, y >>

            <3>1. \/ ~ /\ b \in BOOLEAN
                       /\ [b' = FALSE]_b
                  \/ b' => b
                OBVIOUS
            <3>2. \/ ~ /\ b = TRUE
                       /\ [][b' = FALSE]_b
                  \/ [](b \in BOOLEAN)
                OBVIOUS
            <3> QED
                BY <3>1, <3>2
        <2>1. (\AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b))
                <=> \AA b:  \/ ~ MustUnstep(b)
                            \/ Fr(ClA, b) => FPH(ClG, b)
            OBVIOUS
        <2>2. (\AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b))
                <=> \AA b:  \/ ~ MustUnstep(b)
                            \/ Fr(Q, b) => FPH(R, b)
            BY <1>7, <1>8
        <2>3. ASSUME VARIABLE b
              PROVE \/ ~ MustUnstep(b)
                    \/ Fr(Q, b) <=>
                        \/ ~ IeP(x, y)
                        \/ /\ JeP(x, y)
                           /\ [][b' => NeP(x, y, x', y')]_<< x, y >>
            <3> USE DEF Ie, Je, Ne
            <3>1. Fr(Q, b) <=> \EE u, v:
                    /\ Q(u, v)
                    /\ SamePrefix(b, u, v, x, y)
                BY DEF Fr, Front
            <3>2. \/ ~ MustUnstep(b)
                  \/ (\EE u, v:  /\ Q(u, v)
                                 /\ SamePrefix(b, u, v, x, y))
                      <=>
                        /\ Ie => /\ Je
                                 /\ [][b' => Ne]_<< x, y >>

                <4>1. \/ ~ MustUnstep(b)
                      \/ ~ \EE u, v:  /\ Q(u, v)
                                      /\ SamePrefix(b, u, v, x, y)
                      \/ Ie => /\ Je
                               /\ [][b' => Ne]_<< x, y >>
                    <5> DEFINE
                        F == /\ MustUnstep(b)
                             /\ \EE u, v:  /\ Q(u, v)
                                           /\ SamePrefix(b, u, v, x, y)
                    <5>1. \/ ~ F
                          \/ \EE u, v:
                                /\ Q(u, v)
                                /\ SamePrefix(b, u, v, x, y)
                                /\ MustUnstep(b)
                        BY DEF F
                    <5>2. \/ ~ F
                          \/ \EE u, v:
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][NeP(u, v, u', v')]_<< u, v >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ b = TRUE
                                /\ [][b' = FALSE]_b
                        BY <5>1 DEF Q, SamePrefix, MustUnstep
                    <5>6. \/ ~ F
                          \/ \EE u, v:
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        [NeP(u, v, u', v')]_<< u, v >>
                                        ]_<< u, v, x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ b = TRUE
                                /\ [][b' => b]_<< u, v, x, y >>
                        BY <5>2, <2>7
                    <5>3. \/ ~ F
                          \/ \EE u, v:
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        /\ [NeP(u, v, u', v')]_<< u, v >>
                                        /\ b => (<< u, v >> = << x, y >>)
                                        /\ b' => (<< u', v' >> = << x', y' >>)
                                        ]_<< u, v, x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ b => (<< u, v >> = << x, y >>)
                                /\ b = TRUE
                                /\ [][b' => b]_<< u, v, x, y >>
                        BY <5>6, RuleINV2
                    <5>4. \/ ~ F
                          \/ \EE u, v:
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        /\ \/ ~ (b' /\ b)
                                           \/ [NeP(x, y, x', y')]_<< x, y >>
                                        /\ b => (<< u, v >> = << x, y >>)
                                        /\ b' => (<< u', v' >> = << x', y' >>)
                                        /\ b' => b
                                        ]_<< u, v, x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ << u, v >> = << x, y >>
                        BY <5>3
                    <5>7. \/ ~ F
                          \/ \EE u, v:
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        b' => [NeP(x, y, x', y')]_<< x, y >>
                                        ]_<< u, v, x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ << u, v >> = << x, y >>
                        BY <5>4
                    <5>5. \/ ~ F
                          \/ \EE u, v:
                                \/ ~ IeP(x, y)
                                \/ /\ JeP(x, y)
                                   /\ [][b' => NeP(x, y, x', y')]_<< x, y >>
                        BY <5>4
                    <5> QED
                        <6>1. (\EE u, v:
                                  IeP(x, y)
                                  => /\ JeP(x, y)
                                     /\ [][b' => NeP(x, y, x', y')]_<< x, y >>)
                            <=>
                                Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>
                            BY DEF Ie, Je, Ne
                        <6> QED
                            BY <5>5, <6>1 DEF F
                <4>2. \/ ~ MustUnstep(b)
                      \/ ~ Ie => /\ Je
                                 /\ [][b' => Ne]_<< x, y >>
                      \/ \EE u, v:  /\ Q(u, v)
                                    /\ SamePrefix(b, u, v, x, y)
                    <5> DEFINE
                        H == /\ MustUnstep(b)
                             /\ Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>
                    <5>1. \EE u, v:
                           /\ [](b => (<< u, v >> = << x, y >>))
                           /\ [][b']_<< u, v >>
                        OBVIOUS  (* stutter u, v after b falls *)
                    <5>2. ASSUME VARIABLE u, VARIABLE v,
                            b' \in BOOLEAN /\ [b']_<< u, v >>
                          PROVE [(~ b') => NeP(u, v, u', v')]_<< u, v >>
                        <6>1. SUFFICES ASSUME ~ b' /\ ~ UNCHANGED << u, v >>
                                       PROVE FALSE
                            OBVIOUS
                        <6>2. b'
                            <7>1. ~ UNCHANGED << u, v >>
                                BY <6>1
                            <7> QED
                                BY <7>1, <5>2
                        <6> QED
                            BY <6>1, <6>2  (* goal from <6>1 *)
                    <5>3. \/ ~ MustUnstep(b)
                          \/ \EE u, v:
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][b']_<< u, v >>
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                        <6>1. MustUnstep(b) => [](b \in BOOLEAN)
                            BY DEF MustUnstep
                        <6> QED
                            BY <6>1, <5>1, <5>2
                    <5>4. \/ ~ H
                          \/ /\ [][b' => b]_<< u, v, x, y >>
                             /\ Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>
                        BY <2>7 DEF H, MustUnstep
                    <5>5. \/ ~ H
                          \/ /\ [][b' => b]_<< u, v, x, y >>
                             /\ Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>
                             /\ \EE u, v:
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][b']_<< u, v >>
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                        BY <5>4 DEF H
                    <5>6. \/ ~ H
                          \/ /\ b = TRUE
                             /\ [][b' => b]_<< u, v, x, y >>
                             /\ Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>
                                      /\ \EE u, v:
                                /\ b => (<< u, v >> = << x, y >>)
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][b']_<< u, v >>
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                                /\ \/ ~ IeP(x, y)
                                   \/ /\ JeP(x, y)
                                      /\ [][b' => Ne]_<< x, y >>
                        BY <5>5 DEF H, MustUnstep, Ie, Je, Ne
                    <5>7. \/ ~ H
                          \/ /\ [][b' => b]_<< u, v, x, y >>
                             /\ Ie => /\ Je
                                      /\ [][b' => Ne]_<< x, y >>)
                             /\ \EE u, v:
                                /\ << u, v >> = << x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][b']_<< u, v >>
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                                /\ \/ ~ IeP(x, y)
                                   \/ /\ JeP(x, y)
                                      /\ [][
                                        [b' => NeP(x, y, x', y')]_<< x, y >>
                                        ]_<< x, y, u, v >>
                        BY <5>6
                    <5>8. \/ ~ H
                          \/ \EE u, v:
                                /\ << u, v >> = << x, y >>
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][b']_<< u, v >>
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        /\ [b' => NeP(x, y, x', y')]_<< x, y >>
                                        /\ b' => b
                                        /\ b => (<< u, v >> = << x, y >>)
                                        /\ b' => (<< u', v' >> = << x', y' >>)
                                        ]_<< x, y, u, v >>
                        BY <5>7
                    <5>9. \/ ~ H
                          \/ \EE u, v:
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ JeP(u, v)
                                      /\ [][
                                        /\ [b' => NeP(x, y, x', y')]_<< x, y >>
                                        /\ b' => (<< u, v >> = << x, y >>)
                                        /\ b' => (<< u', v' >> = << x', y' >>)
                                        ]_<< x, y, u, v >>
                        BY <5>8
                    <5>10. \/ ~ H
                           \/ \EE u, v:
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ [][(~ b') => NeP(u, v, u', v')]_<< u, v >>
                                /\ \/ ~ IeP(u, v)
                                   \/ /\ Je(u, v)
                                      /\ [][b' => NeP(u, v, u', v')]_<< u, v >>
                        BY <5>9
                    <5>11. \/ ~ H
                           \/ \EE u, v:
                              /\ [](b => (<< u, v >> = << x, y >>))
                              /\ \/ ~ IeP(u, v)
                              /\ \/ /\ JeP(u, v)
                                    /\ [][(~ b') =>
                                            NeP(u, v, u', v')]_<< u, v >>
                                    /\ [][b' => NeP(u, v, u', v')]_<< u, v >>
                        BY <5>10
                    <5>12. \/ ~ H
                           \/ \EE u, v:
                                /\ [](b => (<< u, v >> = << x, y >>))
                                /\ IeP(u, v) =>
                                   /\ JeP(u, v)
                                   /\ [][NeP(u, v, u', v')]_<< u, v >>
                        <6>1. H => [](b \in BOOLEAN)
                            BY DEF H, MustUnstep
                        <6> QED
                            BY <5>11, <6>1
                    <5>13. \/ ~ H
                           \/ \EE u, v:
                                /\ Q(u, v)
                                /\ SamePrefix(b, u, v, x, y)
                        BY <5>12 DEF Q, SamePrefix
                    <5> QED
                        BY <5>13 DEF H
                <4> QED
                    BY <4>1, <4>2
            <3> QED  (* TODO: turn into a SUFFICES to reduce indentation *)
                BY <3>1, <3>2 DEF Ie, Je, Ne

        <2>4. ASSUME VARIABLE b
              PROVE \/ ~ MustUnstep(b)
                    \/ FPH(R, b) <=> /\ IsP(x, y)
                                     /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                                     /\ [][b => \E r:  NsP(x, y, r, y')]_y
            <3> USE DEF Is, Ns
            (* In this direction, we derive a quantified formula that
            is independent of the bound variables u and v. This allows us to
            eliminate the temporal quantifier \EE.
            *)
            <3>1. FPH(R, b) <=> \EE u, v:
                    /\ R(u, v)
                    /\ SamePrefix(b, u, v, x, y)
                    /\ PlusHalf(b, v, y)
                BY DEF FPH, FrontPlusHalf
            <3>2. SUFFICES
                    \/ ~ MustUnstep(b)
                    \/ (\EE u, v:  /\ R(u, v)
                                   /\ SamePrefix(b, u, v, x, y)
                                   /\ PlusHalf(b, v, y))
                        <=> /\ Is
                            /\ [][b' => Ns]_<< x, y >>
                            /\ [][b => \E r:  NsP(x, y, r, y')]_y
                BY <3>1, <3>2
            <3>3. \/ ~ MustUnstep(b)
                  \/ ~ \EE u, v:  /\ R(u, v)
                                  /\ SamePrefix(b, u, v, x, y)
                                  /\ PlusHalf(b, v, y)
                  \/ /\ Is
                     /\ [][b' => Ns]_<< x, y >>
                     /\ [][b => \E r:  NsP(x, y, r, y')]_y
                <4> DEFINE
                    F == /\ MustUnstep(b)
                         /\ \EE u, v:  /\ R(u, v)
                                       /\ SamePrefix(b, u, v, x, y)
                                       /\ PlusHalf(b, v, y)
                <4>1. \/ ~ F
                      \/ \EE u, v:  /\ R(u, v)
                                    /\ SamePrefix(b, u, v, x, y)
                                    /\ PlusHalf(b, v, y)
                    BY DEF F
                <4>2. \/ ~ F
                      \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [][NsP(u, v, u', v')]_<< u, v >>
                            /\ [](b => (<< u, v >> = << x, y >>))
                            /\ v = y
                            /\ [][b => (v' = y')]_<< b, v, y >>
                    BY <4>1 DEF R, SamePrefix, PlusHalf
                <4>3. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(u, v)
                          /\ [][NsP(u, v, u', v')]_<< u, v >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ v = y
                          /\ [][b => (v' = y')]_<< b, v, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>2 DEF F, MustUnstep
                <4>4. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(u, v)
                          /\ [][NsP(u, v, u', v')]_<< u, v >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ b => (<< u, v >> = << x, y >>
                          /\ v = y
                          /\ [][b => (v' = y')]_<< b, v, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>3
                <4>5. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(u, v)
                          /\ [][NsP(u, v, u', v')]_<< u, v >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ << u, v >> = << x, y >>
                          /\ [][b => (v' = y')]_<< b, v, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>4
                <4>6. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][NsP(u, v, u', v')]_<< u, v >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ [][b => (v' = y')]_<< b, v, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>5
                <4>7. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][NsP(u, v, u', v')]_<< u, v >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ [][b => (v' = y')]_<< b, v, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>6
                <4>8. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][
                            [NsP(u, v, u', v')]_<< u, v >>
                            ]_<< u, v, x, y >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ [][
                            [b => (v' = y')]_<< b, v, y >>
                            ]_<< u, v, x, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>7
                <4>9. \/ ~ F
                      \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][
                            /\ b => (<< u, v >> = << x, y >>)
                            /\ b' => (<< u', v' >> = << x', y' >>)
                            /\ [NsP(u, v, u', v')]_<< u, v >>
                            ]_<< u, v, x, y >>
                          /\ [][
                            /\ b => (<< u, v >> = << x, y >>)
                            /\ [b => (v' = y')]_<< b, v, y >>
                            /\ [NsP(u, v, u', v')]_<< u, v >>
                            ]_<< u, v, x, y >>
                          /\ [](b => (<< u, v >> = << x, y >>))
                          /\ [][
                            [b => (v' = y')]_<< b, v, y >>
                            ]_<< u, v, x, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>8
                <4>10. \/ ~ F
                       \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][
                            (b /\ b') => [NsP(x, y, x', y')]_<< x, y >>
                            ]_<< u, v, x, y >>
                          /\ [][
                            /\ b => (<< u, v >> = << x, y >>)
                            /\ [b => (v' = y')]_<< b, v, y >>
                            /\ [NsP(u, v, u', v')]_v
                            ]_<< u, v, x, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    BY <4>9
                <4>11. \/ ~ F
                       \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][
                            /\ b' => b
                            /\ (b /\ b') => [NsP(x, y, x', y')]_<< x, y >>
                            ]_<< u, v, x, y >>
                          /\ [][
                            /\ b => (<< u, v >> = << x, y >>)
                            /\ [b => (v' = y')]_<< b, v, y >>
                            /\ b => [NsP(x, y, u', y')]_y
                            ]_<< u, v, x, y >>
                          /\ b = TRUE
                          /\ [][b' = FALSE]_b
                    <5>1. ASSUME VARIABLE u, VARIABLE v
                            /\ b => (<< u, v >> = << x, y >>)
                            /\ [b => (v' = y')]_<< b, v, y >>
                            /\ [NsP(u, v, u', v')]_v
                          PROVE b => [NsP(x, y, u', y')]_y
                        <6>1. SUFFICES ASSUME b /\ ~ UNCHANGED y
                                       PROVE NsP(x, y, u', y')
                            OBVIOUS
                        <6>2. (u = x) /\ (v = y)
                            <7>1. b
                                BY <6>1
                            <7>2. << u, v >> = << x, y >>
                                BY <5>1, <7>1
                            <7> QED
                                BY <7>2
                        <6>3. v' = y'
                            <7>1. b
                                BY <6>1
                            <7>2. b => (v' = y')
                                <8>1. y' # y
                                    BY <6>1
                                <8> QED
                                    BY <5>1, <8>1
                            <7> QED
                                BY <7>1, <7>2
                        <6>4. v' # v
                            <7>1. y' # y
                                BY <6>1
                            <7>2. (v = y) /\ (v' = y')
                                BY <6>2, <6>3
                            <7> QED
                                BY <7>1, <7>2
                        <6>5. NsP(u, v, u', v')
                            BY <5>1, <6>4
                        <6> QED
                            <7>1. (u = x) /\ (v = y) /\ (v' = y')
                                BY <6>2, <6>3, <6>4
                            <7> QED
                                BY <6>5, <7>1  (* goal from <6>1 *)
                    <5> QED
                        BY <4>10, <2>7, <5>1
                <4>12. \/ ~ F
                       \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][
                            b' => [NsP(x, y, x', y')]_<< x, y >>
                            ]_<< u, v, x, y >>
                          /\ [][
                            b => [\E r:  NsP(x, y, r, y')]_y
                            ]_<< u, v, x, y >>
                    BY <4>11
                <4>13. \/ ~ F
                       \/ \EE u, v:
                          /\ IsP(x, y)
                          /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                          /\ [][b => \E r:  NsP(x, y, r, y')]_y
                    BY <4>12
                <4>14. \/ ~ F
                       \/ /\ IsP(x, y)
                          /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                          /\ [][b => \E r:  NsP(x, y, r, y')]_y
                    BY <4>13
                <4> QED
                    BY <4>14 DEF F, Is, Ns

            <3>4. \/ ~ /\ MustUnstep(b)
                       /\ Is
                       /\ [][b' => Ns]_<< x, y >>
                       /\ [][b => \E r:  NsP(x, y, r, y')]_y
                  \/ \EE u, v:  /\ R(u, v)
                                /\ SamePrefix(b, u, v, x, y)
                                /\ PlusHalf(b, v, y)

                <4> DEFINE
                    H == /\ MustUnstep(b)
                         /\ Is
                         /\ [][b' => Ns]_<< x, y >>
                         /\ [][b => \E r:  NsP(x, y, r, y')]_y
                <4>1. \EE u, v:
                        /\ [][b]_<< u, v >>  (* stuttering tail *)
                            (* same prefix *)
                        /\ [](b => (<< x, y >> = << u, v >>))
                        /\ [][ b => (v' = y') ]_<< b, v, y >>
                        /\ [][ (b /\ ~ b') =>  (* falling edge *)
                                /\ v' = y'
                                /\ u' = IF y' = y THEN u
                                    ELSE CHOOSE r:  NsP(x, y, r, y')
                            ]_<< b, v, y >>
                    OMITTED  (* TODO *)
                <4>2. ASSUME VARIABLE u, VARIABLE v,
                        b' \in BOOLEAN /\ [b]_<< u, v >>
                      PROVE [(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    OBVIOUS
                <4>3. \/ ~ MustUnstep(b)
                      \/ \EE u, v:
                            /\ [][b]_<< u, v >>
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                            /\ [][ (b /\ ~ b') =>
                                    /\ v' = y'
                                    /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                                ]_<< b, v, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>1, <4>2
                <4>4. \/ ~ H
                      \/ /\ b = TRUE
                         /\ [](b \in BOOLEAN)
                         /\ [][b' => b]_<< u, v, x, y >>
                         /\ Is
                         /\ [][b' => Ns]_<< x, y >>
                         /\ [][b => \E r:  NsP(x, y, r, y')]_y
                    BY <2>7 DEF H, MustUnstep
                <4>5. \/ ~ H
                      \/ /\ b = TRUE
                         /\ [](b \in BOOLEAN)
                         /\ [][b' => b]_<< u, v, x, y >>
                         /\ IsP(x, y)
                         /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                         /\ [][b => \E r:  NsP(x, y, r, y')]_y
                         /\ \EE u, v:
                            /\ [][b]_<< u, v >>
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                            /\ [][ (b /\ ~ b') =>
                                     /\ v' = y'
                                     /\ u' = IF y' = y THEN u
                                         ELSE CHOOSE r:  NsP(x, y, r, y')
                                ]_<< b, v, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>3, <4>4 DEF H, Is, NsP
                <4>6. \/ ~ H
                      \/ \EE u, v:
                            /\ b = TRUE
                            /\ b => << u, v >> = << x, y >>
                            /\ IsP(x, y)
                            /\ [](b \in BOOLEAN)
                            /\ [][b' => b]_<< u, v, x, y >>
                            /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                            /\ [][b => \E r:  NsP(x, y, r, y')]_y
                            /\ [][b]_<< u, v >>
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                            /\ [][ (b /\ ~ b') =>
                                     /\ v' = y'
                                     /\ u' = IF y' = y THEN u
                                         ELSE CHOOSE r:  NsP(x, y, r, y')
                                ]_<< b, v, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>5
                <4>7. \/ ~ H
                      \/ \EE u, v:
                            /\ << u, v >> = << x, y >>
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [][b' => b]_<< u, v, x, y >>
                            /\ [][
                                [b' => NsP(x, y, x', y')]_<< x, y >>
                                ]_<< u, v, x, y >>
                            /\ [][
                                [b => \E r:  NsP(x, y, r, y')]_y
                                ]_<< u, v, x, y >>
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                            /\ [][
                                [ (b /\ ~ b') =>
                                     /\ v' = y'
                                     /\ u' = IF y' = y THEN u
                                         ELSE CHOOSE r:  NsP(x, y, r, y')
                                 ]_<< b, v, y >>
                                ]_<< u, v, x, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>6
                <4>8. \/ ~ H
                      \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [][b' => b]_<< u, v, x, y >>
                            /\ [][
                                /\ b' => b
                                /\ b => (<< x, y >> = << u, v >>)
                                /\ b' => (<< x', y' >> = << u', v' >>)
                                /\ b' => [NsP(x, y, x', y')]_<< x, y >>
                                ]_<< u, v, x, y >>
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                            /\ [][
                                /\ b => (<< x, y >> = << u, v >>)
                                /\ [b => \E r:  NsP(x, y, r, y')]_y
                                /\ \/ ~ (b /\ ~ b')
                                   \/ /\ v' = y'
                                      /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                                ]_<< u, v, x, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    <5>1. ASSUME
                                /\ b \in BOOLEAN
                                /\ b' \in BOOLEAN
                                /\ b /\ ~ b'
                          PROVE
                                ~ UNCHANGED b
                        OBVIOUS
                    <5>2. ASSUME VARIABLE u, VARIABLE v,
                            /\ b \in BOOLEAN
                            /\ b' \in BOOLEAN
                          PROVE
                            \/ ~ \/ ~ (b /\ ~ b')
                                 \/ /\ v' = y'
                                    /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                                 \/ UNCHANGED << b, v, y >>
                            \/ \/ ~ (b /\ ~ b')
                               \/ /\ ~ UNCHANGED b
                                  /\ \/ /\ v' = y'
                                        /\ u' = IF y' = y THEN u
                                            ELSE CHOOSE r:  NsP(x, y, r, y')
                                     \/ UNCHANGED << b, v, y >>
                        BY <5>1
                    <5>3. ASSUME VARIABLE u, VARIABLE v,
                            /\ b \in BOOLEAN
                            /\ b' \in BOOLEAN
                          PROVE
                            \/ ~ \/ ~ (b /\ ~ b')
                                 \/ /\ v' = y'
                                    /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                                 \/ UNCHANGED << b, v, y >>
                            \/ \/ ~ (b /\ ~ b')
                               \/ /\ v' = y'
                                  /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                        BY <5>2
                    <5> QED
                        BY <4>7, <5>3
                <4>9. \/ ~ H
                      \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][
                                /\ (b' /\ b) => (<< x, y >> = << u, v >>)
                                /\ (b' /\ b) => (<< x', y' >> = << u', v' >>)
                                /\ (b' /\ b) => [NsP(u, v, u', v')]_<< u, v >>
                                ]_<< u, v, x, y >>
                            /\ [][
                                /\ b => (<< x, y >> = << u, v >>)
                                /\ [b => \E r:  NsP(x, y, r, y')]_y
                                /\ \/ ~ (b /\ ~ b')
                                   \/ /\ v' = y'
                                      /\ u' = IF y' = y THEN u
                                        ELSE CHOOSE r:  NsP(x, y, r, y')
                                 ]_<< u, v, x, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>8
                <4>10. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][
                                [(b' /\ b) => NsP(u, v, u', v')]_<< u, v >>
                                ]_<< u, v, x, y >>
                            /\ [][
                                \/ ~ (b /\ ~ b')
                                \/ /\ [\E r:  NsP(u, v, r, y')]_y
                                   /\ << x, y >> = << u, v >>
                                   /\ v' = y'
                                   /\ u' = IF v' = v THEN u
                                            ELSE CHOOSE r:  NsP(u, v, r, v')
                                 ]_<< u, v, x, y >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>9
                <4>11. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][(b /\ b') => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][
                                \/ ~ (b /\ ~ b')
                                \/ /\ [\E r:  NsP(u, v, r, v')]_v
                                   /\ << x, y >> = << u, v >>
                                   /\ v' = y'
                                   /\ u' = IF v' = v THEN u
                                            ELSE CHOOSE r:  NsP(u, v, r, v')
                                 ]_<< u, v, x, y >>
                    BY <4>10
                <4>12. ASSUME VARIABLE u, VARIABLE v,
                            /\ b \in BOOLEAN
                            /\ b' \in BOOLEAN
                            /\ ~ UNCHANGED << u, v >>
                            /\ [\E r:  NsP(u, v, r, v')]_v
                            /\ u' = IF v' = v THEN u
                                     ELSE CHOOSE r:  NsP(u, v, r, v')
                       PROVE
                            NsP(u, v, u', v')
                    <5>1. ASSUME UNCHANGED v
                          PROVE FALSE
                        <6>1. u' = IF v' = v THEN u
                                 ELSE CHOOSE r:  NsP(u, v, r, v')
                            BY <4>12
                        <6>2. u' = u
                            BY <5>1, <6>1
                        <6>3. UNCHANGED << u, v >>
                            BY <5>1, <6>2
                        <6>4. ~ UNCHANGED << u, v >>
                            BY <4>12
                        <6> QED
                            BY <6>3, <6>4
                    <5>2. CASE ~ UNCHANGED v
                        <6>1. \E r:  NsP(u, v, r, v')
                            BY <4>12, <5>2
                        <6>2. u' = CHOOSE r:  NsP(u, v, r, v')
                            <7>1. u' = IF v' = v THEN u
                                     ELSE CHOOSE r:  NsP(u, v, r, v')
                                BY <4>12
                            <7> QED
                                BY <7>1, <5>2
                        <6> QED
                            BY <6>1, <6>2
                <4>13. ASSUME VARIABLE u, VARIABLE v,
                            /\ b \in BOOLEAN
                            /\ b' \in BOOLEAN
                            /\ [\E r:  NsP(u, v, r, v')]_v
                            /\ u' = IF v' = v THEN u
                                    ELSE CHOOSE r:  NsP(u, v, r, v')
                       PROVE [NsP(u, v, u', v')]_<< u, v >>
                    BY <4>12
                <4>14. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][(b /\ b') => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][
                                \/ ~ (b /\ ~ b')
                                \/ /\ [\E r:  NsP(u, v, r, v')]_v
                                   /\ u' = IF v' = v THEN u
                                            ELSE CHOOSE r:  NsP(u, v, r, v')
                                   /\ [NsP(u, v, u', v')]_<< u, v >>
                                 ]_<< u, v, x, y >>
                    BY <4>11, <4>13
                <4>15. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][(b /\ b') => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][
                                (b /\ ~ b') => [NsP(u, v, u', v')]_<< u, v >>
                                 ]_<< u, v, x, y >>
                    BY <4>12
                <4>16. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][(b /\ b') => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][
                                [(b /\ ~ b') => NsP(u, v, u', v')]_<< u, v >>
                                 ]_<< u, v, x, y >>
                    BY <4>15
                <4>17. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>

                            /\ [][(~ b) => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(b /\ b') => NsP(u, v, u', v')]_<< u, v >>
                            /\ [][(b /\ ~ b') => NsP(u, v, u', v')]_<< u, v >>
                    BY <4>16
                <4>18. \/ ~ H
                       \/ \EE u, v:
                            /\ IsP(u, v)
                            /\ [][NsP(u, v, u', v')]_<< u, v >>
                            /\ [](b \in BOOLEAN)
                            /\ [](b => (<< x, y >> = << u, v >>))
                            /\ v = y
                            /\ [][ b => (v' = y') ]_<< b, v, y >>
                    BY <4>17
                <4>19. \/ ~ H
                       \/ \EE u, v:
                            /\ R(u, v)
                            /\ SamePrefix(b, u, v, x, y)
                            /\ PlusHalf(b, v, y)
                    BY <4>18 DEF R, SamePrefix, PlusHalf
                <4> QED
                    BY <4>19 DEF H
            <3> QED
                BY <3>3, <3>4  (* goal from <3>2 *)

        <2>5. (\AA b:  (Fr(ClA, b) /\ MustUnstep(b)) => FPH(ClG, b))
              <=> \AA b:
                \/ ~ MustUnstep(b)
                \/ \/ ~ \/ ~ IeP(x, y)
                        \/ /\ JeP(x, y)
                           /\ [][b' => NeP(x, y, x', y')]_<< x, y >>
                   \/ /\ IsP(x, y)
                      /\ [][b' => NsP(x, y, x', y')]_<< x, y >>
                      /\ [][b => \E r:  NsP(x, y, r, y')]_y
            BY <2>2, <2>3, <2>4

        (* At this point we transition to raw TLA+ with past operators. *)
        <2>6. ASSUME
                sigma |= \AA b:
                    \/ ~ MustUnstep(b)
                    \/ ~ \/ ~ Ie
                         \/ /\ Je
                            /\ [][b' => Ne]_<< x, y >>
                    \/ /\ Is
                       /\ [][b' => Ns]_<< x, y >>
                       /\ [][b => \E r:  NsP(x, y, r, y')]_y
              PROVE
                sigma, 0 |=
                    \/ ~ \/ ~ Ie
                         \/ Je
                    \/ /\ Is
                       /\ Ie \/ [](Next /\ SysNext)
                       /\ Ie => [](Earlier(EnvNext) => /\ Earlier(Next)
                                                       /\ SysNext)
            <3> USE DEF Ie, Je, Is, Ns, Ne
            <3>1. sigma, 0 |= \AA b:
                    \/ ~ MustUnstep(b)
                    \/ ~ \/ ~ Ie
                         \/ /\ Je
                            /\ [][b' => Ne]_<< x, y >>
                    \/ /\ Is
                       /\ [][b' => Ns]_<< x, y >>
                       /\ [][b => \E r:  NsP(x, y, r, y')]_y
                BY <2>6
            <3>2. SUFFICES
                    ASSUME sigma, 0 |= Ie => Je
                    PROVE
                        /\ sigma, 0 |= Is
                        /\ \/ sigma, 0 |= Ie
                           \/ sigma, 0 |= [](Next /\ SysNext)

                        /\ \/ ~ sigma, 0 |= Ie
                           \/ \A i \in Nat:
                                sigma, i |=
                                    Earlier(EnvNext) => /\ Earlier(Next)
                                                        /\ SysNext
                OBVIOUS
            <3>3. \A tau:
                    \/ ~ IsABehavior(tau)
                    \/ ~ RefinesUpToVar(tau, sigma, "b")
                    \/ tau, 0 |=  (* with the declaration VARIABLE b *)
                        \/ ~ MustUnstep(b)
                        \/ ~ \/ ~ Ie
                             \/ /\ Je
                                /\ [][b' => Ne]_<< x, y >>
                        \/ /\ Is
                           /\ [][b' => Ns]_<< x, y >>
                           /\ [][b => \E r:  NsP(x, y, r, y')]_y
                BY <3>2 DEF \AA  (* in raw TLA+ *)

            (* The following steps (<3>4, <3>5, <3>6) use sampling
            sequences, defined as tau, in order to draw the target
            conclusions. *)

            <3>4. sigma, 0 |= Is
                <4> DEFINE tau ==
                        LET state(n) == [sigma[n] EXCEPT !["b"] = (n = 0)]
                        IN [n \in Nat |-> state(n)]
                <4>1. /\ IsABehavior(tau)
                      /\ RefinesUpToVar(tau, sigma, "b")
                    BY DEF tau, IsABehavior, RefinesUpToVar,
                        Sim, Natural, EqualUpToVar
                <4>2. tau, 0 |= MustUnstep(b)
                    BY DEF tau, MustUnstep, Unstep, MayUnstep
                <4>3. tau, 0 |= [][b' => Ns]_<< x, y >>
                    BY DEF tau
                <4>4. tau, 0 |= (Ie => Je) => Is
                    BY <3>3, <4>1, <4>2, <4>3
                <4>5. sigma, 0 |= (Ie => Je) => Is
                    BY <4>4 DEF tau  (* IeP, JeP, IsP are CONSTANTS *)
                <4> QED
                    BY <4>5, <3>2 DEF tau

            <3>6. \/ sigma, 0 |= Ie
                  \/ sigma, 0 |= [](Next /\ SysNext)
                <4>1. SUFFICES
                        ASSUME NEW i \in Nat, sigma, 0 |= ~ Ie
                        PROVE sigma, i |= Next /\ SysNext
                    OBVIOUS
                <4>2. Next => SysNext
                    BY DEF Next, SysNext
                <4>3. SUFFICES sigma, i |= Next
                    BY <4>2  (* goal from <4>1 *)
                <4>4. DEFINE tau ==
                        LET state(n) == [
                            sigma[n] EXCEPT !["b"] = (n <= i + 2)]
                        IN [n \in Nat |-> state(n)]
                <4>5. /\ IsABehavior(tau)
                      /\ RefinesUpToVar(tau, sigma, "b")
                    BY DEF tau, IsABehavior, RefinesUpToVar,
                        Sim, Natura, EqualUpToVar
                <4>6. tau, 0 |= MustUnstep(b)
                    BY DEF tau, MustUnstep, Unstep, MayUnstep
                <4>7. tau, 0 |= ~ Ie
                    BY <4>1 DEF tau, Ie  (* IeP is independent of b. *)
                <4>8. tau, 0 |= [][b' => Ns]_<< x, y >>
                    BY <4>5, <4>6, <4>7, <3>3
                <4>9. tau, (i + 1) |= b
                    BY DEF tau
                <4>10. tau, i |= b'
                    BY <4>9
                <4>11. tau, i |= b' /\ [b' => Ns]_<< x, y >>
                    BY <4>10, <4>8
                <4>12. tau, i |= [Ns]_<< x, y >>
                    BY <4>11
                <4>13. tau, i |= Next
                    BY <4>12 DEF Next
                <4> QED
                    BY <4>13 DEF tau, Next  (* Next is independent of b. *)

            <3>5. ASSUME
                        NEW i \in Nat,
                        sigma, 0 |= Ie
                  PROVE
                        sigma, i |= Earlier(EnvNext) => /\ Earlier(Next)
                                                        /\ SysNext
                <4> DEFINE tau ==
                        LET state(n) == [sigma[n] EXCEPT !["b"] = (n <= i)]
                        IN [n \in Nat |-> state(n)]
                <4>1. /\ IsABehavior(tau)
                      /\ RefinesUpToVar(tau, sigma, "b")
                    BY DEF tau, IsABehavior, RefinesUpToVar,
                        Sim, Natural, EqualUpToVar
                <4>2. tau, 0 |= MustUnstep(b)
                    BY DEF tau, MustUnstep, Unstep, MayUnstep
                <4>3. tau, 0 |=
                        \/ ~ \/ ~ Ie
                             \/ /\ Je
                                /\ [][b' => Ne]_<< x, y >>
                        \/ /\ [][b' => Ns]_<< x, y >>
                           /\ [][b => \E r:  NsP(x, y, r, y')]_y
                    BY <4>1, <4>2, <3>3
                <4>4. SUFFICES ASSUME sigma, i |= Earlier(EnvNext)
                               PROVE sigma, i |= /\ Earlier(Next)
                                                 /\ SysNext
                    OBVIOUS
                <4>5. \A k \in 0..(i - 1):
                        << sigma[k], sigma[k + 1] >>[[ EnvNext ]]
                    BY <4>4 DEF Earlier
                <4>6. CASE i = 0
                    <5>1. tau, 0 |= /\ b = TRUE
                                    /\ [](b' = FALSE)
                        <6>1. /\ tau[0]["b"] = TRUE
                              /\ \A j \in Nat \ {0}:  tau[j]["b"] = FALSE
                            BY DEF tau, <4>6
                        <6> QED
                            BY <6>1
                    <5>2. tau, 0 |= /\ Je
                                    /\ [][b' => Ne]_<< x, y >>
                        <6>1. tau, 0 |= Je
                            <7>1. tau, 0 |= Ie
                                BY <3>5 DEF tau
                                (* IeP does not depend on b. *)
                            <7>2. tau, 0 |= Ie => Je
                                BY <3>2 DEF tau
                            <7> QED
                                BY <7>1, <7>2
                        <6>2. tau, 0 |= [][b' => Ne]_<< x, y >>
                            BY <5>1
                        <6> QED
                            BY <6>1, <6>2
                    <5>3. tau, 0 |=
                            /\ [][b' => Ns]_<< x, y >>
                            /\ [][b => \E r:  NsP(x, y, r, y')]_y
                        BY <4>3, <5>2
                    <5>4. tau, i |= SysNext /\ Earlier(Next)
                        <6>1. tau, 0 |= [b => \E r:  NsP(x, y, r, y')]_y
                            BY <5>3
                        <6>2. tau, 0 |= [\E r:  NsP(x, y, r, y')]_y
                            BY <6>1, <5>1
                        <6>3. tau, 0 |= SysNext
                            BY <6>2 DEF SysNext
                        <6>4. tau, 0 |= Earlier(Next)
                            BY DEF Earlier
                        <6>5. tau, 0 |= SysNext /\ Earlier(Next)
                            BY <6>3, <6>4
                        <6> QED
                            BY <6>5, <4>6
                    <5> QED  (* goal from <4>4 *)
                        BY <5>4 DEF tau, SysNext, Earlier, Next
                        (* because variable b does not occur in the formula
                            SysNext /\ Earlier(Next)
                        b is declared as VARIABLE b in <3>3 *)
                <4>7. CASE i > 0
                    <5>1. /\ \A j \in 0..i:  tau[j]["b"] = TRUE
                          /\ \A j \in Nat:  (j > i) => (tau[j]["b"] = FALSE)
                        BY DEF tau
                    <5>2. tau, i |= Earlier(EnvNext)
                        BY <4>4 DEF tau, Earlier, EnvNext, Ne
                    <5>3. \A k \in 0..(i - 1):
                            << tau[k], tau[k + 1] >>[[ EnvNext ]]
                        <6>1. /\ (i - 1) \in Nat
                              /\ (i - 1) >= 0
                              /\ (i - 1) < i
                            BY <3>5, <4>7
                        <6> QED
                            BY <5>2, <6>1 DEF Earlier
                    <5>4. tau, 0 |= [][b' => Ne]_<< x, y >>
                        <6>1. \A k \in 0..(i - 1):
                                << tau[k], tau[k + 1] >>[[
                                    [b' => Ne]_<< x, y >> ]]
                            BY <5>3 DEF EnvNext
                        <6>2. \A k \in Nat:  (k >= i) =>
                                << tau[k], tau[k + 1] >>[[
                                    [b' => Ne]_<< x, y >> ]]
                            <7>1. \A k \in Nat:  (k > i) =>
                                    << tau[k], tau[k + 1] >>[[ ~ b ]]
                                BY <5>1
                            <7>2. \A k \in Nat:  (k >= i) =>
                                    << tau[k], tau[k + 1] >>[[ ~ b' ]]
                                BY <7>1  (* b' = FALSE at these steps. *)
                            <7> QED
                                BY <7>2
                        <7>3. \A k \in Nat:
                                << tau[k], tau[k + 1] >>[[
                                    [b' => Ne]_<< x, y >> ]]
                            BY <6>1, <6>2
                        <7> QED
                            BY <7>3
                    <5>5. tau, 0 |= /\ [][b' => Ns]_<< x, y >>
                                    /\ [][b => \E r:  NsP(x, y, r, y')]_y
                        <6>1. tau, 0 |= /\ Je
                                        /\ [][b' => Ne]_<< x, y >>
                            <7>1. tau, 0 |= Je
                                <8>1. tau, 0 |= Ie
                                    BY <3>5 DEF tau
                                    (* IeP does not depend on b. *)
                                <8>2. tau, 0 |= Ie => Je
                                    BY <3>2 DEF tau
                                <8> QED
                                    BY <8>1, <8>2
                            <7>2. tau, 0 |= [][b' => Ne]_<< x, y >>
                                BY <5>4
                            <7> QED
                                BY <7>1, <7>2
                        <6> QED
                            BY <4>3, <6>1 DEF tau
                    <5>6. tau, i |= [\E r:  NsP(x, y, r, y')]_y
                        <6>1. tau, i |= b
                            BY <5>1
                        <6>2. tau, 0 |= [][b => \E r:  NsP(x, y, r, y')]_y
                            BY <5>5
                        <6>3. tau, i |= [b => \E r:  NsP(x, y, r, y')]_y
                            BY <6>2
                        <6> QED
                            BY <6>1, <6>3
                        BY <5>5, <5>1
                    <5>7. tau, i |= Earlier([Ns]_<< x, y >>)
                        <6>1. \A k \in Nat:  tau, k |= [b' => Ns]_<< x, y >>
                            <7>1. tau, 0 |= [][b' => Ns]_<< x, y >>
                                BY <5>5
                            <7> QED
                                BY <7>1
                        <6>2. \A k \in 0..i:  tau, k |= b
                            BY <5>1
                        <6>3. \A k \in 0..(i - 1):  tau, k |= b'
                            <7>1. /\ (i - 1) \in Nat
                                  /\ (i - 1) >= 0
                                  /\ (i - 1) < i
                                BY <3>5, <4>7
                            <7> QED
                                BY <6>2, <7>1
                        <6>4. \A k \in 0..(i - 1):
                                tau, k |= b' /\ [b' => Ns]_<< x, y >>
                            BY <6>1, <6>3
                        <6>5. \A k \in 0..(i - 1):
                                tau, k |= [Ns]_<< x, y >>
                            BY <6>4
                        <6> QED
                            BY <6>5 DEF Earlier
                        BY <5>1
                    <5>8. tau, i |= SysNext /\ Earlier(Next)
                        BY <5>6, <5>7 DEF SysNext, Next
                    <5> QED  (* goal from <4>4 *)
                        BY <5>8 DEF tau, SysNext, Earlier, Next
                <4> QED
                    <5>1. i \in Nat
                        BY <3>5
                    <5> QED  (* goal from <4>4 *)
                        BY <4>6, <4>7
            <3> QED
                BY <3>4, <3>5, <3>6  (* goal from <3>2 *)

        <2>8. ASSUME
                sigma, 0 |=
                    \/ ~ \/ ~ Ie
                         \/ Je
                    \/ /\ Is
                       /\ Ie \/ [](Next /\ SysNext)
                       /\ Ie => [](Earlier(EnvNext) => /\ Earlier(Next)
                                                       /\ SysNext)
              PROVE
                sigma |= \AA b:
                    \/ ~ MustUnstep(b)
                    \/ ~ \/ ~ Ie
                         \/ /\ Je
                            /\ [][b' => Ne]_<< x, y >>
                    \/ /\ Is
                       /\ [][b' => Ns]_<< x, y >>
                       /\ [][b => \E r:  NsP(x, y, r, y')]_y
            <3> USE DEF Ie, Je, Is, Ns, Ne
            <3>1. SUFFICES
                    ASSUME
                        NEW tau,
                        /\ IsABehavior(tau)
                        /\ RefinesUpToVar(tau, sigma, "b"),
                        VARIABLE b
                    PROVE
                        tau, 0 |=
                            \/ ~ MustUnstep(b)
                            \/ ~ \/ ~ Ie
                                 \/ /\ Je
                                    /\ [][b' => Ne]_<< x, y >>
                            \/ /\ Is
                               /\ [][b' => Ns]_<< x, y >>
                               /\ [][b => \E r:  NsP(x, y, r, y')]_y
                BY DEF \AA
            <3>2. SUFFICES
                    ASSUME
                        tau, 0 |=
                            /\ MustUnstep(b)
                            /\ Ie => /\ Je
                                     /\ [][b' => Ne]_<< x, y >>
                    PROVE
                        tau, 0 |=
                            /\ Is
                            /\ [][b' => Ns]_<< x, y >>
                            /\ [][b => \E r:  NsP(x, y, r, y')]_y
                OBVIOUS  (* goal from <3>1 *)
            <3> DEFINE
                F == \/ ~ \/ ~ Ie
                          \/ Je
                     \/ /\ Is
                        /\ Ie \/ [](Next /\ SysNext)
                        /\ Ie => [](Earlier(EnvNext) => /\ Earlier(Next)
                                                        /\ SysNext)
            <3>3. tau, 0 |= F
                <4>1. \A rho:
                        \/ ~ IsABehavior(rho)
                        \/ ~ Sim(rho, sigma)
                        \/ rho, 0 |= F
                    BY <2>8 DEF F  (* Even though F is not a TLA+ formula,
                        it is stutter-invariant. *)
                <4>2. \A rho, eta:
                        \/ ~ IsABehavior(eta)
                        \/ ~ IsABehavior(rho)
                        \/ ~ EqualUpToVar(rho, eta, "b")
                        \/ (eta, 0 |= F)  <=>  (rho, 0 |= F)
                    BY DEF F  (* The variable b does not occur in F. *)
                <4>3. /\ IsABehavior(tau)
                      /\ \E rho:  /\ IsABehavior(rho)
                                   /\ Sim(rho, sigma)
                                   /\ EqualUpToVar(rho, tau, "b")
                    BY <3>1 DEF RefinesUpToVar
                <4> QED
                    BY <4>1, <4>2, <4>3 DEF F

            <3>4. CASE tau, 0 |= ~ Ie
                <4>1. tau, 0 |=
                        /\ Is
                        /\ [](Next /\ SysNext)
                    BY <3>3, <3>4 DEF F
                <4>2. tau, 0 |=
                        /\ Is
                        /\ [][Ns]_<< x, y >>
                        /\ [][\E r:  NsP(x, y, r, y')]_y
                    BY <4>1 DEF Next, SysNext
                <4> QED  (* goal from <3>2 *)
                    BY <4>2

            <3>5. CASE tau, 0 |= Ie
                <4>1. tau, 0 |= /\ Je
                                /\ [][b' => Ne]_<< x, y >>
                    BY <3>2, <3>5
                <4>2. tau, 0 |=
                        /\ Is
                        /\ [](Earlier(EnvNext) => /\ Earlier(Next)
                                                  /\ SysNext)
                    <5>1. tau, 0 |= Ie => Je
                        BY <3>5, <4>1
                    <5> QED
                        BY <3>3, <5>1, <3>5 DEF F
                <4>3. SUFFICES tau, 0 |=
                            /\ [][b' => Ns]_<< x, y >>
                            /\ [][b => \E r:  NsP(x, y, r, y')]_y
                    BY <4>2  (* goal from <3>2 *)
                <4>4. PICK i \in Nat \ {0}:
                        /\ \A n \in 0..i:  tau, n |= b = TRUE
                        /\ \A n \in Nat:  (n > i) => (tau, n |= b = FALSE)
                    <5>1. tau, 0 |= MustUnstep(b)
                        BY <3>2
                    <5> QED
                        BY <5>1 DEF MustUnstep, Unstep, MayUnstep, tau

                <4>5. tau, i |= SysNext /\ Earlier(Next)
                    <5>1. \A n \in Nat:  tau, n |= [b' => Ne]_<< x, y >>
                        BY <4>1
                    <5>2. \A n \in 0..(i - 1):  tau, n |=  b'
                        BY <4>4
                    <5>3. \A n \in 0..(i - 1):
                            tau, n |= b' /\ [b' => Ne]_<< x, y >>
                        BY <5>2, <5>1
                    <5>4. \A n \in 0..(i - 1):
                            tau, n |= [Ne]_<< x, y >>
                        BY <5>3
                    <5>5. tau, i |= Earlier([Ne]_<< x, y >>)
                        BY <5>4 DEF Earlier
                    <5> QED
                        BY <5>5, <4>2

                <4>6. tau, 0 |= [][b' => Ns]_<< x, y >>
                    <5>1. \A n \in 0..(i - 1):
                            tau, n |= [b' => Ns]_<< x, y >>
                        <6>1. \A n \in 0..(i - 1):
                                tau, n |= [Ns]_<< x, y >>
                            BY <4>5 DEF Next
                        <6> QED
                            BY <6>1
                    <5>2. \A n \in Nat:  (n >= i)
                            => (tau, n |= [b' => Ns]_<< x, y >>)
                        <6>1. \A n \in Nat:
                                (n > i) => (tau, n |= b = FALSE)
                            BY <4>4
                        <6>2. \A n \in Nat:
                                (n >= i) => (tau, n |= b' = FALSE)
                            BY <6>1
                        BY <6>2
                    <5>3. \A n \in Nat:  tau, n |= [b' => Ns]_<< x, y >>
                        BY <5>1, <5>2
                    <5> QED
                        BY <5>3 DEF []
                <4>7. tau, 0 |= [][b => \E r:  Ns(x, y, r, y')]_y
                    <5>1. \A n \in 0..i:
                            tau, n |= [b => \E r:  NsP(x, y, r, y')]_y
                        <6>1. \A n \in 0..(i - 1):
                                tau, n |= \E r:  \/ NsP(x, y, r, y')
                                                 \/ (x = r)  /\  (y = y')
                            BY <4>13
                        <6>2. \A n \in 0..(i - 1):
                                tau, n |= [\E r:  NsP(x, y, r, y')]_y
                            BY <6>1
                        <6>3. \A n \in 0..i:
                                tau, n |= [\E r:  NsP(x, y, r, y')]_y
                            <7>1. tau, i |= [\E r:  NsP(x, y, r, y')]_y
                                BY <4>5 DEF SysNext
                            <7> QED
                                BY <6>2, <7>1
                        <6> QED
                            BY <6>3
                    <5>2. \A n \in Nat:  (n > i) =>
                            tau, n |= [b => \E r:  Ns(x, y, r, y')]_y
                        <6>1. \A n \in Nat:
                                (n > i) => (tau, n |= b = FALSE)
                            BY <4>4
                        <6> QED
                            BY <6>1
                    <5>3. \A n \in Nat:
                            tau, n |= [b => \E r:  Ns(x, y, r, y')]_y
                        BY <5>1, <5>2
                    <5> QED
                        BY <5>3 DEF []
                <4> QED  (* goal from <4>3 *)
                    BY <4>6, <4>7

            <3> QED  (* goal from <3>2 *)
                BY <3>4, <3>5

        <2> QED
            BY <2>5, <2>6, <2>8
    <1> QED
        BY <1>1, <1>2, <1>3, <1>4


(* The above theorem assumes that actions are defined using constants.
Essentially the same proof can be used when the environment action Ne
is defined using constant operators and Earlier (in other words, when this
is an action that results from converting WPH to raw TLA+ with past).

In that case the proof should be modified to address two points:


1. The operator A is declared (not defined) in TLA+ and assumed to be
   equivalent to a raw TLA+ formula. This assumption is:

    ( sigma |= C(A, x, y) )  <=>  (sigma, 0 |=
        Ie => /\ Je
              /\ [][Earlier(Ne) => (Earlier(N) /\ Ns))
        )

    For writing the operator Earlier we have to be within raw TLA+,
    which is why on the left-hand-side we have sigma |= Cl(A, x, y),
    whereas on the right-hand-side we have sigma, 0 |= ... .


2. The proof should be carried out mostly within raw TLA+ with past.
   In other words, we should "move" to raw TLA+ before the step that
   replaces the closure Cl(A, x, y) with a specific formula.

   Again, the reason is that the closure is expressed using past temporal
   operators, so we cannot write it in this form within TLA+.


3. Combining the two previous points, closure and past operators need to
   coexist within the same logic. This requires expressing temporal
   quantification \EE in raw TLA+ with past (since past operators need
   an indexed satisfaction relation (|=), so they are not expressible in TLA+).

   This definition is given in the module TemporalLogic.


4. When we reach the step of substituting u, v with x, y in the environment
   action (and vice versa in the reverse direction of proof), we have to
   do this replacement also within Earlier. This replacement is justified
   by observing that if b is true at some state in a behavior, then it must
   have been true in all previous states. Thus, << x, y >> = << u, v >> in
   all those previous states (similar argument to how SamePrefix is handled).
*)



(* We could write the existential quantifier outside the box [...]_y,
though that would be ungrammatical as an action after [].
*)
PROPOSITION
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT Next(_, _, _, _)
    PROVE
        LET
            (* \E x':  [Next(x, y, x', y')]_<< x, y >> *)
            (* Applying rigid quantification to a primed variable is
            ungrammatical in TLA+. *)
            A == \E u:  \/ Next(x, y, u, y')
                        \/ << u, y' >> = << x, y >>
            B == \E u:  [Next(x, y, u, y']_y
            C ==  [\E u:  Next(x, y, u, y')]_y
        IN
            /\ A <=> B
            /\ B <=> C
    <1> DEFINE
        A == \E u:  \/ Next(x, y, u, y')
                    \/ << u, y' >> = << x, y >>
        B == \E u:  [Next(x, y, u, y']_y
        C ==  [\E u:  Next(x, y, u, y')]_y
    <1>1. A <=> C
        <2>1. (\E u:  \/ Next(x, y, u, y')
                      \/ << u, y' >> = << x, y >>)
                <=>
                \/ \E u:  Next(x, y, u, y')
                \/ \E u:  << u, y' >> = << x, y >>
            OBVIOUS
        <2>2. (\E u:  << u, y' >> = << x, y >>)
                <=>  /\ \E u:  u = x
                     /\ y' = y
            OBVIOUS
        <2>3. (\E u:  \/ Next(x, y, u, y')
                      \/ << u, y' >> = << x, y >>)
                <=> \/ \E u:  Next(x, y, u, y')
                    \/ y' = y
            BY <2>1, <2>2
        <2> QED
            BY <2>3 DEF A, C
    <1>2. B <=> C
        <2>1. (\E u:  [Next(x, y, u, y']_y)
                <=> \E u:  \/ Next(x, y, u, y')
                           \/ y' = y
            OBVIOUS
        <2>2. B <=> \/ \E u:  Next(x, y, u, y')
                    \/ y' = y
            BY <2>1 DEF B
        <2> QED
            BY <2>2 DEF C
    <1> QED
        BY <1>1, <1>2


--------------------------------------------------------------------------------
(* Expressing Unzip in raw TLA+ with past. *)


(* We apply the raw form of WhilePlusHalf twice, recursively. The form was
proved for constant actions, but as noted above the proof can be modified
for the case of an environment action that contains past temporal operators.
*)
THEOREM
    ASSUME
        VARIABLE x, VARIABLE y,
        CONSTANT I(_, _),
        CONSTANT N(_, _),
        TEMPORAL L(_, _)
    PROVE
        LET
            P(u, v) == /\ I(u, v) /\ L(u, v)
                       /\ [][N(u, v, u', v')]_<< u, v >>
            EnvNext == [\E r:  N(x, y, x', r)]_x
            SysNext == [\E r:  N(x, y, r, y')]_y
            Next == [N(x, y, x', y')]_<< x, y >>
            Raw ==
                /\ \E p:  I(p, y)
                /\ \/ ~ \E q:  I(x, q)
                   \/ /\ I(x, y)
                      /\ [] \/ ~ Earlier(EnvNext)
                            \/ SysNext /\ Earlier(Next)
                      /\ ([]EnvNext)  =>  L(x, y)
        IN
            Unzip(P, x, y) <=> Raw
    PROOF
    <1> DEFINE
        P(u, v) == /\ I(u, v) /\ L(u, v)
                   /\ [][N(u, v, u', v')]_<< u, v >>
        Q(u, v) == P(v, u)
    <1>1. Unzip(P, x, y) <=>
            LET
                A(u, v) == WPH(Q, Q, v, u)
            IN
                WPH(A, P, x, y)
        BY DEF Unzip
    <1>2. ASSUME VARIABLE u, VARIABLE v
          PROVE WPH(Q, Q, v, u) <=>
            LET
                F == \E p, q:  I(p, q)
                G == \E q:  I(u, q)
                Ie == F /\ (G => I(u, v))
                Je == G
                Next == [N(u, v, u', v')]_<< u, v >>
                EnvNext == [\E r:  N(u, v, u', r)]_u
            IN
                \/ ~ Ie
                \/ Je /\ [](Earlier(Next) => EnvNext)
        <2> DEFINE
            F == \E p, q:  I(p, q)
            G == \E q:  I(u, q)
            Ie == F /\ (G => I(u, v))
            Je == G
            Next == [N(u, v, u', v')]_<< u, v >>
            EnvNext == [\E r:  N(u, v, u', r)]_u
        <2>1. WPH(Q, Q, v, u) <=>
                \/ ~ \E p, q:  TRUE => I(p, q)
                \/ /\ \E q:  I(u, q)
                   /\ \/ ~ \/ ~ TRUE
                           \/ I(u, v)
                      \/ /\ I(u, v)
                         /\ I(u, v) \/ [](Next /\ EnvNext)
                         /\ \/ ~ I(u, v)
                            \/ [](Earlier(Next) => /\ Earlier(Next)
                                                   /\ EnvNext)
                         /\ \/ ~ \/ ~ TRUE
                                 \/ I(u, v) /\ L(u, v) /\ []Next
                            \/ L(u, v)
            BY RawWhilePlusHalfFull DEF Q
        <2>2. WPH(Q, Q, v, u) <=>
                \/ ~ \E p, q:  I(p, q)
                \/ /\ \E q:  I(u, q)
                   /\ \/ ~ I(u, v)
                      \/ [](Earlier(Next) => EnvNext)
                   /\ \/ ~ L(u, v)
                      \/ ~ []Next
                      \/ L(u, v)
            BY <2>1
        <2>3. WPH(Q, Q, v, u) <=>
                \/ ~ \E p, q:  I(p, q)
                \/ /\ \E q:  I(u, q)
                   /\ \/ ~ I(u, v)
                      \/ [](Earlier(Next) => EnvNext)
            BY <2>2
        <2>4. WPH(Q, Q, v, u) <=>
                \/ ~ F
                \/ /\ G
                   /\ \/ ~ I(u, v)
                      \/ [](Earlier(Next) => EnvNext)
            BY <2>3 DEF F, G
        <2>5. WPH(Q, Q, v, u) <=>
                \/ ~ F
                \/ G /\ ~ I(u, v)
                \/ G /\ [](Earlier(Next) => EnvNext)
            BY <2>4
        <2>6. WPH(Q, Q, v, u) <=>
                \/ ~ /\ F
                     /\ G => I(u, v)
                \/ G /\ [](Earlier(Next) => EnvNext)
            BY <2>5
        <2> QED
            BY <2>6 DEF Ie, Je, F, G
    <1> DEFINE
        F == \E p, q:  I(p, q)
        G == \E q:  I(x, q)
        Ie == F /\ (G => I(x, y))
        Je == G
        (* These definitions differ from those in <1>2 because they are
        in terms of x, y instead of u, v. *)
        Next == [N(x, y, x', y')]_<< x, y >>
        EnvNext == [\E r:  N(x, y, x', r)]_x
        SysNext == [\E r:  N(x, y, r, y')]_y
    <1>3. Unzip(P, x, y) <=>
            \/ ~ \E u, v:
                \/ ~ /\ \E p, q:  I(p, q)
                     /\ \/ ~ \E q:  I(u, q)
                        \/ I(u, v)
                \/ \E q:  I(u, q)
            \/ /\ \E p:  I(p, y)
               /\ \/ ~ \/ ~ Ie
                       \/ Je
                  \/ /\ I(x, y)
                     /\ Ie \/ [](Next)
                     /\ Ie => [] \/ ~ Earlier(Earlier(Next) => EnvNext)
                                 \/ SysNext /\ Earlier(Next)
                     /\ \/ ~ \/ ~ Ie
                             \/ Je /\ [](Earlier(Next) => EnvNext)
                        \/ L(x, y)
        BY DEF <1>1, <1>2, WhilePlusHalfStepwiseForm
            (* with the caveat about WhilePlusHalfStepwiseForm and past
            operators within the environment action that was noted ealier *)
    <1>11. \/ ~ I(x, y)
           \/ Ie
        <2>1. I(x, y)  =>  F
            <3>1. I(x, y)  =>  \E p, q:  I(p, q)
                OBVIOUS
            <3> QED
                BY <3>1 DEF F
        <2>2. I(x, y)  =>  (G => I(x, y))
            OBVIOUS
        <2> QED
            BY <2>1, <2>2 DEF Ie
    <1>4. \E u, v:
            \/ ~ /\ \E p, q:  I(p, q)
                 /\ \/ ~ \E q:  I(u, q)
                    \/ I(u, v)
            \/ \E q:  I(u, q)
        <2>1. (\E u, v:
                    \/ ~ /\ \E p, q:  I(p, q)
                         /\ \/ ~ \E q:  I(u, q)
                            \/ I(u, v)
                    \/ \E q:  I(u, q)
                ) <=> (
                \/ \E u, v:
                    ~ /\ \E p, q:  I(p, q)
                      /\ \/ ~ \E q:  I(u, q)
                         \/ I(u, v)
                \/ \E u, v:  \E q:  I(u, q)
                )
            OBVIOUS
        <2>2. (\E u, v:  \E q:  I(u, q))
                <=> \E p, q:  I(p, q)
            OBVIOUS
        <2>3. (~ /\ \E p, q:  I(p, q)
                 /\ \/ ~ \E q:  I(u, q)
                     \/ I(u, v)
                ) <=> (
                    \/ ~ \E p, q:  I(p, q)
                    \/ ~ \/ ~ \E q:  I(u, q)
                         \/ I(u, v))
            OBVIOUS
        <2>4. (\E u, v:
                ~ /\ \E p, q:  I(p, q)
                  /\ \/ ~ \E q:  I(u, q)
                     \/ I(u, v))
                <=> (
                    \/ \E u, v:  ~ \E p, q:  I(p, q)
                    \/ \E u, v:  /\ \E q:  I(u, q)
                                 /\ ~ I(u, v))
            BY <2>3
        <2>5. (\E u, v:
                    \/ ~ /\ \E p, q:  I(p, q)
                         /\ \/ ~ \E q:  I(u, q)
                            \/ I(u, v)
                    \/ \E q:  I(u, q)
                ) <=> (
                \/ \E u, v:  ~ \E p, q:  I(p, q)
                \/ \E u, v:  /\ \E q:  I(u, q)
                             /\ ~ I(u, v)
                \/ \E p, q:  I(p, q)
                )
            BY <2>1, <2>2, <2>4
        <2>6. (\E u, v:
                    \/ ~ /\ \E p, q:  I(p, q)
                         /\ \/ ~ \E q:  I(u, q)
                            \/ I(u, v)
                    \/ \E q:  I(u, q)
                ) <=> (
                \/ ~ \E p, q:  I(p, q)
                \/ \E p, q:  I(p, q)
                \/ \E u, v:  /\ \E q:  I(u, q)
                             /\ ~ I(u, v)
                )
            BY <2>5
        <2> QED
            BY <2>6
    <1>5. \/ ~ F
          \/ (Ie => Je)  <=> G
        <2>1. (Ie => Je)
                <=> ((F /\ (G => I(x, y))) =>  G)
            BY DEF Ie, Je
        <2>2. \/ ~ F
              \/ (Ie => Je)
                    <=> ((G => I(x, y)) => G)
            BY <2>1
        <2>3. G  <=>  ((G => I(x, y)) => G)
            <3>1. ((G => I(x, y)) => G)
                    <=> \/ ~ (G => I(x, y))
                        \/ G
                OBVIOUS
            <3>2. ((G => I(x, y)) => G)
                    <=> \/ G /\ ~ I(x, y)
                        \/ G
                BY <3>1
            <3> QED
                BY <3>2
        <2> QED
            BY <2>2, <2>3
    <1>6. Unzip(P, x, y) <=>
            /\ \E p:  I(p, y)
            /\ \/ ~ G
               \/ /\ I(x, y)
                  /\ [] \/ ~ Earlier(Earlier(Next) => EnvNext)
                        \/ SysNext /\ Earlier(Next)
                  /\ \/ ~ (Je /\ [](Earlier(Next) => EnvNext)
                     \/ L(x, y)
        BY <1>3, <1>4, <1>5, <1>11
    <1>7. ASSUME NEW sigma, IsABehavior(sigma)
           PROVE
            (sigma, 0 |=
                [] \/ ~ Earlier(Earlier(Next) => EnvNext)
                   \/ SysNext /\ Earlier(Next))
            <=>
            sigma, 0 |=
                [] \/ ~ Earlier(EnvNext)
                   \/ SysNext /\ Earlier(Next)
        <2> DEFINE
            A == \/ ~ Earlier( \/ ~ Earlier(Next)
                               \/ EnvNext )
                 \/ SysNext /\ Earlier(Next)
            B == Earlier(EnvNext) => /\ Earlier(Next)
                                     /\ SysNext
        <2>1. ASSUME \A n \in Nat:  sigma, n |= A
              PROVE \A n \in Nat:  sigma, n |= B
            <3>1. SUFFICES
                    ASSUME NEW n \in Nat,
                    PROVE sigma, n |= B
                OBVIOUS
            <3>2. SUFFICES
                    ASSUME sigma, n |= Earlier(EnvNext)
                    PROVE sigma, n |= SysNext /\ Earlier(Next)
                BY DEF B  (* goal from <3>1 *)
            <3>3. SUFFICES
                    sigma, n |= Earlier(Earlier(Next) => EnvNext)
                <4>1. sigma, n |= A
                    <5>1. \A k \in Nat:  sigma, k |= A
                        BY <2>1
                    <5>2. n \in Nat
                        BY <3>1
                    <5> QED
                        BY <5>1, <5>2
                <4>2. sigma, n |= \/ ~ Earlier(Earlier(Next) => EnvNext)
                                  \/ SysNext /\ Earlier(Next)
                    BY <4>1 DEF B
                <4> QED
                    BY <3>3, <4>2  (* goal from <3>2 *)
            <3>4. (sigma, n |= Earlier(EnvNext))
                    => sigma, n |= Earlier(EnvNext \/ ~ Earlier(Next))
                <4>1. EnvNext => (EnvNext \/ ~ Earlier(Next))
                    OBVIOUS
                <4>2. n \in Nat
                    BY <3>1
                <4>3. IsABehavior(sigma)
                    BY <1>99
                <4> QED
                    BY <4>1, <4>2, <4>3 DEF Earlier
            <3> QED
                BY <3>2, <3>4  (* goal from <3>3 *)
        <2>2. ASSUME \A n \in Nat:  sigma, n |= B
              PROVE \A n \in Nat:  sigma, n |= A
            <3>1. SUFFICES
                    ASSUME NEW n \in Nat
                    PROVE sigma, n |= A
                OBVIOUS
            <3>2. SUFFICES
                    ASSUME sigma, n |= Earlier(Earlier(Next) => EnvNext)
                    PROVE sigma, n |= SysNext /\ Earlier(Next)
                BY DEF A  (* goal from <3>1 *)
            <3>3. SUFFICES
                    sigma, n |= Earlier(EnvNext)
                <4>1. sigma, n |= B
                    <5>1. \A k \in Nat:  sigma, n |= B
                        BY <2>2
                    <5>2. n \in Nat
                        BY <3>1
                    <5> QED
                        BY <5>1, <5>2
                <4>2. sigma, n |= Earlier(EnvNext) => /\ Earlier(Next)
                                                      /\ SysNext
                    BY <4>1 DEF B
                <4> QED
                    BY <3>3, <4>2  (* goal from <3>2 *)
            <3>4. SUFFICES
                    ASSUME sigma, n |= ~ Earlier(EnvNext)
                    PROVE FALSE
                OBVIOUS  (* goal from <3>3 *)
            <3>5. sigma, 0 |= ~ Earlier(EnvNext)
                <3> DEFINE
                    P(m) == sigma, m |= ~ Earlier(EnvNext)
                (* prepare for downward induction *)
                <4>1. \A k \in 1..n:  P(k) => P(k - 1)
                    <5>1. CASE n = 0
                        OBVIOUS
                    <5>2. SUFFICES ASSUME n > 0
                                   PROVE \A k \in 1..n:  P(k) => P(k - 1)
                        <6>1. n \in Nat
                            BY <3>1
                        <6> QED
                            BY <6>1, <5>1, <5>2
                    <5>3. SUFFICES
                            ASSUME NEW k \in 1..n, P(k)
                            PROVE P(k - 1)
                        OBVIOUS  (* goal from <5>2 *)
                    <5>4. sigma, k |= ~ Earlier(EnvNext)
                        BY <5>3 DEF P
                    <5>5. PICK j \in 0..(k - 1):  sigma, j |= ~ EnvNext
                        <7>1. (k > 0) /\ (k \in Nat)
                            <8>1. k \in 1..n
                                BY <5>3
                            <8>2. 1 \in 1..n  (* thus 1..n # {} *)
                                BY <3>1, <5>2
                            <8> QED
                                BY <8>1, <8>2
                        <7>2. ~ \A r \in 0..(k - 1):  sigma, r |= EnvNext
                            BY <5>4 DEF Earlier  (* the general DEF for past
                                operators *)
                        <7>3. \E r \in 0..(k - 1):  sigma, r |= ~ EnvNext
                            BY <7>2
                        <7>4. 0 \in 0..(k - 1)  (* thus 0..(k - 1) # {} *)
                            BY <7>1
                        <7> QED
                            BY <7>3, <7>4
                    <5>6. j \in 0..(n - 1)
                        <6>1. k \in 1..n
                            BY <5>3
                        <6>2. j \in 0..(k - 1)
                            BY <5>5
                        <6> QED
                            BY <6>1, <6>2
                    <5>7. sigma, j |= Earlier(Next) => EnvNext
                        <6>1. sigma, n |= Earlier(Earlier(Next) => EnvNext)
                            BY <3>2
                        <6>2. \A r \in 0..(n - 1):
                                sigma, r |= Earlier(Next) => EnvNext
                            BY <6>1, <3>1 DEF Earlier
                        <6> QED
                            BY <6>2, <5>6
                    <5>8. sigma, j |= ~ Earlier(Next)
                        BY <5>5, <5>7
                    <5>9. sigma, (k - 1) |= ~ Earlier(Next)
                        BY <5>8, <5>5, <3>1 DEF Earlier
                    <5>10. sigma, (k - 1) |=
                            \/ ~ Earlier(EnvNext)
                            \/ Earlier(Next) /\ SysNext
                        <6>1. (k - 1) \in 0..(n - 1)
                            BY <5>3
                        <6>2. (k - 1) \in Nat
                            BY <6>1
                        <6> QED
                            BY <2>2, <6>1 DEF B  (* n <- (k - 1) *)
                    <5>11. sigma, (k - 1) |= ~ Earlier(EnvNext)
                        BY <5>9, <5>10
                    <5> QED
                        BY <5>11 DEF P  (* goal from <5>3 *)
                <4>2. P(0)
                    BY <3>5, DownwardNatInduction
                    (* see NaturalsInduction *)
                <4> QED
                    BY <4>2 DEF P
            <3>6. sigma, 0 |= Earlier(EnvNext)
                BY DEF Earlier
            <3> QED
                BY <3>5, <3>6
        <2> QED
            BY <2>1, <2>2 DEF []
    <1>8. Unzip(P, x, y) <=>
            /\ \E p:  I(p, y)
            /\ \/ ~ G
               \/ /\ I(x, y)
                  /\ [] \/ ~ Earlier(EnvNext)
                        \/ SysNext /\ Earlier(Next)
                  /\ \/ ~ (Je /\ [](Earlier(Next) => EnvNext)
                     \/ L(x, y)
        BY <1>6, <1>7
    <1>9. Unzip(P, x, y) <=>
            /\ \E p:  I(p, y)
            /\ \/ ~ \E q:  I(x, q)
               \/ /\ I(x, y)
                  /\ [](Earlier(EnvNext) => /\ Earlier(Next)
                                            /\ SysNext)
                  /\ \/ ~ [](Earlier(Next) => EnvNext)
                     \/ L(x, y)
        BY <1>8 DEF G
    <1>10. \/ ~ [](Earlier(EnvNext) => /\ Earlier(Next)
                                       /\ SysNext)
           \/ ( [](Earlier(Next) => EnvNext) )
              <=> []EnvNext
        OMITTED
    <1> QED
        BY <1>9, <1>10

================================================================================
