-------------------------- MODULE WhilePlusTheorems ----------------------------
(* We prove the equivalence of the "while-plus" operator -+-> to a formula in
raw TLA+ with the past operator Earlier. In other words, we convert -+-> from
TLA+ to a stepwise formula in raw TLA+ with past (PastRTLA+) that is more
suitable for using synthesis algorithms originally developed for LTL [5].
The result that we formally prove is analogous to [4, Lemma B.1 on p.70].

Due to the past operator, the satisfaction relation |= of PastRTLA+ resembles
that of LTL (it includes an index of the behavior state).
So for PastRTLA+ formulas we will use the notation

    sigma, i |= phi

and for TLA+ formulas the notation

    sigma |= phi

If phi is a TLA+ formula, then we can apply the equivalence

    (sigma |= phi)   <=>   (sigma, 0 |= phi)


For the closure of a behavior sigma:

    (sigma |= Cl(F))   <=>   (sigma |= \A n \in Nat:  PrefixSat(sigma, n, F))

In PastRTLA+ we will allow writing [A]_v as shorthand for A \/ (v = v').
On its own, this expression is ungrammatical in TLA+.

The directive BY Semantics refers to PastRTLA+ and TLA+ semantics.


Author:  Ioannis Filippidis


References
==========

[1] M. Abadi and L. Lamport, "Conjoining specifications", TOPLAS, 1995
    10.1145/203095.201069

[2] L. Lamport, "Miscellany", 21 April 1991

[3] M. Abadi and S. Merz, "On TLA as a logic", Deductive Program Design, 1996

[4] B. Jonsson and Y.-K. Tsay, "Assumption/guarantee specifications in
    linear-time temporal logic", TCS, 1996
    10.1016/0304-3975(96)00069-2

[5] Y.-K. Tsay, "Compositional verification in linear-time temporal logic",
    FOSSACS, 2000
    10.1007/3-540-46432-8_23

[6] U. Klein and A. Pnueli, "Revisiting synthesis of GR(1) specifications",
    HVC, 2010
    10.1007/978-3-642-19583-9_16

[7] L. Lamport, "Specifying concurrent program modules", TOPLAS, 1983
    10.1145/69624.357207

[8] K.L. McMillan, "Circular compositional reasoning about liveness",
    CHARME, 1999, 10.1007/3-540-48153-2_30

[9] K.S. Namjoshi and R.J. Trefler,
    "On the completeness of compositional reasoning methods", TOCL, 2010
    10.1145/1740582.1740584

--------------------------------------------------------------------------------
Copyright 2016-2017 by California Institute of Technology.
All rights reserved. Licensed under 3-clause BSD.
*)
EXTENDS TLASemantics, TemporalLogic, Integers, TLAPS


(* Definitions of past operators. For A an action, UpToNow corresponds
to Historically in LTL and Earlier to WeakPrevious Historically.
A different definition is needed when A is a temporal formula (using the
Suffix operator), but we apply these operators to actions only.
*)
sigma, i |= UpToNow(A) == \A k \in 0..i:
                            << sigma[k], sigma[k + 1] >>[[ A ]]
sigma, i |= Earlier(A) == \A k \in 0..(i - 1):
                            << sigma[k], sigma[k + 1] >>[[ A ]]

(* The definitions that work for A an arbitrary temporal formula. *)
sigma, i |= UpToNowTemporal(A) ==
    \A k \in 0..i:  Suffix(sigma, k), 0 |= A
sigma, i |= EarlierTemporal(A) ==
    \A k \in 0..(i - 1):  Suffix(sigma, k), 0 |= A

(* The syntactic definition of closure requires keeping track of variables,
which is cumbersome. In this module we use the following semantic definition.
*)
sigma |= Cl(P) == \A n \in Nat:  PrefixSat(sigma, n, P)

--------------------------------------------------------------------------------
(* Incremental implication spread over a behavior.
The operator Earlier is of PastRTLA+.
*)
StepwiseImpl(EnvNext, SysNext) == [](Earlier(EnvNext) => SysNext)
(* Causal but not strictly [6] *)
WeakStepwiseImpl(EnvNext, SysNext) == [](UpToNow(EnvNext) => SysNext)

(* The "trianglelefteq" operator defined in [7, p.220].
*)
sigma |= AsLongAs(P, Q) ==
    \A n \in Nat:
        (\A m \in 0..n:  Suffix(sigma, m) |= P)
        => (Suffix(sigma, n) |= Q)
(* The "vartriangleleft" operator defined in [7, p.220].
The operators OneStepLonger and PrefixPlusOne are inequivalent.
*)
sigma |= OneStepLonger(P, Q) ==
    \A n \in Nat:
        (\A m \in 0..(n - 1):  Suffix(sigma, m) |= P)
        => (Suffix(sigma, n) |= Q)
(* The operator OneStepLonger can be expressed using the operator AsLongAs.
*)
THEOREM ASSUME TEMPORAL P, Q
        PROVE OneStepLonger(P, Q) <=> AsLongAs(Q => P, Q)
    PROOF OMITTED

(* An operator defined in [9, p.16:3] and slightly differently in [8]. *)
NotUntil(EnvNext, SysNext) == ~ Until(EnvNext, ~ SysNext)

(* Comparing the definitions of Lamport [7], Klein and Pnueli [6],
MacMillan [8], Namjoshi and Trefler [9].
*)
THEOREM ASSUME ACTION E, S
        PROVE StepwiseImpl(E, S) <=> OneStepLonger(E, S)
    PROOF OMITTED

THEOREM ASSUME ACTION E, S
        PROVE WeakStepwiseImpl(E, S) <=> AsLongAs(E, S)

THEOREM ASSUME ACTION E, S
        PROVE NotUntil(E, S) <=> OneStepLonger(E, S)
    PROOF OMITTED


--------------------------------------------------------------------------------
(*  The RawWhilePlus operator is essentially the same with that studied
by Klein and Pnueli [6]. The differences are in the strict causality and
the initial condition
(akin to comparing `^$-\hspace{-.18em}\triangleright$^' and -+->).

If the component initial condition Is constrains the initial value of
component variables y, then use appropriate DEF of realizability.

If SysNext constrains x' (next env var values), then RawWhilePlus is
unrealizable (for the same reason -+-> is unrealizable in that case).
LTL synthesis literature passes SysNext that leaves x' unconstrained,
so unrealizability does not arise there, but other issues do.

If SysNext results by rewriting a property as the conjunction of
a machine-closed pair, then x' can happen to be constrained.
If so, then unrealizability arises.

Any closed-system property G in A -+-> G has this issue (because the
rewriting is always possible, and then the claims we prove apply).
Only if G leaves x entirely unconstrained is unrealizability avoided.
However, in that case G allows wild behavior within PrefixSat(sigma, n, G).
*)
RawWhilePlus(
        IeP(_, _), Ie, Is,
        EnvNext, SysNext,
        Le, Ls) ==
    \/ ~ \E p, q:  IeP(p, q)  (* unsatisfiable assumption ? *)
    \/ /\ Is
       /\ \/ ~ Ie
          \/ /\ StepwiseImpl(EnvNext, SysNext)
             /\ ([]EnvNext /\ Le) => Ls

(* The RawWhilePlus operator offers 5 degrees of freedom,
emphasized by the following canonical forms. The forms differ by whether the
main operator is conjunction or disjunction.
*)
RawWhilePlusConj(InitA, InitB, EnvNext, SysNext, Liveness) ==
    /\ InitB
    /\ InitA => /\ StepwiseImpl(EnvNext, SysNext)
                /\ \/ <> ~ EnvNext
                   \/ Liveness

RawWhilePlusDisj(InitC, InitD, EnvNext, SysNext, Liveness) ==
    InitC => /\ InitD
             /\ StepwiseImpl(EnvNext, SysNext)
             /\ \/ <> ~ EnvNext
                \/ Liveness

(* The operators RawWhilePlusConj and RawWhilePlusDisj can express the
same properties, as shown by the following two theorems.
*)
THEOREM
    ASSUME
        CONSTANT IeP(_, _),
        STATE Ie, STATE Is,
        ACTION EnvNext, ACTION SysNext,
        TEMPORAL Le, TEMPORAL Ls
    PROVE
        LET
            InitB == (\E p, q:  IeP(p, q) => Is
            InitA == Ie
            Liveness == Le => Ls
        IN
            RawWhilePlusConj(InitA, InitB, EnvNext, SysNext, Liveness)
            <=> RawWhilePlus(IeP, Ie, Is, EnvNext, SysNext, Le, Ls)
    PROOF OBVIOUS


THEOREM
    ASSUME
        CONSTANT IeP(_, _),
        STATE Ie, STATE Is,
        ACTION EnvNext, ACTION SysNext,
        TEMPORAL Le, TEMPORAL Ls,
    PROVE
        LET
            InitC == /\ \E p, q:  IeP(p, q)
                     /\ Is => Ie
            InitD == Is
            Liveness == Le => Ls
        IN
            RawWhilePlusDisj(InitC, InitD, EnvNext, SysNext, Liveness)
            <=> RawWhilePlus(IeP, Ie, Is, EnvNext, SysNext, Le, Ls)
    PROOF OBVIOUS


PROPOSITION AlwaysSysNextImpliesStepwiseImpl ==
        \/ ~ [] SysNext
        \/ StepwiseImpl(EnvNext, SysNext)
    PROOF
    <1>1. ([] SysNext)
          => [](Earlier(EnvNext) => SysNext)
        BY PTL
    <1> QED
        BY <1>1 DEF StepwiseImpl


PROPOSITION AlwaysEnvNextAndStepwiseImpl ==
        \/ ~ /\ [] EnvNext
             /\ StepwiseImpl(EnvNext, SysNext)
        \/ [] SysNext
    PROOF
    <1>1. ([] EnvNext)
          => [] Earlier(EnvNext)
        BY DEF Earlier
    <1>2. \/ ~ /\ [] Earlier(EnvNext)
               /\ [](Earlier(EnvNext) => SysNext)
          \/ [] SysNext
        BY PTL
    <1> QED
        BY <1>1, <1>2


(* Converting between PastRTLA+ and TLA+.

The raw logic allows for stutter-sensitive properties, though the
motivation for using the raw logic is to translate to a stepwise form
and connect with results on fixpoint algorithms.

The satisfaction relation (|=) can be defined in two ways: with or without
an explicit index of a state in the behavior (i.e., sigma |= P versus
sigma, index |= P). TLA does not use such an index. An index is necessary
to define past operators, because an index stores information from previous
states in a behavior. We use an index, in order to include past operators.

There are two flavors of temporal quantification: one that preserves
stutter-invariance (\EE), and one that does not. The definition of \EE in TLA
and raw past TLA differ, because we are using |= with an index. See the module
TemporalLogic for how |EE is defined in raw TLA with past.
*)
LEMMA CommonModels ==
    ASSUME TEMPORAL F,
           IsATLAPlusFormula(F)
    PROVE (sigma, 0 |= F)  <=>  (sigma |= F)
PROOF
BY Semantics

--------------------------------------------------------------------------------
(* Relating PrefixSat to closure. *)


LEMMA PrefixSatForClosure ==
    ASSUME
        TEMPORAL P,
        NEW n \in Nat,
        NEW sigma,
        IsABehavior(sigma)
    PROVE
        PrefixSat(sigma, n, P) <=> PrefixSat(sigma, n, Cl(P))
    PROOF
    <1>1. PrefixSat(sigma, n, P)
            <=> \E tau:  /\ IsABehavior(tau)
                         /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                         /\ tau |= P
        BY DEF PrefixSat
    <1>2. PrefixSat(sigma, n, Cl(P))
            <=> \E tau:  /\ IsABehavior(tau)
                         /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                         /\ tau |= Cl(P)
        BY DEF PrefixSat
    <1>3. PrefixSat(sigma, n, P) => PrefixSat(sigma, n, Cl(P))
        <2>1. P => Cl(P)
            BY ClosureImplied
        <2> QED
            BY <1>1, <2>1, <1>2
    <1>4. PrefixSat(sigma, n, Cl(P)) => PrefixSat(sigma, n, P)
        <2>1. SUFFICES ASSUME PrefixSat(sigma, n, Cl(P))
                       PROVE PrefixSat(sigma, n, P)
            OBVIOUS
        <2>2. PICK tau:
                /\ IsABehavior(tau)
                /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                /\ tau |= Cl(P)
            BY <2>1, <1>2
        <2>3. \A r \in Nat:  PrefixSat(tau, r, P)
            <3>1. tau |= Cl(P)
                BY <2>2
            <3> QED
                BY <3>1 DEF Cl  (* the semantic definition of closure *)
        <2>4. PrefixSat(tau, n, P)
            BY <2>3
        <2>5. PICK eta:  /\ IsABehavior(eta)
                         /\ \A i \in 0..(n - 1):  eta[i] = tau[i]
                         /\ eta |= P
            BY <2>4 DEF PrefixSat
        <2>6. \A i \in 0..(n - 1):  eta[i] = sigma[i]
            <3>1. /\ \A i \in 0..(n - 1):  eta[i] = tau[i]
                  /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                BY <2>2, <2>5
            <3> QED
                BY <3>1
        <2>7. /\ IsABehavior(eta)
              /\ \A i \in 0..(n - 1):  eta[i] = sigma[i]
              /\ eta |= P
            BY <2>5, <2>6
        <2> QED
            BY <2>7, <1>1
    <1> QED
        BY <1>3, <1>4

--------------------------------------------------------------------------------
(* One direction of PhiEquivRawPhi. *)


PROPOSITION RawPhiImpliesPhiStep11 ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        CONSTANT IeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
            RawPhi == RawWhilePlus(
                IeP, Ie, Is,
                EnvNext, SysNext, Le, Ls)
        IN
            sigma, 0 |= RawPhi
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
        IN
            sigma |= A => G
PROOF
<1> DEFINE
    v == << x, y >>
    Is == IsP(x, y)
    Ie == IeP(x, y)
    Ne == NeP(x, y, x', y')
    Ns == NsP(x, y, x', y')
    A == Ie /\ [][Ne]_v /\ Le
    G == Is /\ [][Ns]_v /\ Ls
    EnvNext == [Ne]_v
    SysNext == [Ns]_v
    RawPhi == RawWhilePlus(
        IeP, Ie, Is,
        EnvNext, SysNext, Le, Ls)
<1>1. (sigma, 0 |= A)  <=>  (sigma |= A)
    <2>1. IsATLAPlusFormula(A)
        BY DEF A, Ie, Ne
    <2> QED
        BY <2>1, CommonModels DEF A
<1>2. (sigma, 0 |= G)  <=>  (sigma |= G)
    <2>1. IsATLAPlusFormula(G)
        BY DEF G, Is, Ns
    <2> QED
        BY <2>1, CommonModels DEF G
<1>3. SUFFICES ASSUME sigma, 0 |= A
               PROVE sigma |= G
    <2>1. (sigma |= A => G)
           <=> ( (sigma |= A) => (sigma |= G) )
       BY Semantics
    <2>2. CASE ~ (sigma, 0 |= A)
        <3>1. ~ (sigma |= A)
            BY <2>2, <1>1
        <3>2. (sigma |= A) => (sigma |= G)
            BY <3>1
        <3> QED
            BY <3>2, <2>1
    <2>3. CASE sigma, 0 |= A
        <3>1. sigma |= G
            BY <1>3
        <3>2. (sigma |= A) => (sigma |= G)
            BY <3>1
        <3> QED
            BY <3>2, <2>1
    <2> QED
        BY <2>2, <2>3
<1>4. /\ sigma |= Ie /\ [][Ne]_v /\ Le
      /\ sigma, 0 |= Ie /\ [][Ne]_v /\ Le
    <2>1. sigma |= A
        BY <1>3, <1>1
    <2>2. sigma |= Ie /\ [][Ne]_v /\ Le
        BY <2>1 DEF A
    <2>3. sigma, 0 |= Ie /\ [][Ne]_v /\ Le
        BY <2>2, CommonModels DEF Ie, Ne
    <2> QED
        BY <2>2, <2>3
<1>5. \E p, q:  IeP(p, q)  (* The assumption is satisfiable. *)
    <2>1. sigma |= Ie
        BY <1>4
    <2>2. sigma |= IeP(x, y)
        BY <2>1 DEF Ie
    <2> QED
        BY <2>2
<1>6. sigma, 0 |= /\ Is
                  /\ StepwiseImpl(EnvNext, SysNext)
                  /\ (Le /\ [] EnvNext) => Ls
    <2>1. sigma, 0 |= /\ Ie
                      /\ \E p, q:  IeP(p, q)
        BY <1>4, <1>5
    <2>2. sigma, 0 |=
            \/ ~ \E p, q:  IeP(p, q)
            \/ /\ Is
               /\ \/ ~ Ie
                  \/ /\ StepwiseImpl(EnvNext, SysNext)
                     /\ ([]EnvNext /\ Le) => Ls
        <3>1. sigma, 0 |= RawPhi
            OBVIOUS  (* BY RawPhiImpliesPhiStep11!assumption *)
        <3> QED
            BY <3>1 DEF RawPhi, RawWhilePlus
    <2> QED
        BY <2>2, <2>1
<1>7. sigma, 0 |= Ls
    <2>1. sigma, 0 |= Le /\ [] EnvNext
        BY <1>4 DEF EnvNext
    <2> QED
        BY <1>6, <2>1
<1>8. sigma, 0 |= [][Ns]_v
    <2>1. sigma, 0 |= [] EnvNext
        BY <1>4 DEF EnvNext
    <2>2. sigma, 0 |= StepwiseImpl(EnvNext, SysNext)
        BY <1>6
    <2>3. sigma, 0 |= [] SysNext
        BY <2>1, <2>2, AlwaysEnvNextAndStepwiseImpl
    <2> QED
        BY <2>3 DEF SysNext
<1>9. sigma, 0 |= G
    <2>1. sigma, 0 |= Is /\ [][Ns]_v /\ Ls
        BY <1>6, <1>7, <1>8
    <2> QED
        BY <2>1 DEF G
<1> QED
    BY <1>9, <1>2  (* => goal of <1>3 *)


(* If the first (n - 1) steps of a behavior sigma satisfy the assumption
Ie /\ [][Ne]_v, and (causal) stepwise implication holds of sigma,
then the first n steps of sigma satisfy the guarantee Is /\ [][Ns]_v.

Note that such any TLA+ safety property (like Ie /\ [][Ne]_v) is
stutter-extensible [4], so it suffices to talk about the first (n - 1) steps,
as opposed of the first n states. The n-th state matters only for the last
step. The property Ie /\ [][Ne]_v can be satisfied by any n-th state,
by stuttering forever.
*)
LEMMA TakeOneMoreStep ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        NEW n \in Nat,
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        /\ \A a, b, c, d:  NeP(a, b, c, d)
        /\ \A a, b, c, d:  NsP(a, b, c, d),
        LET
            v == << x, y >>
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
        IN
            /\ PrefixSat(sigma, n, [][Ne]_v)
            /\ sigma, 0 |= StepwiseImpl(EnvNext, SysNext)
    PROVE
        LET
            v == << x, y >>
            Ns == NsP(x, y, x', y')
        IN
            \A r \in 0..(n - 1):
                << sigma[r], sigma[r + 1] >>[[ [Ns]_v ]]
PROOF
<1> DEFINE
    v == << x, y >>
    Is == IsP(x, y)
    Ie == IeP(x, y)
    Ne == NeP(x, y, x', y')
    Ns == NsP(x, y, x', y')
    EnvNext == [Ne]_v
    SysNext == [Ns]_v
    PlusOne == Earlier(EnvNext) => SysNext)
(* Behavior sigma's first (n - 1) steps of sigma satisfy EnvNext. *)
<1>1. ASSUME NEW k \in 0..(n - 2)
      PROVE << sigma[k], sigma[k + 1] >>[[ EnvNext ]]
    <2>1. PrefixSat(sigma, n, Ie /\ [][Ne]_v)
        OBVIOUS  (* BY TakeOneMoreStep!assumption *)
    <2>2. PICK tau:  /\ IsABehavior(tau)
                     /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                     /\ tau |= Ie /\ [][Ne]_v
        BY <2>1 DEF PrefixSat
    <2>3. ASSUME NEW i \in Nat
          PROVE << tau[i], tau[i + 1] >>[[ [Ne]_v ]]
        BY <2>2, Semantics
    <2>4. << tau[k], tau[k + 1] >> = << sigma[k], sigma[k + 1] >>
        <3>1. /\ k \in 0..(n - 1)
              /\ (k + 1) \in 0..(n - 1)
            BY <1>1
        <3> QED
            BY <3>1, <2>2
    <2> QED
        <3>1. << tau[k], tau[k + 1] >>[[ EnvNext ]]
            BY <1>1 DEF EnvNext
        <3> QED
            BY <3>1, <2>4
(* Convert to a statement that uses Earlier. *)
<1>2. ASSUME NEW r \in 0..(n - 1)
      PROVE sigma, r |= Earlier(EnvNext)
    <2>1. SUFFICES ASSUME NEW k \in 0..(r - 1)
                   PROVE << sigma[k], sigma[k + 1] >>[[ EnvNext ]]
        BY DEF Earlier
    <2>2. k \in 0..(n - 2)
        <3>1. (k \in Nat) /\ (r \in Nat)
            BY <2>1, <1>2
        <3>2. (k <= r - 1) /\ (r <= n - 1)
            BY <2>1, <1>2
        <3>3. k <= n - 2
            BY <3>1, <3>2
        <3> QED
            BY <3>1, <3>3
    <2> QED
        BY <2>2, <1>1
(* Plus one step for SysNext. *)
<1>3. ASSUME NEW r \in Nat,
             sigma, r |= Earlier(EnvNext)
      PROVE << sigma[r], sigma[r + 1] >>[[ SysNext ]]
    <2>1. sigma, 0 |= [] PlusOne
        BY DEF StepwiseImpl, PlusOne
            (* and TakeOneMoreStep!assumption *)
    <2>2. \A i \in Nat:  sigma, i |= PlusOne
        BY <2>1, Semantics
    <2>3. sigma, r |= PlusOne
        BY <2>2, <1>3
    <2>4. \/ ~ sigma, r |= Earlier(EnvNext)
          \/ sigma, r |= SysNext
        BY <2>3, Semantics DEF PlusOne
    <2>5. sigma, r |= SysNext
        BY <2>4, <1>3
    <2> QED
        BY <2>5, Semantics
<1>4. ASSUME NEW r \in 0..(n - 1)
      PROVE << sigma[r], sigma[r + 1] >>[[ SysNext ]]
    <2>1. sigma, r |= Earlier(EnvNext)
        BY <1>2, <1>4
    <2>2. r \in Nat:
        BY <1>4
    <2> QED
        BY <2>1, <2>2
<1>QED
    BY <1>4 DEF SysNext


PROPOSITION RawPhiImpliesPhiStep12 ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        CONSTANT IeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
            RawPhi == RawWhilePlus(
                IeP, Ie, Is,
                EnvNext, SysNext, Le, Ls)
        IN
            /\ IsMachineClosed(Ie /\ [][Ne]_v, Le)
            /\ IsMachineClosed(Is /\ [][Ns]_v, Ls)
            /\ sigma, 0 |= RawPhi
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
        IN
            \A n \in Nat:
                PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
PROOF
<1> DEFINE
    v == << x, y >>
    Is == IsP(x, y)
    Ie == IeP(x, y)
    Ne == NeP(x, y, x', y')
    Ns == NsP(x, y, x', y')
    A == Ie /\ [][Ne]_v /\ Le
    G == Is /\ [][Ns]_v /\ Ls
    ClA == Cl(A)
    ClG == Cl(G)
    EnvNext == [Ne]_v
    SysNext == [Ns]_v
    RawPhi == RawWhilePlus(
        IeP, Ie, Is,
        EnvNext, SysNext, Le, Ls)
<1>4. /\ ClA <=> (Ie /\ [][Ne]_v)
      /\ ClG <=> (Is /\ [][Ns]_v)
    BY DEF ClA, ClG, A, G, IsMachineClosed
        (* and RawPhiImpliesPhiStep12!assumption *)
<1>8. sigma, 0 |=
        \/ ~ \E p, q:  IeP(p, q)  (* unsatisfiable assumption ? *)
        \/ /\ Is
           /\ \/ ~ Ie
              \/ /\ StepwiseImpl(EnvNext, SysNext)
                 /\ ([]EnvNext /\ Le) => Ls
    <2>1. sigma, 0 |= RawPhi
        OBVIOUS  (* BY RawPhiImpliesPhiStep12!assumption *)
    <2> QED
        BY <2>1 DEF RawPhi, RawWhilePlus
<1>1. SUFFICES ASSUME NEW n \in Nat
               PROVE PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
    OBVIOUS
<1>2. SUFFICES PrefixSat(sigma, n, ClA) => PrefixSat(sigma, n + 1, ClG)
    <2>1. IsABehavior(sigma)
        OBVIOUS  (* BY RawPhiImpliesPhiStep12!assumption *)
    <2>2. IsTemporalLevel(A)  (* META *)
        BY DEF A, Ie, Ne, IsTemporalLevel
    <2>3. IsTemporalLevel(G)  (* META *)
        BY DEF G, Is, Ns, IsTemporalLevel
    <2>4. PrefixSat(sigma, n, A) <=> PrefixSat(sigma, n, ClA)
        BY <1>1, <2>1, <2>2, PrefixSatForClosure
    <2>5. PrefixSat(sigma, n + 1, G) <=> PrefixSat(sigma, n + 1, ClG)
        BY <1>1, <2>1, <2>3, PrefixSatForClosure
    <2> QED
        BY <1>2, <2>4, <2>5
<1>3. SUFFICES ASSUME PrefixSat(sigma, n, ClA)
               PROVE PrefixSat(sigma, n + 1, ClG)
    OBVIOUS
<1>5. PrefixSat(sigma, n, Ie /\ [][Ne]_v)
    BY <1>3, <1>4
<1>6. SUFFICES PrefixSat(sigma, n + 1, Is /\ [][Ns]_v)
    BY <1>4
(* First we handle the initial conditions. *)
<1>7. \E p, q  IeP(p, q)  (* IeP is satisfiable, so A is satisfiable. *)
    <2>1. PICK tau:  /\ IsABehavior(tau)
                     /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                     /\ tau |= Ie /\ [][Ne]_v
        BY <1>5 DEF PrefixSat
    <2>2. tau |= Ie
        BY <2>1
    <2>3. tau |= IeP(x, y)
        BY <2>2 DEF Ie
    <2> QED
        BY <2>3, Semantics
<1>12. sigma, 0 |=
        /\ Is
        /\ \/ ~ Ie
           \/ /\ StepwiseImpl(EnvNext, SysNext)
              /\ ([]EnvNext /\ Le) => Ls
    BY <1>8, <1>7
<1>9. ASSUME n = 0
      PROVE PrefixSat(sigma, n + 1, Is /\ [][Ns]_v)
    (* In this case satisfiability of the assumption suffices to *)
    (* prove that the consequent holds. *)
    <2>1. SUFFICES PrefixSat(sigma, 1, Is /\ [][Ns]_v)
        BY <1>9
    <2>2. SUFFICES \E tau:  /\ IsABehavior(tau)
                            /\ tau[0] = sigma[0]
                            /\ tau |= Is /\ [][Ns]_v
        BY DEF PrefixSat
    <2>3. sigma[0] |= Is
        BY <1>12
    <2> DEFINE tau == [i \in Nat |-> sigma[0]]
    <2>4. IsAState(sigma[0])
        <3>1. IsABehavior(sigma)
            OBVIOUS  (* BY RawPhiImpliesPhiStep12!assumption *)
        <3> QED
            BY <3>1 DEF IsABehavior
    <2>5. IsABehavior(tau)
        BY <2>4 DEF tau, IsABehavior
    <2>6. tau[0] = sigma[0]
        BY DEF tau
    <2>7. tau[0] |= Is
        BY <2>3, <2>6
    <2>8. tau |= [][Ns]_v
        <3>1. ASSUME i \in Nat  (* all tau steps are stuttering *)
              PROVE tau[i + 1] = tau[i]
            BY DEF tau
        <3>2. ASSUME i \in Nat, tau[i + 1] = tau[i]
              PROVE << tau[i], tau[i + 1] >>[[ [Ns]_v ]]
            BY Semantics  (* stuttering step *)
        <3> QED
            BY <3>1, <3>2, Semantics
    <2>9. tau |= Is /\ [][Ns]_v
        BY <2>7, <2>8, Semantics
    <2> QED
        BY <2>5, <2>6, <2>9  (* WITNESS tau for goal of <2>2 *)
<1>10. SUFFICES ASSUME n > 0
                PROVE PrefixSat(sigma, n + 1, Is /\ [][Ns]_v)
        (* current goal from <1>6 *)
    <2>1. (n = 0) \/ (n > 0)
        BY <1>1
    <2>2. CASE n = 0
        BY <1>9
    <2>3. CASE n > 0
        BY <1>10
    <2> QED
        BY <2>1, <2>2, <2>3
<1>15. (n \in Nat) /\ (n > 0)
    BY <1>1, <1>10
<1>11. sigma, 0 |= Ie
    <2>1. PICK tau:  /\ IsABehavior(tau)
                     /\ \A i \in 0..(n - 1):  tau[i] = sigma[i]
                     /\ tau |= Ie /\ [][Ne]_v
        BY <1>5 DEF PrefixSat
    <2>2. tau[0] |= Ie
        BY <2>1
    <2>3. sigma[0] = tau[0]
        BY <2>1, <1>15
    <2>4. sigma[0] |= Ie
        BY <2>2, <2>3
    <2> QED
        <3>1. IsStateLevel(Ie)
            BY DEF Ie
        <3> QED
            BY <2>4, <3>1, Semantics
<1>13. sigma, 0 |= /\ Is
                   /\ StepwiseImpl(EnvNext, SysNext)
            (* We omit the liveness conjunct, because irrelevant to this
            part of the proof, which concerns stepwise implication,
            thus only safety. *)
    BY <1>12, <1>11
(* Done with initial conditions. We address below the stepwise implication. *)
<1>14. SUFFICES \E w:  /\ IsABehavior(w)
                       /\ \A i \in 0..n:  w[i] = sigma[i]
                       /\ w |= Is /\ [][Ns]_v
      (* current goal from <1>10 *)
  <2>1. \E w:  /\ IsABehavior(w)
               /\ \A i \in 0..((n + 1) - 1):  w[i] = sigma[i]
               /\ w |= Is /\ [][Ns]_v
      BY <1>14
  <2> QED
      BY <2>1 DEF PrefixSat
<1> DEFINE eta == StutterAfter(sigma, n)  (* Infinitely stuttering tail. *)
(* prove that eta is the WITNESS w *)
<1>16 eta |= [][Ns]_v
    (* The first (n - 1) steps of eta satisfy the action [Ns]_v. *)
    <2>1. ASSUME NEW k \in 0..(n - 1)
          PROVE << eta[k], eta[k + 1] >>[[ SysNext ]]
        <3>1. << sigma[k], sigma[k + 1] >>[[ SysNext ]]
            <4>1. PrefixSat(sigma, n, [][Ne]_v)
                BY <1>5 DEF PrefixSat
            <4>2. sigma, 0 |= StepwiseImpl(EnvNext, SysNext)
                BY <1>13
            <4> QED
                BY <4>1, <4>2, TakeOneMoreStep
        <3>2. << eta[k], eta[k + 1] >> = << sigma[k], sigma[k + 1] >>
            <4>1. /\ k \in 0..n
                  /\ (k + 1) \in 0..n
                BY <2>1
            <4>2. \A i \in 0..n:  eta[i] = sigma[i]
                BY DEF eta, StutterAfter
            <4>3. /\ eta[k] = sigma[k]
                  /\ eta[k + 1] = sigma[k + 1]
                BY <4>1, <4>2
            <4> QED
                BY <4>3
        <3> QED
            BY <3>1, <3>2
    (* Steps of eta from the n-th onwards satisfy the action [Ns]_v.
    These are stuttering steps, so this step's proof has no dependencies. *)
    <2>2. ASSUME NEW k \in Nat, k >= n
          PROVE << eta[k], eta[k + 1] >>[[ SysNext ]]
        <3>1. << eta[k], eta[k + 1] >> = << sigma[n], sigma[n] >>
            BY <2>2, StutteringTail DEF eta
        <3>2. << sigma[n], sigma[n] >>[[ SysNext ]]
            <4>1. << sigma[n], sigma[n] >>[[ v' = v ]]
                BY DEF v
            <4> QED
                BY <4>1, Semantics DEF SysNext
        <3> QED
            BY <3>1, <3>2
    <2>3. ASSUME NEW k \in Nat
          PROVE << eta[k], eta[k + 1] >>[[ SysNext ]]
        BY <2>1, <2>2
    <2> QED
        BY <2>3, Semantics DEF SysNext
<1>17. eta |= Is
    <2>1. sigma, 0 |= Is
        BY <1>13
    <2>2. sigma[0] |= Is
        BY <2>1, Semantics DEF Is
    <2>3. eta[0] = sigma[0]
        <3>1. n > 0
            BY <1>10
        <3> QED
            BY <3>1 DEF eta, StutterAfter
    <2>4. eta[0] |= Is
        BY <2>2, <2>3
    <2> QED
        BY <2>4, Semantics DEF Is
<1>18. IsABehavior(eta)
    BY StutterAfterIsABehavior
<1>19. ASSUME NEW k \in 0..n
       PROVE eta[k] = sigma[k]
    <2>1. CASE k < n
        BY StutterAfterHasSamePrefix DEF eta
    <2>2. CASE k = n
        <3>1. eta[k] = sigma[n]
            BY StutteringTail
        <3> QED
            BY <3>1, <2>2
    <2>3. (k < n) \/ (k = n)
        BY <1>19
    <2> QED
        BY <2>1, <2>2, <2>3
<1> QED
    <2>1. /\ IsABehavior(eta)
          /\ \A i \in 0..n:  eta[i] = sigma[i]
          /\ eta |= Is /\ [][Ns]_v
        BY <1>18, <1>19, <1>17, <1>16
    <2> QED
        BY <2>1  (* goal from <1>14 *)


LEMMA RawPhiImpliesPhi ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma)
        CONSTANT IeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
            RawPhi == RawWhilePlus(
                IeP, Ie, Is,
                EnvNext, SysNext, Le, Ls)
        IN
            /\ IsMachineClosed(Ie /\ [][Ne]_v, Le)
            /\ IsMachineClosed(Is /\ [][Ns]_v, Ls)
            /\ sigma, 0 |= RawPhi
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
            Phi == A -+-> G
        IN
            sigma |= Phi
    PROOF
    <1> DEFINE
        v == << x, y >>
        Is == IsP(x, y)
        Ie == IeP(x, y)
        Ne == NeP(x, y, x', y')
        Ns == NsP(x, y, x', y')
        A == Ie /\ [][Ne]_v /\ Le
        G == Is /\ [][Ns]_v /\ Ls
        Phi == A -+-> G
        EnvNext == [Ne]_v
        SysNext == [Ns]_v
        RawPhi == RawWhilePlus(
            IeP, Ie, Is,
            EnvNext, SysNext, Le, Ls)
    <1> SUFFICES
            /\ sigma |= A => G
            /\ \A n \in Nat:  PrefixSat(sigma, n, A)
                                  => PrefixSat(sigma, n + 1, G)
        BY DEF -+->, PrefixPlusOne, A, G, Is, Ie, Ns, Ne
    <1>1. sigma |= A => G  (* The liveness part. *)
        BY RawPhiImpliesPhiStep11
    <1>2. \A n \in Nat:  (* The safety part. *)
            PrefixSat(sigma, n, A) => PrefixSat(sigma, n + 1, G)
        BY RawPhiImpliesPhiStep12
    <1> QED
        BY <1>1, <1>2

--------------------------------------------------------------------------------
(* The other direction of PhiEquivRawPhi. *)

LEMMA PhiImpliesRawPhi ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        CONSTANT IeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
            Phi == A -+-> G
        IN
            /\ IsMachineClosed(Ie /\ [][Ne]_v, Le)
            /\ IsMachineClosed(Is /\ [][Ns]_v, Ls)
            /\ sigma |= Phi
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
            RawPhi == RawWhilePlus(
                IeP, Ie, Is,
                EnvNext, SysNext, Le, Ls)
        IN
            sigma, 0 |= RawPhi
PROOF
<1> DEFINE
    v == << x, y >>
    Is == IsP(x, y)
    Ie == IeP(x, y)
    Ne == NeP(x, y, x', y')
    Ns == NsP(x, y, x', y')
    A == Ie /\ [][Ne]_v /\ Le
    G == Is /\ [][Ns]_v /\ Ls
    Phi == A -+-> G
    ClA == Cl(A)
    ClG == Cl(G)
    EnvNext == [Ne]_v
    SysNext == [Ns]_v
    RawPhi == RawWhilePlus(
        IeP, Ie, Is,
        EnvNext, SysNext, Le, Ls)
<1>9. /\ ClA <=> (Ie /\ [][Ne]_v)
      /\ ClG <=> (Is /\ [][Ns]_v)
    <2>1. Cl(Ie /\ [][Ne]_v /\ Le) <=> (Ie /\ [][Ne]_v)
        BY DEF IsMachineClosed, Ie, Ne, v
            (* and PhiImpliesRawPhi!assumption *)
    <2>2. Cl(A) <=> (Ie /\ [][Ne]_v)
        BY <2>1 DEF A
    <2>3. Cl(Is /\ [][Ns]_v /\ Ls) <=> (Is /\ [][Ns]_v)
        BY DEF IsMachineClosed, Is, Ns, v
            (* and PhiImpliesRawPhi!assumption *)
    <2>4. Cl(G) <=> (Is /\ [][Ns]_v)
        BY <2>3 DEF G
    <2> QED
        BY <2>2, <2>4 DEF ClA, ClG
<1>1. SUFFICES ASSUME sigma, 0 |= \E p, q:  IeP(p, q)
               PROVE sigma, 0 |= /\ Is
                                 /\ \/ ~ Ie
                                    \/ /\ StepwiseImpl(EnvNext, SysNext)
                                       /\ ([]EnvNext /\ Le) => Ls
    BY DEF RawWhilePlus
<1>2. sigma, 0 |= Is
    <2>1. A -+-> G
        OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
    <2>2. PrefixSat(sigma, 0, A) => PrefixSat(sigma, 1, G)
        BY <2>1 DEF -+->, PrefixPlusOne
    <2>3. PrefixSat(sigma, 0, A)
        <3>1. SUFFICES PrefixSat(sigma, 0, ClA)
            <4>1. IsTemporalLevel(A)
                BY DEF A, Ie, Ne
            <4>2. 0 \in Nat
                OBVIOUS
            <4>3. IsABehavior(sigma)
                OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
            <4> QED
                BY <4>1, <4>2, <4>3, PrefixSatForClosure
        <3>2. SUFFICES PrefixSat(sigma, 0, Ie /\ [][Ne]_v)
            BY DEF ClA, IsMachineClosed
                (* and PhiImpliesRawPhi!assumption *)
        <3>3. SUFFICES \E tau:  /\ IsABehavior(tau)
                                /\ tau |= Ie /\ [][Ne]_v
            BY DEF PrefixSat
        <3>4. PICK p, q:  IeP(p, q)
            BY <1>1
        <3>5. DEFINE
            state == [var \in VarNames |-> IF var = "x" THEN p ELSE q]
            eta == Stutter(state)
        <3>6. eta |= Ie
            <4>1. eta[0] = state
                BY DEF eta, Stutter
            <4>2. /\ state.x = p
                  /\ state.y = q
                BY DEF state
            <4>3. state[[ Ie ]]
                BY <4>2, Semantics DEF state, Ie
            <4> QED
                BY <4>3, <4>1, Semantics DEF eta
        <3>7. eta |= [][Ne]_v
            <4>1. SUFFICES ASSUME NEW i \in Nat
                           PROVE eta[i] = eta[i + 1]
                BY <4>1, Semantics
            <4> QED
                BY DEF eta, Stutter
        <3>8. eta |= Ie /\ [][Ne]_v
            BY <3>6, <3>7
        <3>9. IsABehavior(eta)
            <4>1. IsAState(state)
                BY DEF state, IsAState
            <4> QED
                BY DEF eta, Stutter, IsABehavior
        <3> QED
            BY <3>8, <3>9  (* goal from <3>3 *)
    <2>5. PICK tau:  /\ IsABehavior(tau)
                     /\ \A i \in 0..(1 - 1):  tau[i] = sigma[i]
                     /\ tau |= ClG
        <3>1. PrefixSat(sigma, 1, G)
            BY <2>2, <2>3
        <3>2. PrefixSat(sigma, 1, ClG)
            BY <3>1, PrefixSatForClosure DEF ClG
        <3> QED
            BY <3>2 DEF PrefixSat
    <2>6. tau[0] = sigma[0]
        BY <2>5
    <2>7. tau |= Is /\ [][Ns]_v
        BY <2>5 DEF IsMachineClosed, ClG, G
            (* and PhiImpliesRawPhi!assumption *)
    <2>8. tau[0] |= Is
        BY <2>7
    <2>9. sigma[0] |= Is
        BY <2>8, <2>7
    <2> QED
        BY <2>9, Semantics
<1>6. SUFFICES ASSUME sigma, 0 |= Ie
               PROVE sigma, 0 |= /\ StepwiseImpl(EnvNext, SysNext)
                                 /\ ([]EnvNext /\ Le) => Ls
        (* Previous goal from <1>1 *)
    BY <1>2, <1>6
<1>7. sigma, 0 |= StepwiseImpl(EnvNext, SysNext)  (* safety part *)
    <2>1. SUFFICES ASSUME NEW n \in Nat
                   PROVE sigma, n |= Earlier(EnvNext) => SysNext)
        BY Semantics DEF StepwiseImpl
    <2>2. SUFFICES ASSUME sigma, n |= Earlier(EnvNext)
                   PROVE sigma, n |= SysNext
        OBVIOUS
    <2> DEFINE eta == StutterAfter(sigma, n)
    <2>6. IsABehavior(eta)
        BY <2>1, StutterAfterIsABehavior DEF eta
    <2>3. eta |= [][Ne]_v
        <3>1. SUFFICES ASSUME NEW k \in Nat
                       PROVE << eta[k], eta[k + 1] >>[[ [Ne]_v ]]
            <4>1. IsATLAPlusFormula( [][Ne]_v )
                BY DEF Ne, v, IsATLAPlusFormula
            <4> QED
                BY <3>1, <4>1, Semantics
        <3>2. (n \in Nat) /\ (k \in Nat)
            BY <2>1, <3>1
        <3>3. IsABehavior(sigma)
            OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
        <3>4. CASE k < n
            <4>1. eta[k] = sigma[k]
                BY <3>2, <3>3, <3>4, StutterAfterHasSamePrefix DEF eta
            <4>2. eta[k + 1] = sigma[k + 1]
                <5>1. ((k + 1) \in Nat) /\ (n \in Nat)
                    BY <3>2
                <5>2. CASE (k + 1) < n
                    BY <5>1, <5>2, <3>3, StutterAfterHasSamePrefix DEF eta
                <5>3. CASE (k + 1) >= n
                    <6>1. eta[k + 1] = sigma[n]
                        BY <5>1, <5>3, <3>3, StutteringTail DEF eta
                    <6>2. (k + 1) = n
                        <7>1. (k < n) /\ (k + 1) >= n
                            BY <3>4, <5>3
                        <7>2. (k + 1) <= n
                            BY <7>1, <5>1
                        <7>3. /\ (k + 1 <= n) /\ (k + 1 >= n)
                              /\ ((k + 1) \in Nat) /\ (n \in Nat)
                            BY <7>1, <7>2, <5>1
                        <7> QED
                            BY <7>3
                    <6> QED
                        BY <6>1, <6>2
                <5> QED
                    BY <5>1, <5>2, <5>3
            <4>3. << eta[k], eta[k + 1] >> = << sigma[k], sigma[k + 1] >>
                BY <4>1, <4>2
            <4>4. << sigma[k], sigma[k + 1] >>[[ [Ne]_v ]]
                <5>1. sigma, n |= Earlier( [Ne]_v )
                    BY <2>2 DEF EnvNext
                <5>2. \A i \in 0..(n - 1):
                        << sigma[i], sigma[i + 1] >>[[ [Ne]_v ]]
                    BY DEF Earlier
                <5>3. k \in 0..(n - 1)
                    BY <3>1, <3>4
                <5> QED
                    BY <5>2, <5>3
            <4> QED
                BY <4>3, <4>4  (* goal from <3>1 *)
        <3>5. CASE k >= n
            <4>1. eta[k] = sigma[n]
                BY <3>2, <3>5, <3>3, StutteringTail DEF eta
            <4>2. eta[k + 1] = sigma[n]
                <5>1. (k + 1) >= n
                    BY <3>2, <3>5
                <5>2. ((k + 1) \in Nat) /\ (n \in Nat)
                    BY <3>2
                <5> QED
                    BY <5>2, <5>1, <3>3, StutteringTail DEF eta
            <4>3. eta[k] = eta[k + 1]  (* A stuttering step. *)
                BY <4>1, <4>2
            <4>4. << eta[k], eta[k + 1] >>[[v' = v]]
                BY <4>3, Semantics DEF v
            <4> QED
                BY <4>4, Semantics  (* goal from <3>1 *)
        <3> QED
            BY <3>2, <3>4, <3>5
    <2>4. PrefixSat(eta, n + 1, Ie /\ [][Ne]_v)
        <3>1. eta |= Ie /\ [][Ne]_v
            <4>1. IsATLAPlusFormula(Ie)
                BY DEF Ie, IsATLAPlusFormula
            <4>2. sigma[0] |= Ie
                BY <1>6, Semantics DEF Ie
            <4>3. eta[0] = sigma[0]
                BY StutterAfterInit DEF eta
            <4>4. eta[0] |= Ie
                BY <4>2, <4>3
            <4>5. eta |= Ie
                BY <4>1, <4>4, <2>6, Semantics DEF Ie
            <4> QED
                BY <4>5, <2>3
        <3>2. /\ IsABehavior(eta)
              /\ \A i \in 0..((n + 1) - 1):  eta[i] = eta[i]
              /\ eta |= Ie /\ [][Ne]_v
            BY <2>6, <3>1
        <3> QED
            BY <3>2 DEF PrefixSat
    <2>5. PrefixSat(sigma, n + 1, A)
        <3>1. (n + 1) \in Nat
            BY <2>1
        <3>2. PrefixSat(eta, n + 1, ClA)
            BY <2>4, <1>9
        <3>3. PrefixSat(eta, n + 1, A)
                (* Note that if Le is non-trivial, then eta may violate A,
                   so we could not have used eta directly as the witness
                   for PrefixSat(eta, n + 1, A),
                   only for PrefixSat(eta, n + 1, ClA). *)
            BY <3>1, <3>2, PrefixSatForClosure DEF ClA
        <3>4. Prefix(eta, n + 1) = Prefix(sigma, n + 1)
        <3> QED
            <4>1. IsABehavior(sigma) /\ IsABehavior(eta)
                BY <2>6  (* and PhiImpliesRawPhi!assumption *)
            <4>2. IsTemporalLevel(A)
                BY DEF A, Ie, Ne, v
            <4> QED
                BY <3>1, <3>3, <3>4, <4>1, <4>2, SamePrefixImpliesPrefixSatToo
    <2>7. PrefixSat(sigma, n + 2, [][Ns]_v)
        <3>1. ASSUME NEW r \in Nat,
                     PrefixSat(sigma, r, A)
              PROVE PrefixSat(sigma, r + 1, G)
            <4>1. sigma |= A -+-> G
                OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
            <4>2. sigma |= \A k \in Nat:
                    \/ ~ PrefixSat(sigma, k, A)
                    \/ PrefixSat(sigma, k + 1, G)
                BY <4>1, WhilePlusProperties
            <4> QED
                BY <4>2, <3>1
        <3>2. PrefixSat(sigma, n + 2, G)
            <4>1. (n + 1) \in Nat
                BY <2>1
            <4>2. PrefixSat(sigma, (n + 1) + 1, G)
                BY <4>1, <2>5, <3>1
            <4>3. (n + 1) + 1 = n + 2
                BY <2>1
            <4> QED
                BY <4>2, <4>3
        <3>3. PrefixSat(sigma, n + 2, ClG)
            <4>1. IsTemporalLevel(G)
                BY DEF G, Is, Ns, v
            <4>2. (n + 2) \in Nat
                BY <2>1
            <4>3. IsABehavior(sigma)
                OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
            <4> QED
                BY <4>1, <4>2, <4>3, PrefixSatForClosure DEF ClG
        <3> QED
            BY <3>3, <1>9
    <2>8. << sigma[n], sigma[n + 1] >>[[ [Ns]_v ]]
        <3>1. PICK tau:  /\ IsABehavior(tau)
                         /\ \A i \in 0..((n + 2) - 1):  tau[i] = sigma[i]
                         /\ tau |= [][Ns]_v
            BY <2>7 DEF PrefixSat
        <3>2. /\ n \in Nat
              /\ (n + 1) \in Nat
            BY <2>1
        <3>3. /\ tau[n] = sigma[n]
              /\ tau[n + 1] = sigma[n + 1]
            <4>1. \A i \in 0..(n + 1):  tau[i] = sigma[i]
                BY <2>1, <3>1
            <4> QED
                BY <4>1, <3>2
        <3>4. << tau[n], tau[n + 1] >>[[ [Ns]_v ]]
            BY <3>1, <3>2
        <3> QED
            BY <3>3, <3>4
    <2> QED
        <3>1. IsABehavior(sigma)
            OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
        <3>2. n \in Nat
            BY <2>1
        <3>3. sigma, n |= [Ns]_v
            BY <2>8, <3>1, <3>2, Semantics DEF Ns, v
        <3> QED
            BY <3>3 DEF SysNext  (* goal from <2>2 *)
<1>8. sigma, 0 |= ([]EnvNext /\ Le) => Ls  (* liveness part *)
    <2>1. SUFFICES ASSUME sigma, 0 |= Le /\ []EnvNext
                   PROVE sigma, 0 |= Ls
        BY Semantics
    <2>2. sigma, 0 |= A
        <3>1. sigma, 0 |= Ie /\ Le /\ []EnvNext
            BY <1>6, <2>1
        <3> QED
            BY <3>1 DEF A, EnvNext
    <2>3. sigma, 0 |= A => G
        <3>1. sigma |= A -+-> G
            OBVIOUS  (* BY PhiImpliesRawPhi!assumption *)
        <3>2. sigma |= A => G
            BY <3>1 DEF -+->
        <3>3. IsATLAPlusFormula(A => G)
            BY DEF A, G, Ie, Is, Ne, Ns, v
        <3> QED
            BY <3>2, <3>3, CommonModels
    <2>4. sigma, 0 |= G
        BY <2>2, <2>3
    <2> QED
        BY <2>4 DEF G
<1> QED
    BY <1>7, <1>8

--------------------------------------------------------------------------------
(* This theorem proves that the solvers synthesize open-system TLA+ specs,
whenever the pairs happen to be machine-closed, and Ns does not mention x'.
If Ns does mention x' then the property resulting from -+-> can be
unrealizable (unless \A u:  Ns(x, y, u, y') is not FALSE).
*)
THEOREM  PhiEquivRawPhi ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        CONSTANT IeP(_, _),  (* The suffix "P" stands for "parametric". *)
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,  (* thus TLA+ formulas *)
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
        IN
            /\ IsMachineClosed(Ie /\ [][Ne]_v, Le)
            /\ IsMachineClosed(Is /\ [][Ns]_v, Ls)
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
            Phi == A -+-> G
            EnvNext == [Ne]_v  (* RTLA+ but not TLA+ expression *)
            SysNext == [Ns]_v
            RawPhi == RawWhilePlus(
                IeP, Ie, Is,
                EnvNext, SysNext, Le, Ls)
        IN
            (sigma, 0 |= RawPhi)   <=>   (sigma |= Phi)
    PROOF
    BY RawPhiImpliesPhi, PhiImpliesRawPhi


--------------------------------------------------------------------------------
(* Machine-unclosed representations. *)


(* This theorem tells us how to convert -+-> to the
stepwise form. The only difference with PhiEquivRawPhi
is that A, G are not defined by machine-closed representations
(meaning a conjunction of a machine-closed pair of properties).

A, G may be defined by machine-unclosed representations.
So this theorem tells us that in general we have to first
compute a machine-closed representation, before converting
from -+-> to the stepwise form, which we do in order to
decide realizability via fixpoint computations.

In other words, this theorem differs from PhiEquivRawPhi in that
defined symbols have been replaced by declarations of symbols
together with axioms about their properties.
So those symbols were defined by machine-closed expressions,
whereas here they are only declared, and could be defined by
machine-unclosed expressions.

In implementation we need to compute the closure of properties,
so the closure needs to be expressible directly, without
using temporal quantification. For open-system specifications
where only finitely many relevant states, this rewriting
is always possible.
*)
THEOREM MachineUnclosedWhilePlus ==
    ASSUME
        VARIABLE x, VARIABLE y,
        NEW sigma,  (* META NEW *)
        IsABehavior(sigma),
        CONSTANT IeP(_, _),
        CONSTANT IsP(_, _),
        CONSTANT NeP(_, _, _, _),
        CONSTANT NsP(_, _, _, _),
        TEMPORAL Le, TEMPORAL Ls,
        TEMPORAL A, TEMPORAL G,
        /\ \A u, v:  IeP(u, v) \in BOOLEAN
        /\ \A u, v:  IsP(u, v) \in BOOLEAN
        /\ \A a, b, c, d:  NeP(a, b, c, d) \in BOOLEAN
        /\ \A a, b, c, d:  NsP(a, b, c, d) \in BOOLEAN,
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
        IN
            /\ A <=> (Ie /\ [][Ne]_v /\ Le)
            /\ IsMachineClosed(Ie /\ [][Ne]_v, Le)
            /\ G <=> (Is /\ [][Ns]_v /\ Ls)
            /\ IsMachineClosed(Is /\ [][Ns]_v, Ls)
    PROVE
        LET
            v == << x, y >>
            Is == IsP(x, y)
            Ie == IeP(x, y)
            Ne == NeP(x, y, x', y')
            Ns == NsP(x, y, x', y')
            A == Ie /\ [][Ne]_v /\ Le
            G == Is /\ [][Ns]_v /\ Ls
            EnvNext == [Ne]_v
            SysNext == [Ns]_v
            Phi == A -+-> G
            RawPhi == RawWhilePlus(
                IeP, Ie, Is, EnvNext, SysNext, Le, Ls)
        IN
            (sigma, 0 |= RawPhi)  <=>  (sigma |= Phi)
PROOF (* SKETCH *)
<1> DEFINE
    v == << x, y >>
    Is == IsP(x, y)
    Ie == IeP(x, y)
    Ne == NeP(x, y, x', y')
    Ns == NsP(x, y, x', y')
    A == Ie /\ [][Ne]_v /\ Le
    G == Is /\ [][Ns]_v /\ Ls
    P == Ie /\ [][Ne]_v /\ Le
    Q == IS /\ [][Ns]_v /\ Ls
<1>1. A -+-> G  <=>  P -+-> Q
    <2>1. A <=> P
        BY DEF A, P
    <2>2. G <=> Q
        BY DEF G, Q
    <2> QED
        BY <2>1, <2>2
<1> QED
    BY <1>1, PhiEquivRawPhi

--------------------------------------------------------------------------------
(*
Alternative proof structure
===========================

The proof can be structured differently by using the identity:

A -+-> G  <=>  /\ C(A) -+-> C(G)
               /\ A => G

The second conjunct is present in the definitions of both
of the operators -+-> and RawWhilePlus. Only the first
conjunct needs a lengthier proof, which reduces to

PrefixPlusOne(Cl(A), Cl(G))
    <=> \/ ~ \E p, q:  IeP(p, q)
        \/ /\ Is
           /\ IeP(x, y) => StepwiseImpl([Ne]_v, [Ns]_v)

where the actions and state predicates are those of the
machine-closed canonical forms, as in the proof above.

Jonsson and Tsay structure their proof in this way.
The module WhilePlusHalfTheorems follows this approach for
the operator WhilePlusHalf.
*)
--------------------------------------------------------------------------------
(* [4, Lemma B.1 on p.70] does not hold for the case that H_E is
unsatisfiable. Below is the analysis of that case. That case is covered
by the theorems above.
*)


(* The below proposition shows that:

~ |= ( ([]FALSE) -+-> ([]FALSE) )  <=>  [](Earlier(FALSE) => FALSE)
*)
PROPOSITION
    /\ |= TRUE <=> (([]FALSE) -+-> ([]FALSE))
    /\ raw|=  FALSE <=> [](Earlier(FALSE) => FALSE)
    (* "raw" stands for "raw TLA+ with past" *)
    PROOF
    <1>1. |= TRUE <=> (([]FALSE) -+-> ([]FALSE))
        <2>1. |= FALSE <=> []FALSE
            OBVIOUS
        <2>2. (FALSE -+-> FALSE) <=> (([]FALSE) -+-> ([]FALSE))
            BY <2>1
        <2>3. TRUE <=> (FALSE -+-> FALSE)
            BY PhiEquivRawPhi
        <2> QED
            BY <2>2, <2>3
    <1>2. raw|=  FALSE <=> [](Earlier(FALSE) => FALSE)
        <2>1. ASSUME NEW sigma, IsABehavior(sigma)
              PROVE (sigma, 0 |= [](Earlier(FALSE) => FALSE))
                <=> \A n \in Nat:  sigma, n |= Earlier(FALSE) => FALSE
            BY DEF []
        <2>2. ASSUME NEW sigma, IsABehavior(sigma)
              PROVE (sigma, 0 |= [](Earlier(FALSE) => FALSE))
                <=> \A n \in Nat:  sigma, n |= ~ Earlier(FALSE)
            BY <2>1
        <2>3. SUFFICES
                ASSUME NEW sigma, IsABehavior(sigma)
                PROVE \E n \in Nat:  sigma, n |= Earlier(FALSE)
            BY <2>2
        <2>4. sigma, 0 |= Earlier(FALSE)
            BY DEF Earlier
        <2> QED  (* goal from <2>3 *)
            BY <2>4
    <1> QED
        BY <1>1, <1>2
================================================================================
