function [ll] = log_likelihood_sharedCSP(modelfcn, nSp, logpjoint, parametermap, ...
    tvec, dataset, dosevals, dosemap, idMS, logresvec, stdev)
% log_likelihood_sharedCSP Set up stacked common CSP model and compute likelihoods. 
%   This function takes a dynamic model with Environment Specific
%   Parameters and Circuit Specific Parameters and returns the log
%   likelihoods for a multi environment model where the CSPs are shared and 
%   the ESPs are individual. 
%
%   The 4D array datasets is a nT x nMS x nIC x
%   nEnv array of datapoints correspointing to the timepoints of the length
%   nT vector nT. The residuals are computed using the dataset and a stack
%   of the modelfcn, where the CSPs are shared across the fucntion, but the
%   ESPs are unique to each function. 
%
% modelfcn is a function that takes as input a vector of timepoints tvec, 
% a vector of initial conditions y and a vector of parameters logp, and returns a
% matrix array of species trajectories at the time points in tvec. 
%
% logpjoint must be a vector with length (nEnv*qE + qC), where qE and qC are the
% number of (ESP+ESSP) and CSP (respectively) in the model given by the handle modelfcn.
% 
% idMS is the index (in the model) of the measured species, and acts as a
% map between the second dimension pf the dataset array and the model's
% simulated output. 
% 
% IC is a nSpecies x nIC x nEnv array, where nSpecies is the number of species in
% the model modelfcn (not just the measured species. This initial condition pattern 
% can change for the internal species between extracts. Indeed, this will
% be used to demonstrate that some of the internal species like RNAP and
% Ribosomoes can vary across the different extracts. 
% 
% logpjoint is a vector containing the esp, essp and the csp in the
% following order: 
% [esp1; essp1; esp2; essp2; ... ; espnEnv; esspnEnv; csp]
%
% parametermap is a map that specifies which parameters and species of the
% model in modelfcn are environment specific parameters and species, and
% which ones are csp, and what they correspond to in the vector logpjoint. 
% for now it is a 3 by 1 cell array (in the future it will be an element of a class)
% of vectors with the following interpretation: 
% parametermap = {espIX, essSpIX, cspIX}, where espIX and cspIX are the indices of the
% ESP and CSP (respectively) in the model parameter vector logp (with
% numel(logp) = numel(espIX)+ numel(cspIX))
% and esSpIX are the indices of the species in sp0 that correspond to the
% essp values in the logpjoint vector (essp1 to esspnEnv). These values are used in setting up
% the array of initial conditions during simulations
%
% dosemap is a nDosedSpecies by 1, and it specifies the indices of the
% initial condition vector that get modified by the doses. 
% 
% dosevals is a nDosedSpecies x nDoses array.  Each column gives a set of
% dosed initial condition values that get put into the initial conditions
% vector at the positions specified by the dosemap vector. 
% 
% OUTPUTS
% ll is the log likelihood for the full stacked model at all the inital
% conditions. 
%
% Vipul Singhal, 2017, Caltech

    
% setup the data normalizatin weights. These normalize the relative
% importance of different species. So for example, a 2000nM GFP signal is
% not 100 times more important than a 20nM mRNA signal. 
meanVals = mean(mean( mean(dataset, 1), 3), 4); % this is still a 1 x nMS sized array. (row vec)
wt = sum(meanVals)./meanVals; %higher mean = lower wt
relWt = wt/sum(wt); %relwt is a 1 x nMS vector. 
nEnv = size(dataset, 4);

espIX = parametermap{1};
esspIX = parametermap{2};
cspIX = parametermap{3};

if intersect(esspIX, dosemap)
    error('the dosemap and the esspIX arrays must not have any common elements')
end


nESP = length(espIX); % the ESP indices in the model (not in logpjoint)
nESSP = length(esspIX);% the Env specific species indices in the model (not in logpjoint)
nCSP = length(cspIX); % the CSP indices in the model (not in logpjoint)

cspindices = ((nESSP + nESP)*nEnv+1):length(logpjoint);
logpcsp = logpjoint(cspindices);

paramvec = zeros(nESP+nCSP, 1);
paramvec(cspIX) = logpcsp;
nDoses = size(dosevals, 2);
% icvec needs to be updated with the doses and the environment specific 
% values of the species initial concentrations. This happens in the nested
% loops below. 
icvec = zeros(nSp, 1); 

ll = 0;
% for each extract and each initial condition and each measured species, 
% compute the residuals and and update the log likelihood. 
for envid = 1:nEnv
    espindices = (envid-1)*(nESP+nESSP) + (1:nESP);
    logpesp = logpjoint(espindices);
    paramvec(espIX) = logpesp;
    esspindices = ((envid-1)*(nESP+nESSP) + nESP + 1):envid*(nESP+nESSP);
    
    % set the values of the initial condition vector to the parameters
    icvec(esspIX) = exp(logpjoint(esspindices)); 
    
    for doseID = 1:nDoses
        icvec(dosemap) = dosevals(:, doseID);


%             we compute the residuals for each measured species and weight
%             that by the relative size of the data trajectories of that
%             species. 

            % simulate the model 
            [~, simdata] = modelfcn(paramvec, icvec, tvec);
            
                
            
            % compute and reweight errors
            errmat = dataset(:, :, doseID, envid) - simdata(:, idMS);
            errnormalized = repmat(relWt, length(tvec), 1).*errmat;
            errvec = reshape(errnormalized, numel(errnormalized), 1);

%           compute log likelihood
            ll = ll + sum(logresvec(errvec, stdev)); 
            % this can be whatever function, as long as the result gets 
            % added (multiplication of probablity turns into addition after 
            % logarithms are taken.) I am not sure what happens when the 2
            % norm is replaced for the 1 norm or the infty norm. 
            
        
    end
end


end

