% This script implements calibration correction on a simple set of tetR
% mediated repression data. The characterization experiments used were the
% constitutive expression of the ptet promoter with the UTR1 ribosome
% binding site, and the deGFP fluorescent promoter. 
% The data being corrected was the repression of the GFP expression by a
% tetR expressing constitutive promoter. 
% We used a simple model with the equations 
%
% Characterization model:
% D_G + E <-> D_G:E -> D_G + E + G 
%
% Calibration model:
% D_T + E <-> D_T:E -> D_T + E + T
% D_G + E <-> D_G:E -> D_G + E + G
% 2 T <-> T2
% D_G + T2 <-> D_G:T2
%
% Data is stored in the file:
% 
% 
% Identifiability of this and similar models is studies in the directory:
%
% Vipul Singhal, 2017, California Institute of Technology
% 
%% Step 0: Initialize things

close all
clear all
clc

% add source code directory
addpath('/Users/vipulsinghal/Dropbox/Documents/vipul_repo/MCMC/Code_MCMC/src')

parallalflag = true;
nsteps = 2e5;
nW = 600;
stepsz = 3;
lb =-8; ub = 8;

%% Step 1: Collect data 
[~ , calibration_data, dosevals_calib] = import_ptetconstitutive;
[tvec, correction_data, dosevals_corr] = import_tetR_repression; 
% in this case both the tvec are the same. 

%%
figure;
% 1000*correction_data(1:13,:,:,3)

for i = 1:8 
    pdata(i) = plot(tvec, 1000*correction_data(:,1,i,3)) 
    hold on
end
legend(pdata, {'tetR 0nM', 'tetR 0.25nM', 'tetR 0.5nM', 'tetR 0.75nM',...
    'tetR 1nM', 'tetR 2nM', 'tetR 5nM', 'tetR 10nM' })

%% Step 2: Setup models

%%%%%%%%%% CALIBRATION STEP MODEL %%%%%%%%%%%%
model_calib = @(logp, sp0, tspan)...
    ode15s(@(t,sp) constitutive_gfp3(t,sp,logp), tspan, sp0);

% parameter map specifies which parameters in the model are ESP, CSP and
% ESSp (species). 
espix = 1;
cspix = 2:3;
esspix = 2; 
pmap_calib = {espix, esspix, cspix };
nEnv = size(calibration_data, 4);
nparam_calib = nEnv*(length(espix)+ length(esspix))+ length(cspix); 
%length of logpjoint

nSp_calib = 4; 
dosemap_calib = 1; % first species is the GFP dna, D_G, and is what gets dosed. 
idMS_calib = 4;
stdev = 1;
tightening = 2;


%%%%%%%%%%% TEST (CORRECTION STEP) MODEL %%%%%%%%%%%%%
model_corr = @(logp, sp0, tspan) ...
    ode15s(@(t,sp) tetR_repression(t,sp,logp), tspan, sp0);
espix = 1;
cspix = 2:9;
esspix = 2; 
nEnv = size(correction_data, 4);
nparam_corr = nEnv*(length(espix)+ length(esspix))+ length(cspix); %length of logpjoint
pmap_corr = {espix, esspix, cspix};
nSp_corr = 9; 
dosemap_corr = [6,1]; 
idMS_corr = 9;


%% Step 3: Define the likelihood and prior functions for both calibration and correction

lognormvec=@(res,sig) -(res./sig).^2 -log(sqrt(2*pi)).*sig;

loglike_calib = @(logpjoint) log_likelihood_sharedCSP(model_calib,...
    nSp_calib, logpjoint, pmap_calib, tvec(1:13),...
    1000*calibration_data(1:13,:,:,:), dosevals_calib,...
    dosemap_calib, idMS_calib,lognormvec, stdev/tightening);

loglike_corr = @(logpjoint) log_likelihood_sharedCSP(model_corr,...
    nSp_corr, logpjoint, pmap_corr, tvec(1:13),...
    1000*correction_data(1:13,:,:,:), dosevals_corr,...
    dosemap_corr, idMS_corr,lognormvec, stdev/tightening);

% the function loglike_corr1, defined later, is the one that actually 
% gets used for the initial correction. I will use
% the function above, loglike_corr later as a check that the csp that get
% estimated are indeed correct. 



logprior = @(logp) all(lb < logp) && all(logp < ub); 

%% Step 4: Perform Calibrations (MCMC parameter estimation)



mdpts = (lb+ub)/2;
width = abs(lb)+ abs(ub);
% rnsamp = 0.1*width*randn(nparam_calib,nW)-0.1*width/2; 
 lhsamp = width*(lhsdesign(nW, nparam_calib)-0.5); 




minit=bsxfun(@plus,mdpts,lhsamp');

% load('t015_calib_20171023_031136_5_MBP', 'm')
% minit = m(:,:, end);
% burn in
tic
[m, lPburnin] =gwmcmc_vse(minit,{logprior loglike_calib},nsteps, 'StepSize',stepsz , ...
    'ThinChain',20, 'Parallel', parallalflag);%
toc
datestring = datestr(now, 'yyyymmdd_HHMMSS');
% Run the actual MCMC simulation 
for i = 1:20
%     pause(10)
minit = m(:,:,end);
clear m

tic
[m, lp]=gwmcmc_vse(minit,{logprior loglike_calib},nsteps,'ThinChain',20,...
    'Parallel', parallalflag, 'StepSize', stepsz);%/(i+1)
toc
 eval(['save(''t015_calib_' datestring '_' num2str(i) '_MBP'');'])
  

end


%% Step 5: Perform the first correction step
% ie, fix the ESP and the ESSP in the correction model and estimate the CSP
% for the correction model 
% we do the estimation in environment number 3.

% sample the environment specific parameters from the calibration step (kc
% and P). m is the array that gets left from the previous step, and we can
% just sample from this. it has dimensions nparam_calib x nW x nsteps
% (where nsteps is the thinned # of markov chain steps). the first
% dimension is arranged as logpjoint was arranged in the calibration step. 

% in this case, we have three environments, 1 ESP, 1 ESSP, and 2 CSPs.
% thus, logpjoint (calib) is 8 parameters long. we fit the third
% environments parameters, and estimate the CSPs from this envrionments
% test data. 

eval(['load(''t015_calib_' datestring '_5_MBP'', ''m'');']) 
% datestring = '20171023_151627'
mstacked = m(:,:)';
medn = median(mstacked);
kc_calib = medn(5) %use medians
P_calib = medn(6) %use medians
exp(medn)
% kc_calib = -3; <--- wrong?
% P_calib = 2.5; <<< --- wrong?

% actually: [this is correct: VS Aug 10, 2018.] the medn(5) and medn(6)
% above should give these two. 
% kc_calib = -0.28209
% P_calib = 1.3714

loglike_corr1 = @(logp_corrcsp) log_likelihood_sharedCSP(model_corr,...
    nSp_corr, [kc_calib; P_calib; logp_corrcsp], pmap_corr, tvec(1:13), ...
    1000*correction_data(1:13,:,:,3), dosevals_corr, dosemap_corr, idMS_corr,...
    lognormvec, stdev/tightening);
nparam_corr1 = 8;


 %rnsamp = 0.1*width*randn(nparam_corr1,nW)-0.1*width; 
 lhsamp = width*(lhsdesign(nW, nparam_corr1)-0.5); 

minit=bsxfun(@plus,mdpts,lhsamp');
% burn in
tic
[m, lP] =gwmcmc_vse(minit,{logprior loglike_corr1},nsteps, 'StepSize',stepsz , ...
    'ThinChain',10, 'Parallel', parallalflag);%
toc

%%
corr_SD = cell(nIter,1);

load('t015_corr1_20171023_151627_6_MBP', 'm');
nIter = 20
stepsz = 0.95
nsteps=1e5;
% Run the actual MCMC simulation 
for i = 11:nIter
minit = m(:,:,end);
clear m

tic
[m, lp]=gwmcmc_vse(minit,{logprior loglike_corr1},nsteps,'ThinChain',10,...
    'Parallel', parallalflag, 'StepSize', stepsz);
toc
svstr = ['t015_corr1_' datestring '_' num2str(i) '_MBP'];
save(svstr);
% corr_SD{i} = svstr;


end

%% 
nW = 600;
legends = {'kfPT'    'krPT'  'kfPG'    'krPG'   'kfdim'    'krdim', 'kfrep', 'krrep'};
corr_SD = {    't015_corr1_20171023_151627_2_MBP'
    't015_corr1_20171023_151627_3_MBP'
    't015_corr1_20171023_151627_4_MBP'
    't015_corr1_20171023_151627_5_MBP'
    't015_corr1_20171023_151627_6_MBP'
    't015_corr1_20171023_151627_7_MBP'
    't015_corr1_20171023_151627_8_MBP'
    't015_corr1_20171023_151627_9_MBP'
    't015_corr1_20171023_151627_10_MBP'}

corr_SD2 = {    't015_corr1_20171023_151627_11_MBP'
    't015_corr1_20171023_151627_12_MBP'
    't015_corr1_20171023_151627_13_MBP'
    't015_corr1_20171023_151627_14_MBP'
    't015_corr1_20171023_151627_15_MBP'
    't015_corr1_20171023_151627_16_MBP'
    't015_corr1_20171023_151627_17_MBP'}
%%
close 
close 

mcat = catMC(corr_SD);

%

 plotChains(mcat(:,1:30:end, :), nW, legends );
 
 mcat2 = catMC(corr_SD2);

%

 plotChains(mcat2(:,1:30:end, :), nW, legends );
 
 %%
 figure
 ecornerplot_vse(mcat(:,:, 1:10:end),...
    'scatter', false, 'color',[.6 .35 .3], ...
    'names', legends);
%%
figure
 ecornerplot_vse(mcat(:,:, 1:10:end),...
    'scatter', true,'transparency',0.025, 'color',[.6 .35 .3], ...
    'names', legends);

% load handel.mat
% sound(y)

%% Step 6: Perform the second correction step. 
% ie, use the csp estimated in the previous step (step 5) to estimate the
% corrected behavior in the other models. 
