function p = plot_platereader_data(datafunc, indices, varargin) 
% This function creates an array of subplots of the data collected from a platereader
% Mostly for internal reporting. Not paper quality figures (Those are
% carefully hand created -> adobe illustrator edited. 
%
%%%%% INPUTS %%%%%
%
% 1) dataFunc is a function handle to a function that returns a matrix D of
% #timepoints x #wells and optionally returns a time vector T (specified in
% seconds) and a struct META, which contains information on the run (data,
% platereader, gain, calibration, temperature, ex/em, etc)
% 
% 2) indices is a cell array of vectors containing column indices of the
% data matrix D, and is used to specify the layout
% of the subplots as well. For example, a 3x4 cell array gives a 3x4
% subplot set, with the data for each subplot specified using the indices
% in the vector. 
%
% OPTIONAL INPUTS
% 3) legends: cell array of cell arrays of strings. The top level array is
% in the same dimensions as the number of subplots. legends can also be
% 1x1 cell containing a cell array of strings,
% in which case the legends are placed in the top right plot, or
% #row x 1 (contianing..), in which case the legends are only placed in the final column. 
%
% 4) titles is a cell array of strings of the same dimension as the
% subplots
%
% 5) xlabels is a  cell array of strings or just a string, in which case it
% is applied as a global label. 
% 
% 6) ylabels is a cell array of strings or just a string, in which case it
% is applied as a global label. 
% 
% 7) Includemeta is a flag which, when set to true, includes meta data in a
% left column of the figure
% 
% 8) savespec is a struct containing the boolean flag, filename, data formats,
% resolutions, figuresize and the location of where to save the file. 
% 
%
%%%%% OUTPUTS %%%%%
% a handle to the plot. and if saveflag is set, some figures saved to disk.



p = inputParser;
p.addParameter('legends',{},@iscell);
p.addParameter('titles', {}, @iscell) 
p.addParameter('xlabels',{},@(x) iscell(x) || ischar(x));
p.addParameter('ylabels',{} ,@(x) iscell(x) || ischar(x))
p.addParameter('includemeta',false ,@islogical)
p.addParameter('savespec',[] ,@isstruct)
p.addParameter('dir',[] ,@ischar) % directory data is saved in
p.parse(varargin{:});
p=p.Results;

legs = p.legends;
titles = p.titles;
xlab = p.xlabels;
ylab = p.ylabels;
metflag = p.includemeta;
savespec=p.savespec;

if isempty(p.dir)
datadir=pwd;
else
    datadir = p.dir;
end


% parse data into arrays using the indices
%
% !TODO a function that takes the alphanemeric well IDs used by the
% platereader and converts them into the indices. 

currdir=pwd;
cd(datadir)
[dataarray, time, metadata]=feval(datafunc);
cd(currdir);

nr=size(indices, 1);
nc=size(indices, 2);

fig = figure;


%create subplots
for r=1:nr
    for c=1:nc
        indexvector = indices{r,c};
        subplot(nr, nc, (r-1)*nc + c);
        
        % plot individual curves
        for i = 1:length(indexvector)
            h(i) = plot(time/3600, dataarray(:,indexvector(i)));
            hold on 
        end
        
        %add legends, titles, axis labels
        if ~isempty(legs)
            if numel(legs)==1 % global legends, place them in the top right plot
                if c == nc && r == 1
                    legend(h, legs{1}, 'FontSize', 11);
                end
            elseif size(legs, 1)==nr && size(legs, 2)==1
                if c==nc
                    legend(h, legs{r,1}, 'FontSize', 11);
                end
            else
                legend(h, legs{r,c}, 'FontSize', 11);
            end
        end
        if ~isempty(titles)
            title(titles(r,c), 'FontSize', 14);
        end
        if ~isempty(xlab) && iscell(xlab)
            if size(xlab, 1) == nr && size(xlab, 2) == nc
                xlabel(xlab{r,c}, 'FontSize', 14)
            else
                error(['xlabel does not have the right dimensions. It should' ,...
                    'either be a string, or a cell array of strings with dimensions',...
                    'the same as the indices vector.'])
            end
        end
        if ~isempty(ylab) && iscell(ylab)
            if size(ylab, 1) == nr && size(ylab, 2) == nc
                ylabel(ylab{r,c}, 'FontSize', 14);
            else
                error(['ylabel does not have the right dimensions. It should' ,...
                    'either be a string, or a cell array of strings with dimensions',...
                    'the same as the indices vector.'])
            end
        end
    end
end

if ischar(xlab)
    [ax, h] = suplabel(xlab)
    set(h,'FontSize',14)
end
if ischar(ylab)
    [ax, h] = suplabel(ylab, 'y')
    set(h,'FontSize',14)
end


% Include metadata support and saving support in a future iteration. 



end
