function mcmc_info = mcmc_info_constgfp3ii(modelObj)
% mcmc_info_constgfp3ii.m 
% This file contains information on how to set up a constitutive GFP
% experiment using the Simbiology model generated by the function 
% model_protein3. It is a simple example of the concurrent 
% parameter inference problem, which involves a single circuit (i.e., one 
% topology) being run in two different environments. Thus, the two models
% (one in each environment) should have different values estimated for the
% environment specific parameters, but the same value for the circuit
% specific parameters. In this sense, we have two geometries (type 
% help mcmc_info in the command window for more information on topologies 
% and geometries).that jointly estimate the circuit specific parameters 
% but estimate individual values for the environment specific parameters
% between the two geometries. The mcmc_info struct generated by this file
% is to be used with the following project files: 
% 
% proj_protein_constgfp3ii.m
% proj_protein_constgfp3ii_function.m
% proj_mcmc_tutorial_II.m
% 
% 
% The model to be used is set up by the file model_protein3.m in the 
% models_and_supporting_files directory, and has equations 
% 
% % dG + pol <-> dG_pol  (kf, kr) 
% dG_pol -> dG + pol + pG (kc) 
% 
% where dG is the GFP protein DNA, pol is a species signifying the
% transcription and translation machinery, and pG is the GFP protein output
% of the model. 
% 
% In the experiment, the initial dG concentration is set to different
% values (10, 30 and 60). I.e., the dosedNames in mcmc_info.model_info
% has a single species 'dG' in it, and the dosedVals is the row vector 
% [10 30 60]. For each dose, the corresponding pG trajectories are 
% measured. In the parameter inference problem, these simulated model 
% trajectories are compared to artificial 'data' trajectories obtained from 
% simulating the model. 
% 
% We fix kf to 5, and estimate the rest of the parameters. The circuit (i.e.
% part) specific parameter is kr, and this gets jointly estimated between the 
% two models (one model in each environment) and the environment specific 
% parameters are the initial concentration of pol, and the rate kc, and
% each environment has its own value for these. Thus, there are six 
% parameters in the master vector, (kf, kr, kc1, kc2, pol1, pol2), of which
% the first is fixed at a value of 5, and the remaining five are to be
% estimated. Also, the first two parameters (kf and kr) are to be shared 
% between the two topologies, while the remaining 4 are to be applied to
% individual topologies as defined by the paramMaps in the code below. 
% 
% Type help mcmc_info to read more about the mcmc_info struct array. 

% Copyright (c) 2018, Vipul Singhal, Caltech
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:

% The above copyright notice and this permission notice shall be included in all
% copies or substantial portions of the Software.

% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.

% User readable description of the circuit. Will be used in the log file generated
% from the MCMC inference procedure.
circuitInfo = ...
    [' D + pol <-> D__pol  (k_f, k_r \n'... )
    'D__pol -> D + pol + protien (kc)\n'...
    'single topology, two geometries.'];

rkfdG = 5; % nM-1s-1
rkrdG = 300; % s-1
rkcp1 = 0.012; %s-1
rkcp2 = 0.024; %s-1
cpol1 = 100; % nM
cpol2 = 200; % nM

activeNames = ...
    {'kfdG'
    'krdG'
    'kcp'
    'pol'};

estParams = {'krdG'
    'kcp1'
    'kcp2'
    'pol1'
    'pol2'};

masterVector = log([...
rkfdG 
rkrdG
rkcp1
rkcp2
cpol1
cpol2]);

% fixedParams vector
fixedParams = [1]; % just the rkfdG parameter is fixed

estParamsIx = setdiff((1:length(masterVector))', fixedParams);

% There are two geometries for this topology, and so two columns in the
% paramMaps field of the model_info struct. 
paramMap1 = [1 2 3 5]';
paramMap2 = [1 2 4 6]';
paramMap = [paramMap1 paramMap2];

% log transformed parameter ranges
paramRanges =  [masterVector(estParamsIx)-3 masterVector(estParamsIx)+3];

% The two geometries map to the first and second elements of the data_info
% struct array. 
dataIndices = [1 2];

%% next we define the dosing strategy.
dosedNames = {'dG'}; 
dosedVals = [10 30 60];

measuredSpecies = {{'pG'}};
msIx = 1; %

%% setup the MCMC simulation parameters
stdev = 10; % the standard deviation in the likelihood function.

tightening = 1; % default is 1. Type in help mcmc_info for more information 

nW = 100;  % number of walkers. good values: 200 - 400

stepsize = 1.1; % MCMC step size. try: 1.1 to 4 ish. DO NOT USE 1.

niter = 10;  % try: 2 - 50. Number of times to loop the MCMC. "help mcmc_info"

npoints = 2e4; % actual: 2e4 to 2e5 ish (or even 1e6 of the number of
%                        params is small)

thinning = 10;  % good values: 10 to 40 ish. 
% Number of steps to skip before recording positions of the walkers. 

%% pull all this together into an output struct.
% the mcmc info struct now is an array struct, the way struct should be used!

runsim_info = struct('stdev', {stdev}, ...
    'tightening', {tightening}, ...
    'nW', {nW}, ...
    'stepSize', {stepsize}, ...
    'nIter', {niter}, ...
    'nPoints', {npoints}, ...
    'thinning', {thinning}, ...
    'parallel', false);

% for now we simply make the model_info have just one model (topology).
% But the code will be written in a way such that multiple models can be used.

model_info = struct(...
    'circuitInfo',{circuitInfo},...
    'modelObj', {modelObj},... % array of model objects (different topologies)
    'modelName', {modelObj.name},...; % model names.
    'namesUnord', {activeNames}, ... % names of parameters per model, unordered.
    'paramMaps', {paramMap}, ... % paramMap: matrix mapping elements in the 
    ...                   % master vector to the parameters or species given 
    ...                   % by active names for a given topology. 
    ...                   % Type help mcmc_info for more information. 
    'dosedNames', {dosedNames},... % cell arrays of species. cell array corresponds
    ...                               % to a model.
    'dosedVals', {dosedVals},...  % matrices of dose vals
    'measuredSpecies', {measuredSpecies}, ... % cell array of cell arrays of
    ...                  % species names. the elements of the inner
    ...                  % cell array get summed.
    'measuredSpeciesIndex', {msIx},...
    'dataToMapTo', dataIndices); % each dataToMapTo property within an 
% element of the model_info array is a vector of length # of geometries.


%% IGNORE this for now. It has a subtle use, and we will update the
% documentation to describe how this is used in a future release. 

% Note to self (please ignore this). I tried the following:
% semanticGroups = {1, [2 4] [3 5]}; % cant do this, then the points never
% get differentiated at all. need some jitter I think. think about this....
semanticGroups = num2cell((1:length(estParams))'); 
%num2cell((1:5)'); 
%arrayfun(@num2str, 1:10, 'UniformOutput', false);


%% master parameter vector, param ranges,
master_info = struct(...
    'estNames', {estParams},...
    'masterVector', {masterVector},...
    'paramRanges', {paramRanges},...
    'fixedParams', {fixedParams},...
    'semanticGroups', {semanticGroups});


mcmc_info = struct('runsim_info', runsim_info, ...
    'model_info', model_info,...
    'master_info', master_info);

% Copyright (c) 2018, Vipul Singhal, Caltech
% Permission is hereby granted, free of charge, to any person obtaining a copy
% of this software and associated documentation files (the "Software"), to deal
% in the Software without restriction, including without limitation the rights
% to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
% copies of the Software, and to permit persons to whom the Software is
% furnished to do so, subject to the following conditions:

% The above copyright notice and this permission notice shall be included in all
% copies or substantial portions of the Software.

% THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
% IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
% FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
% AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
% LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
% OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
% SOFTWARE.
end