#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Solve inverse problem with cubic B-spline setup
// Input options and waves [dimensions in brackets]:
//	AmatrixInput				[m x ncoeff]		kernel matrix, describing measurement collection
//	SplineCoeffKnots		[ncoeff=n+d-1]	binning for coefficients of size distribution solution;
//														number of points must match y-dimension of Kmatrix
//	SplineControlptKnots	[n]			binning for control points of size distribution solution
//	SplineKnotWave		[n+2d]		wave of knot values needed to calculate Bsplines
//	ModelWaveOutput		[any]		wave to hold size distribution solution; any number/spacing
//	ModelDataBinning		[any]		binning for size distribution solution;
//												number of points must match ModelWaveOutput
//	Bvector					[m]			vector of measured signals (raw data)
//	Bvector_uncertainties	[m]			vector of uncertainty/error values associated with
//												measured signals (raw data)
//	SolnMethod - string to call solution computation algorithm
//
// Output is solution in "ModelWaveOutput"
function cSplineInv(AmatrixInput,SplineCoeffKnots,SplineControlptKnots,SplineKnotWave,ModelWaveOutput,ModelDataBinning,BvectorInput,Bvector_uncertainties,SolnMethod)
	wave AmatrixInput,SplineCoeffKnots,SplineControlptKnots,SplineKnotWave
	wave ModelWaveOutput,ModelDataBinning,BvectorInput,Bvector_uncertainties
	string SolnMethod
	
	// create working space
	string OldDf=GetDataFolder(1)
	NewDataFolder/O/S root:Packages
	NewDataFolder/O/S root:Packages:cSplineInv
	// create local copies of input waves
	duplicate/o AmatrixInput, Kmatrix
	duplicate/o SplineCoeffKnots, SplineCoeffBin
	duplicate/o SplineControlptKnots, SplineControlptBin
	duplicate/o SplineKnotWave, SplineKnotBin
	duplicate/o ModelDataBinning, ModelDataBin
	duplicate/o BvectorInput, Bvector
	duplicate/o Bvector_uncertainties, Uncertainties
	redimension/d Kmatrix,SplineCoeffKnots,SplineControlptKnots,SplineKnotWave
	redimension/d ModelDataBin,Bvector,Uncertainties
	
	variable n = dimsize(Kmatrix,1)
	make/o/n=(n) M_x; M_x=0
	
	strswitch(SolnMethod)
		case "NNLS":
			NNLS(Kmatrix,Bvector,M_x)
			wave M_x
			break
		case "TNNLS":
			IPG_TNNLS(Kmatrix,M_x,Bvector,Uncertainties,0.6,100)
			wave M_x
			break
		case "reg":
			Regularization(Kmatrix,M_x,Bvector,Uncertainties)
			wave M_x
			break
	endswitch
	
	cubicBspline(M_x,SplineControlptBin,SplineKnotBin,ModelDataBin,ModelWaveOutput)
	wave ModelWaveOutput
	SetDataFolder OldDf
end
//======================================================================

// Evaluate cubic spline interpolation, given coefficient values from inverse solution
// Input waves [dimensions in brackets]:
//	SplineCoeffWave		[ncoeff]			inverse solution coefficients
//	SplineControlptKnots	[nDias]			binning for control points of size distribution solution
//	SplineKnotWave		[(ncoeff+d+1)]	wave of knot values needed to calculate Bsplines
//	SplineOutputBin		[any]				binning for final size distribution solution; can be very fine
//	cSplineOutputWv		[any]				wave to hold size distribution solution;
//														number of points must match SplineOutputBin
// Output is solution in "cSplineOutputWv"
function cubicBspline(SplineCoeffWave,SplineControlptKnots,SplineKnotWave,SplineOutputBin,cSplineOutputWv)
	wave SplineCoeffWave,SplineControlptKnots,SplineKnotWave
	wave SplineOutputBin,cSplineOutputWv
	
	FindKnotPt(SplineKnotWave,SplineControlptKnots,SplineOutputBin)
	wave KnotPt, KnotPt_indx
	
	variable fpnts=numpnts(SplineOutputBin)
	variable controlpnts=numpnts(SplineControlptKnots)
	variable x0=SplineControlptKnots[0], xN=SplineControlptKnots[controlpnts-1]
	
	variable xi, xk,xk1,xk2,xk3,xk4
	variable xk_3,xk_2, xk_1
	variable cj_3,cj_2,cj_1,cj
	variable Bj_3,Bj_2,Bj_1,Bj
	variable dBj_3,dBj_2,dBj_1,dBj
	variable x0N,Cspline0N,dCspline0N
	variable Cspline_i
	variable knotindx
	variable i
	for(i=0;i<fpnts;i+=1)
		xi=SplineOutputBin[i]
		knotindx=KnotPt_indx[i]
		
		cj_3=SplineCoeffWave[knotindx-3]
		cj_2=SplineCoeffWave[knotindx-2]
		cj_1=SplineCoeffWave[knotindx-1]
		cj=SplineCoeffWave[knotindx]
		
		xk_3=SplineKnotWave[knotindx-3]
		xk_2=SplineKnotWave[knotindx-2]
		xk_1=SplineKnotWave[knotindx-1]
		xk=SplineKnotWave[knotindx]
		xk1=SplineKnotWave[knotindx+1]
		xk2=SplineKnotWave[knotindx+2]
		xk3=SplineKnotWave[knotindx+3]
		xk4=SplineKnotWave[knotindx+4]
		
		if (xi<x0 || xi>=xN)
			if (xi<x0)
				x0N=x0
			elseif (xi>=xN)
				x0N=xN
			endif
			
			Bj_3=BsplineC4(x0N,xk_3,xk_2,xk_1,xk,xk1)
			Bj_2=BsplineC3(x0N,xk_2,xk_1,xk,xk1,xk2)
			Bj_1=BsplineC2(x0N,xk_1,xk,xk1,xk2,xk3)
			Bj=BsplineC1(x0N,xk,xk1,xk2,xk3,xk4)
			Cspline0N=cj_3*Bj_3+cj_2*Bj_2+cj_1*Bj_1+cj*Bj
			
			dBj_3=dBsplineC4(x0N,xk_3,xk_2,xk_1,xk,xk1)
			dBj_2=dBsplineC3(x0N,xk_2,xk_1,xk,xk1,xk2)
			dBj_1=dBsplineC2(x0N,xk_1,xk,xk1,xk2,xk3)
			dBj=dBsplineC1(x0N,xk,xk1,xk2,xk3,xk4)
			dCspline0N=cj_3*dBj_3+cj_2*dBj_2+cj_1*dBj_1+cj*dBj
			
			Cspline_i=max((Cspline0N+dCspline0N*(xi-x0N)),0)
		else
			Bj_3=BsplineC4(xi,xk_3,xk_2,xk_1,xk,xk1)
			Bj_2=BsplineC3(xi,xk_2,xk_1,xk,xk1,xk2)
			Bj_1=BsplineC2(xi,xk_1,xk,xk1,xk2,xk3)
			Bj=BsplineC1(xi,xk,xk1,xk2,xk3,xk4)
			Cspline_i=cj_3*Bj_3+cj_2*Bj_2+cj_1*Bj_1+cj*Bj
		endif
		
		cSplineOutputWv[i]=max(Cspline_i,0)
		Cspline_i=0
	endfor
end
//======================================================================

// Find primary knot point to correspond to secondary (fine) knot point
function FindKnotPt(KnotPtWave,ControlPtWave,InterpFinePtWave)
	wave KnotPtWave,ControlPtWave,InterpFinePtWave
	
	duplicate/o InterpFinePtWave, KnotPt, KnotPt_indx
	variable p=numpnts(InterpFinePtWave)
	variable k=numpnts(KnotPtWave)
	
	variable c0knot=ControlPtWave[0]
	variable cNknot=ControlPtWave[numpnts(ControlPtWave)-1]
	findvalue/v=(c0knot) KnotPtWave
	variable c0knotindx=V_value
	findvalue/v=(cNknot) KnotPtWave
	variable cNknotindx=V_value
	
	variable j,i
	for(i=c0knotindx;i<cNknotindx;i+=1)
		for(j=0;j<p;j+=1)
			if (InterpFinePtWave[j]<KnotPtWave[c0knotindx])
				KnotPt[j]=KnotPtWave[c0knotindx]
				KnotPt_indx[j]=c0knotindx
			elseif (InterpFinePtWave[j]>=KnotPtWave[cNknotindx-1])
				KnotPt[j]=KnotPtWave[cNknotindx-1]
				KnotPt_indx[j]=cNknotindx-1
			elseif (InterpFinePtWave[j]>=KnotPtWave[i] && InterpFinePtWave[j]<KnotPtWave[i+1])
				KnotPt[j]=KnotPtWave[i]
				KnotPt_indx[j]=i
			endif
		endfor
	endfor
end
//======================================================================

// B-spline functional forms; first derivatives
function dBsplineC1(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable dB1=(3*(xi-xk)^2)/((xk3-xk)*(xk2-xk)*(xk1-xk))
	return dB1
end

function dBsplineC2(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable dB2=(-xk^2-2*xk*xk2+4*xk*xi+2*xk2*xi-3*xi^2)/((xk3-xk)*(xk2-xk)*(xk2-xk1))
	dB2+=(-xk*xk1-(xk+xk1)*xk3+2*(xk+xk1)*xi+2*xk3*xi-3*xi^2)/((xk3-xk)*(xk3-xk1)*(xk2-xk1))
	dB2+=(-xk1^2-2*xk1*xk4+4*xk1*xi+2*xk4*xi-3*xi^2)/((xk4-xk1)*(xk3-xk1)*(xk2-xk1))
	return dB2
end

function dBsplineC3(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable dB3=(xk3^2+2*xk3*xk-4*xk3*xi-2*xk*xi+3*xi^2)/((xk3-xk)*(xk3-xk1)*(xk3-xk2))
	dB3+=(xk3*xk4+(xk3+xk4)*xk1-2*(xk3+xk4)*xi-2*xk1*xi+3*xi^2)/((xk4-xk1)*(xk3-xk1)*(xk3-xk2))
	dB3+=(xk4^2+2*xk2*xk4-4*xk4*xi-2*xk2*xi+3*xi^2)/((xk4-xk1)*(xk4-xk2)*(xk3-xk2))
	return dB3
end

function dBsplineC4(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable dB4=(3*(xk4-xi)^2)/((xk4-xk1)*(xk4-xk2)*(xk4-xk3))
	return dB4
end
//======================================================================

// B-spline functional forms; second derivatives
function d2BsplineC1(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable d2B1=(6*(xi-xk))/((xk3-xk)*(xk2-xk)*(xk1-xk))
	return d2B1
end

function d2BsplineC2(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable d2B2=(4*xk+2*xk2-6*xi)/((xk3-xk)*(xk2-xk)*(xk2-xk1))
	d2B2+=(2*(xk+xk1)+2*xk3-6*xi)/((xk3-xk)*(xk3-xk1)*(xk2-xk1))
	d2B2+=(4*xk1+2*xk4-6*xi)/((xk4-xk1)*(xk3-xk1)*(xk2-xk1))
	return d2B2
end

function d2BsplineC3(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable d2B3=(-4*xk3-2*xk+6*xi)/((xk3-xk)*(xk3-xk1)*(xk3-xk2))
	d2B3+=(-2*(xk3+xk4)-2*xk1+6*xi)/((xk4-xk1)*(xk3-xk1)*(xk3-xk2))
	d2B3+=(-4*xk4-2*xk2+6*xi)/((xk4-xk1)*(xk4-xk2)*(xk3-xk2))
	return d2B3
end

function d2BsplineC4(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable d2B4=(6*(xk4-xi))/((xk4-xk1)*(xk4-xk2)*(xk4-xk3))
	return d2B4
end
//=============================================================================