#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Read datamatrix for single scan of DMA data to set measurement/instrument parameters
function DMAparamSet(datamatrix,DMAparams2specify)
	wave datamatrix,DMAparams2specify
	//-------------------------------------------------------------------
	// datamatrix columns:
	//	0:	scan index
	//	1:	Qaerosol [lpm]
	//	2:	Qmono [lpm]
	//	3:	Qsheath [lpm]
	//	4:	Qexcess [lpm]
	//	5:	HVset [volts]
	//	6:	raw count data [#]
	//	7:	tsample [s]
	//	8:	Tambient [K]
	//	9:	Pambient [Pa]
	//	10:	deltaT_detector [degC]
	//	11:	Qdetector [lpm]
	//-------------------------------------------------------------------
	//-------------------------------------------------------------------
	// DMAparams2specify columns:
	//	0:	inner radius, R1 [m]
	//	1:	outer radius, R2 [m]
	//	2:	column length, L [m]
	//	3:	equivalent pipe length, DMA inlet/outlet [m]
	//	4:	equivalent pipe length, sum of devices/components with DMA sample flow [m]
	//	5:	particle density [kg/m3]
	//-------------------------------------------------------------------
	
	// Create global variables from input datamatrix
	variable/g Qae,Qmono,Qsheath,Qex,tsample,Tamb,Pamb,deltaTdet,Qdet
	Qae=datamatrix[0][1]
	Qmono=datamatrix[0][2]
	Qsheath=datamatrix[0][3]
	Qex=datamatrix[0][4]
	tsample=datamatrix[0][7]
	Tamb=datamatrix[0][8]
	Pamb=datamatrix[0][9]
	deltaTdet=datamatrix[0][10]
	Qdet=datamatrix[0][11]
	
	// Load parameters/conditions into wave "DMAmeasParams"
	// calculations call from this wave to use these parameters; all in SI units
	makeWvDMAmeasParams()
	wave DMAmeasParams
	DMAmeasParams[%Qa_lpm]=(Qae+Qmono)/2				// lpm
	DMAmeasParams[%Qsh_lpm]=(Qsheath+Qex)/2			// lpm
	DMAmeasParams[%betaDMA]=QaDMA()/QshDMA()
	DMAmeasParams[%deltaDMA]=(Qae-Qmono)/(Qae+Qmono)
	DMAmeasParams[%tsample_s]=tsample						// s
	DMAmeasParams[%dTdet_C]=deltaTdet						// degC
	DMAmeasParams[%Qdet_lpm]=Qdet							// lpm
	DMAmeasParams[%Tamb_K]=Tamb							// K
	DMAmeasParams[%Pamb_Pa]=Pamb						// Pa
	DMAmeasParams[%particleDensity_kgm3]=DMAparams2specify[%particleDensity_kgm3]
	
	// Load instrument geometry parameters into wave "DMAgeomParams"
	// calculations call from this wave to use these parameters; all in SI units
	makeWvDMAgeomParams()
	wave DMAgeomParams
	DMAgeomParams[%R1dma_m]=DMAparams2specify[%R1dma_m]
	DMAgeomParams[%R2dma_m]=DMAparams2specify[%R2dma_m]
	DMAgeomParams[%Ldma_m]=DMAparams2specify[%Ldma_m]
	DMAgeomParams[%f_dma]=fDMA()
	DMAgeomParams[%G_dma]=Gdma()
	DMAgeomParams[%Leq_DMA_m]=DMAparams2specify[%Leq_DMA_m]
	DMAgeomParams[%Leq_other_m]=DMAparams2specify[%Leq_other_m]
end
//======================================================================

// Analyze a single scan of DMA data
// Input options and waves:
//	datamatrix - data recorded from measurement campaign
//	diffLossMatrix - equivalent pipe lengths and flow parameters for system components
//	DMAparams2specify - instrument geometry and Leq; particle density
//	calcresponsematrixParams - options for kernel matrix setup
//	CalcTFformula - text wave to call form of transfer function
//	InvMethod - text wave to call solution computation algorithm
//	Kernel_precalc - can input a previously calculated kernel matrix to save computation time;
//							otherwise, input empty set {0}
// Output:
//	dN/dlnDp [cm-3]	size distribution inverse solution(s) in "invsoln_xxx" waves
//	Dp [m]				corresponding particle diameters in "dpwave_xxx" waves
function DMAanalysis(datamatrix,diffLossMatrix,DMAparams2specify,calcresponsematrixParams,CalcTFformula,InvMethod,Kernel_precalc)
	wave datamatrix,diffLossMatrix,DMAparams2specify
	wave calcresponsematrixParams
	wave/t CalcTFformula,InvMethod
	wave Kernel_precalc
	
	string TFformula,InvSolnMethod
	TFformula=CalcTFformula[0]
	InvSolnMethod=InvMethod[0]
	
	// Set measurement/instrument parameters
	DMAparamSet(datamatrix,DMAparams2specify)
	wave DMAmeasParams, DMAgeomParams
	nvar Qae,Qmono,Qsheath,Qex,tsample,Qdet
	
	// Generate channel parameters
	variable/g numchan=DimSize(datamatrix,0)	// number of measured points in scan
	variable/g bbeta=betDMA()
	variable/g ddelta=deltaDMA()
	
	make/o/n=(numchan) ZstarChan, VstarChan, DstarChan
	VstarChan=datamatrix[p][5]						// read channel voltages from datamatrix;
	ZstarChan=ZpCalc_Vdma(VstarChan)			// convert to mobility and...
	DstarChan=DpCalc_Vdma(VstarChan)		//...diameter targeted by instrument
	variable/g ZmaxDMA=ZstarChan[0]
	variable/g ZminDMA=ZstarChan[numchan-1]
	
	// Set up kernel matrix for inversion	
	if (dimsize(Kernel_precalc,1)==0)
		calcresponsematrix(calcresponsematrixParams,TFformula)
	endif
	wave ChanZ_inv, ChanV_inv, ChanDp_inv
	wave Kernel_inv, lnDpbin_inv, lnDpbin_invmeas
	wave BsplineKnotWave_inv, BsplineCoeffKnots_j_inv
	wave BsplineCoeffKnots_inv, BsplineControlptKnots_inv
	nvar nChan,nDias,mKern,nKern
	
	// Set up wave of raw count data
	matrixop/o Counts=col(datamatrix,6)				// read counts from datamatrix; raw signal [#]
	// extrapolate for inversion if input "extrapYorN" set to 1
	duplicate/o ZstarChan, invZstarChan; invZstarChan=1/ZstarChan
	duplicate/o ChanZ_inv, invChanZ_inv; invChanZ_inv=1/ChanZ_inv
	extrapolateCounts_exp(Counts,invZstarChan,invChanZ_inv,2)
	wave ExtrCounts
	duplicate/o ExtrCounts, Counts_inv, Sigma_counts_inv
	duplicate/o ChanDp_inv, CountBin_inv
	// estimate uncertainty using Poisson statistics
	Counts_inv = Counts_inv < 1 ? 0 : Counts_inv	// if raw count [#] shows <1 particle, set to zero
	Sigma_counts_inv=1.96*sqrt(Counts_inv)
	
	// Account for flow rates and sample time in kernel matrix;
	// calculate sampled volume for each data point
	variable/g sampleQt_m3=Qae*tsample/60000	// should depend on aerosol sample inlet flow
	if (Qdet<Qmono)	//...unless low detector flow requires diverting part of classified outlet flow
		sampleQt_m3*=Qdet/Qmono	//...then account for sampling fraction of particle flux
	endif
	matrixop/o Kernel_invQt_cc=Kernel_inv*sampleQt_m3*1e6	// [cm3]
	
	// Cubic spline inversion
	make/o/n=(mKern) CountData_inv; CountData_inv=Counts_inv
	make/o/n=(mKern) Sigma_countdata_inv; Sigma_countdata_inv=Sigma_counts_inv
	make/o/n=(mKern,nKern) KernelMatrix_inv; KernelMatrix_inv=Kernel_invQt_cc
	invsoln_Bspline(KernelMatrix_inv,lnDpbin_inv,lnDpbin_invmeas,BsplineCoeffKnots_inv,BsplineControlptKnots_inv,BsplineKnotWave_inv,CountData_inv,Sigma_countdata_inv,InvSolnMethod)
	wave invsoln_coeffs	// inverse solution: coefficient values to use to evaluate size distribution...
	wave invsoln_controlpts,dpwave_controlptdistbin	//...at spline control points
	wave invsoln_model,dpwave_modeldistbin	//...at model knot points
	wave invsoln_meas,dpwave_measdistbin	//...at target Dp of measurement channels
								//...or any other Dp set; see function invsoln_fineEval(...)
	variable/g n_invModel=numpnts(lnDpbin_inv)
	variable/g mKernel_inv=dimsize(Kernel_inv,0)
	variable/g nKernel_inv=dimsize(Kernel_inv,1)
end
//======================================================================

// Analyze multiple scans of DMA data
// Input options and waves:
//	datamatrix - data recorded from measurement campaign
//	diffLossMatrix - equivalent pipe lengths and flow parameters for system components
//	DMAparams2specify - instrument geometry and Leq; particle density
//	calcresponsematrixParams - options for kernel matrix setup
//	CalcTFformula - text wave to call form of transfer function
//	InvMethod - text wave to call solution computation algorithm
//	Kernel_precalc - can input a previously calculated kernel matrix to save computation time;
//							otherwise, input empty set {0}
// Output:
//	invsolnMeas		[m x nScans]		dN/dlnDp [cm-3] evaluated over measurement range
//	dp_meas			[m x nScans]		Dp [m] corresponding to measurement range
//	invsolnCoeffs	[ncoeff x nScans]	inverse solution coefficients; use to evaluate
//													cubic spline solution over size range of interest
function DMAinversion(datamatrix,diffLossMatrix,DMAparams2specify,calcresponsematrixParams,CalcTFformula,InvMethod,Kernel_precalc)
	wave datamatrix,diffLossMatrix,DMAparams2specify
	wave calcresponsematrixParams
	wave/t CalcTFformula,InvMethod
	wave Kernel_precalc
	
	matrixop/o scanindxWv = col(datamatrix,0)
	variable nScans = scanindxWv[numpnts(scanindxWv)-1]-scanindxWv[0]+1
	variable ndatapnts = numpnts(scanindxWv)/(nScans)
	
	variable i=0
	variable scanindx = scanindxWv[0]+i
	extract/indx/o scanindxWv, indexforscan, scanindxWv==scanindx
	duplicate/o/r=[indexforscan[0],indexforscan[numpnts(indexforscan)-1]] datamatrix, scandata
	
	DMAanalysis(scandata,diffLossMatrix,DMAparams2specify,calcresponsematrixParams,CalcTFformula,InvMethod,Kernel_precalc)
	wave CountData_inv,CountBin_inv,Sigma_countdata_inv
	wave invsoln_meas,dpwave_measdistbin
	wave invsoln_coeffs,lnDpbin_inv,BsplineKnotWave_inv
	wave Kernel_inv
	
	duplicate/o CountData_inv, rawdata_inv; redimension/n=(-1,nScans) rawdata_inv
	duplicate/o CountBin_inv, dpmeas_inv; redimension/n=(-1,nScans) dpmeas_inv
	duplicate/o Sigma_countdata_inv, sigmadata_inv; redimension/n=(-1,nScans) sigmadata_inv
	duplicate/o invsoln_meas, invsolnMeas; redimension/n=(-1,nScans) invsolnMeas
	duplicate/o dpwave_measdistbin, dp_meas; redimension/n=(-1,nScans) dp_meas
	duplicate/o invsoln_coeffs, invsolnCoeffs; redimension/n=(-1,nScans) invsolnCoeffs
	duplicate/o lnDpbin_inv, invsolnSplineModelpts
	duplicate/o BsplineKnotWave_inv, invsolnSplineKnots
	
	for(i=1;i<nScans;i+=1)
		scanindx = scanindxWv[0]+i
		extract/indx/o scanindxWv, indexforscan, scanindxWv==scanindx
		duplicate/o/r=[indexforscan[0],indexforscan[numpnts(indexforscan)-1]] datamatrix, scandata
		
		DMAanalysis(scandata,diffLossMatrix,DMAparams2specify,calcresponsematrixParams,CalcTFformula,InvMethod,Kernel_inv)
		wave CountData_inv,CountBin_inv,Sigma_countdata_inv
		wave invsoln_meas,dpwave_measdistbin
		wave invsoln_coeffs
		
		rawdata_inv[][i] = CountData_inv[p]
		dpmeas_inv[][i] = CountBin_inv[p]
		sigmadata_inv[][i] = Sigma_countdata_inv[p]
		invsolnMeas[][i] = invsoln_meas[p]
		dp_meas[][i] = dpwave_measdistbin[p]
		invsolnCoeffs[][i] = invsoln_coeffs[p]
	endfor
end
//======================================================================

// Evaluate inverse solution(s) over size range of interest
// Input waves and variables:
//	SplineCoeffMatrix		[ncoeff x nSolns]			(i.e., "invsolnCoeffs")
//	SplineControlptKnots	[nDias x nSolns]			(i.e., "invsolnSplineModelpts")
//	SplineKnotWave		[(ncoeff+d+1) x nSolns]	(i.e., "invsolnSplineKnots")
//	diam0_m, diamN_m, numdps - size range and number of points to evaluate
//											for final solution representation
// Output:
//	invsolnFinal	[numdps x nSolns]		dN/dlnDp [cm-3] evaluated over size range of interest
//	dp_final		[numdps x nSolns]		Dp [m] corresponding to final solution range
function invsoln_fineEval(SplineCoeffMatrix,SplineControlptKnots,SplineKnotWave,diam0_m,diamN_m,numdps)
	wave SplineCoeffMatrix,SplineControlptKnots,SplineKnotWave
	variable diam0_m,diamN_m,numdps
	
	variable lndiam0=ln(diam0_m)
	variable lndiamN=ln(diamN_m)
	variable dlnDpWvSet=(lndiamN-lndiam0)/(numdps-1)
	make/o/n=(numdps) finalDistBinning, dpwave_finaldistbin
	finalDistBinning=lndiam0+dlnDpWvSet*x
	dpwave_finaldistbin=exp(finalDistBinning)
	
	variable i=0
	variable nSolns=dimsize(SplineCoeffMatrix,1)
	variable ncoeffs=dimsize(SplineCoeffMatrix,0)
	variable nptsfinaldist=numpnts(finalDistBinning)
	make/o/n=(ncoeffs) SplineCoeffWave_i
	make/o/n=(nptsfinaldist) invsoln_final_i
	duplicate/o invsoln_final_i, invsolnFinal; redimension/n=(-1,nSolns) invsolnFinal
	duplicate/o dpwave_finaldistbin, dp_final; redimension/n=(-1,nSolns) dp_final
	dp_final[p][]=dpwave_finaldistbin[p]
	
	for(i=0;i<nSolns;i+=1)
		SplineCoeffWave_i[]=SplineCoeffMatrix[p][i]
		invsoln_final_i=0
		cubicBspline(SplineCoeffWave_i,SplineControlptKnots,SplineKnotWave,finalDistBinning,invsoln_final_i)
		wave invsoln_final_i
		invsolnFinal[][i]=invsoln_final_i[p]
	endfor
end
//======================================================================


// Some data processing functions
function dataPreProcess(dataInput)
	wave dataInput
	
	matrixop/o scanindxWv = col(dataInput,0)
	variable nScans = scanindxWv[numpnts(scanindxWv)-1]-scanindxWv[0]+1
	
	variable i=0
	variable scanindx = scanindxWv[0]+i
	extract/indx/o scanindxWv, indexforscan, scanindxWv==scanindx
	duplicate/o/r=[indexforscan[0],indexforscan[numpnts(indexforscan)-1]] dataInput, scandata
	make/o QcolindxWave={1,2,3,4,11}
	avgFlowRates(scandata,QcolindxWave)
	wave scandata_Qavg
	duplicate/o scandata_Qavg, scandataMatrix
	
	for(i=1;i<nScans;i+=1)
		scanindx = scanindxWv[0]+i
		extract/indx/o scanindxWv, indexforscan, scanindxWv==scanindx
		duplicate/o/r=[indexforscan[0],indexforscan[numpnts(indexforscan)-1]] dataInput, scandata
		avgFlowRates(scandata,QcolindxWave)
		wave scandata_Qavg
		concatenate/np=0 {scandata_Qavg}, scandataMatrix
	endfor
	
	variable ndatapnts=dimsize(scandataMatrix,0)
	wave datamatrix
	redimension/n=(ndatapnts,-1) datamatrix
	datamatrix[][]=scandataMatrix[p][q]
end

// Take averages of flow rate values recorded throughout single scan
// input wave "QcolindxWv" should be wave of integers indicating indices of columns to average...
function avgFlowRates(datamatrix,QcolindxWv)	//...for "datamatrix" QcolindxWv={1,2,3,4,11}
	wave datamatrix,QcolindxWv
	
	duplicate/o datamatrix,scandata_Qavg
	variable nQcolumns=numpnts(QcolindxWv)
	variable i,Qcol_i
	for(i=0;i<nQcolumns;i+=1)
		Qcol_i=QcolindxWv[i]
		matrixop/o getQcol_i = col(datamatrix,(Qcol_i))
		wavestats/q getQcol_i
		scandata_Qavg[][Qcol_i] = V_avg
	endfor
end
//======================================================================