#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// DMA diffusive transfer function developed by Stolzenburg
// (1988 thesis, Eq 2.69; see also Stolzenburg and McMurry 2008, AST, Eq [13])
function omegaDMA_stolz(sigmap,zzz,bbeta,delta)
	variable sigmap,zzz,bbeta,delta
	
	variable e1,e2,e3,e4,dummy,om
	dummy=(zzz-(1-bbeta))/(2^0.5*sigmap)
	e1=dummy*erf(dummy)+exp(-dummy^2)/pi^0.5
	dummy=(zzz-(1+bbeta))/(2^0.5*sigmap)
	e2=dummy*erf(dummy)+exp(-dummy^2)/pi^0.5
	dummy=(zzz-(1-delta*bbeta))/(2^0.5*sigmap)
	e3=dummy*erf(dummy)+exp(-dummy^2)/pi^0.5
	dummy=(zzz-(1+delta*bbeta))/(2^0.5*sigmap)
	e4=dummy*erf(dummy)+exp(-dummy^2)/pi^0.5
	om=sigmap/(2^0.5*bbeta*(1-delta))*(e1+e2-e3-e4)
	
	return om
end

// Non-diffusive transfer function; Stolzenburg and McMurry 2008
function omegaDMA_nd(zzz,bbeta,delta)
	variable zzz,bbeta,delta
	
	variable om=(abs(zzz-(1+bbeta))+abs(zzz-(1-bbeta)))
	om+=(-abs(zzz-(1+bbeta*delta))-abs(zzz-(1-bbeta*delta)))
	om*=1/(2*bbeta*(1-delta))
	
	return om
end

// Diffusive transfer function; Gaussian approximation
// (with mean zzz=1 and standard deviation = sigmap)
function omegaDMA_diff(sigmap,zzz,bbeta,delta)
	variable sigmap,zzz,bbeta,delta
	
	variable om=bbeta*(1+delta)/sigmap/sqrt(2*pi)*exp(-((zzz-1)^2)/(2*sigmap^2))
	
	return om
end
//======================================================================

// Calculate value of transfer function for particle mobility Zp
// DMA set to voltage ChanV, with associated mobility ChanZ
function TransferFcn_DMA(Zp,ncharges,ChanZ,ChanV,bbeta,delta,TFformula)
	variable Zp,ncharges,ChanZ,ChanV,bbeta,delta
	string TFformula
	
	variable Ztil,Pe,sigmaStolz
	Ztil=Zp/ChanZ		// dimensionless mobility
	Pe=ncharges*echg()*ChanV*fDMA()/(kB()*TK())		// Peclet number for particle migration...
	sigmaStolz=sqrt((Gdma()*Ztil)/Pe)	// diffusion parameter			//...across electrode gap
	
	variable omega=omegaDMAcalc(sigmaStolz,Ztil,bbeta,delta,TFformula)
	if (omega<1e-5)		// limit for computational efficiency
		omega = 0
	endif
	
	return omega
end

// Call one of the "omegaDMA_xxx" functions to calculate omega value
function omegaDMAcalc(sigmap,zzz,bbeta,delta,TFformula)
	variable sigmap,zzz,bbeta,delta
	string TFformula
	
	variable om
	strswitch(TFformula)
		case "stolz":
			om=omegaDMA_stolz(sigmap,zzz,bbeta,delta)
			break
		case "nd":
			om=omegaDMA_nd(zzz,bbeta,delta)
			break
		case "diff":
			om=omegaDMA_diff(sigmap,zzz,bbeta,delta)
			break
	endswitch
	
	return om
end
//======================================================================

// Evaluate transmission probability for particle through DMA
function KernelFcn_DMA(Dpart,ChanZ,ChanV,bbeta,delta,chargemax,TFformula)
	variable Dpart,ChanZ,ChanV,bbeta,delta,chargemax
	string TFformula
	// Dpart is some measure of the particle diameter; need to make sure it is expressed as Dp [m]
	//	before passing to other functions (in case input is ln(Dp))
	// ChanZ is the channel-targeted mobility of the instrument; Z [m2/V/s] 
	variable Dp
	if(Dpart<0)
		Dp=exp(Dpart)
	else
		Dp=Dpart
	endif
	
	variable Zp=ZpCalc_DpDMA(Dp)
	variable Zp_i=Zp
	variable phi=0					// charging probability
	variable psi=0					// transfer function
	variable Gphipsi=0				// overall kernel value
	variable i=0
	variable echarges=0
	for (i=1;i<=chargemax;i+=1)	// set max number of charges to consider
		Zp_i=abs(Zp*i)
		echarges=-i
		phi=chargeprob(Dp,echarges)
		psi=TransferFcn_DMA(Zp_i,i,ChanZ,ChanV,bbeta,delta,TFformula)
		Gphipsi+=phi*psi
	endfor
	
	wave diffLossMatrix
	variable Leff,Q,eta; eta=1
	for (i=0;i<dimsize(diffLossMatrix,0);i+=1)
		Leff=diffLossMatrix[i][0]
		Q=diffLossMatrix[i][1]/60000
		eta*=eta_pen(Dp,Leff,Q)
	endfor
	eta*=eta_pen(Dp,Leff_total(),QaDMA())
	eta*=counteff_CPC3025(Dp)
	variable F=1*eta				// factor for penetration and other efficiencies
	
	variable kernel=Gphipsi*F	// *efficiencies
	if (kernel<1e-9)					// limit for kernel matrix; lessens burden on inversion algorithm
		kernel=0
	endif
	
	return kernel
end
//======================================================================


// Functions to call parameters and constants
//
// Flow rates
function QaDMA()
	wave DMAmeasParams
	return DMAmeasParams[%Qa_lpm]/60000			// m^3/s
end
function QshDMA()
	wave DMAmeasParams
	return DMAmeasParams[%Qsh_lpm]/60000			// m^3/s
end
function betDMA()
	variable bet=QaDMA()/QshDMA()						// flow ratio
	wave DMAmeasParams; DMAmeasParams[%betaDMA]=bet
	return bet
end
function deltaDMA()
	wave DMAmeasParams
	return DMAmeasParams[%deltaDMA]					// flow balance
end
//
// Instrument geometry
function R1dma()
	wave DMAgeomParams
	return DMAgeomParams[%R1dma_m]					// m
end
function R2dma()
	wave DMAgeomParams
	return DMAgeomParams[%R2dma_m]					// m
end
function Ldma()
	wave DMAgeomParams
	return DMAgeomParams[%Ldma_m]					// m
end
//
// Geometry parameters
function fDMA()
	variable r1=R1dma(), r2=R2dma()
	variable f=(r2-r1)/(r2*ln(r2/r1))
	wave DMAgeomParams; DMAgeomParams[%f_dma]=f
	return f
end
function Gdma()
	variable r1=R1dma(), r2=R2dma(), L=Ldma(), bet=betDMA(), delt=deltaDMA()
	variable G=Gc(r1,r2,L,bet,delt)
	wave DMAgeomParams; DMAgeomParams[%G_dma]=G
	return G
end

// Equivalent length values for calculating diffusional losses
function LeffDMA()
	wave DMAgeomParams
	return DMAgeomParams[%Leq_DMA_m]				// m
end
function Leff_other()
	wave DMAgeomParams
	return DMAgeomParams[%Leq_other_m]				// m
end
function Leff_total()
	variable Leff_tot=LeffDMA()+Leff_other()
	return Leff_tot												// m
end
//
// Sampling parameters
function tmeasDMA()
	wave DMAmeasParams
	return DMAmeasParams[%tsample_s]					// s
end
function dTdetDMA()
	wave DMAmeasParams
	return DMAmeasParams[%dTdet_C]					// degC
end
function QdetDMA()
	wave DMAmeasParams
	return DMAmeasParams[%Qdet_lpm]/60000			// m^3/s
end
//
// Ambient temperature and pressure conditions
function TK()
	wave DMAmeasParams
	return DMAmeasParams[%Tamb_K]					// K
end
function PPa()
	wave DMAmeasParams
	return DMAmeasParams[%Pamb_Pa]					// Pa
end

// Particle density
function rhop()
	wave DMAmeasParams
	return DMAmeasParams[%particleDensity_kgm3]	// kg/m^3
end
//
// Air properties and other constants
function MW()						// molecular weight of air 
	return 0.028						// kg/mol
end
function lambda()					// air mean free path; Seinfeld & Pandis p.399
	return 2*etagas()/(PPa()*(8*MW()*1.660538921e-27*1e3/(pi*TK()*kB()))^.5)	// m
end
function etagas()
	variable visc = 18.27e-6		// viscosity of air in reference temperature, t0v, [kg/m/s]
	variable suthc = 120			// Sutherland constant, value 120 for air (111 for N2 & 127 for O2)
	variable t0v = 291.5			// reference temperature for viscosity [K]
	visc = visc*(t0v+suthc)/(TK()+suthc)*(TK()/t0v)^1.5
	return visc						// viscosity of air [kg/m/s]
end
function rhogas()
	return PPa()/(TK()*287.058)	// density of air [kg/m3]
end
function nugas()
	return etagas()/rhogas()		// kinematic viscosity of air [m2/s]
end
function Rgas()
	return 8.3144621				// gas constant [J/mol/K]
end
function kB()
	return 1.3806488e-23			// Boltzmann constant [J/K]
end
function echg()
	return 1.602176565e-19		// elementary charge [C]
end
function ep0()
	return 8.854e-12				// dielectric constant [C^2/(N m^2)]
end
//======================================================================


// Functions to find G parameter; assuming laminar flow
function Gc(r1,r2,L,beta,delta)
	variable r1,r2,L,beta,delta
	
	variable gamma=(r1/r2)^2
	variable kappa=(L*r2)/(r2^2-r1^2)
	variable fcin=(beta*(1-delta))/(2*(1+beta))
	variable fcout=(2+beta-beta*delta)/(2*(1+beta))
	variable win=wfracc(fcin,gamma)			// find w values corresponding to...
	variable wout=wfracc(fcout,gamma)		// ...characteristic streamline at inlet & outlet
	
	variable Gc=4*(1+beta)^2*(1-gamma^0.5)/((1-gamma)^2)
	Gc*=(Ic(wout,gamma)-Ic(win,gamma)+(win-wout)/kappa^2)
	return Gc
end
function fracc(coefs,w)
	wave coefs
	variable w
	
	variable FCfd=(w*ln(w)+(1-w)+(1/2)*(1-w)^2*ln(coefs[1])/(1-coefs[1]))
	FCfd*=((1/2)*(1+coefs[1])*ln(coefs[1])+(1-coefs[1]))^(-1)
	return coefs[0]-FCfd
end
function wfracc(fc,gam)
	variable fc,gam
	
	make/d/o coefs={fc,gam}
	FindRoots/L=(gam)/H=1/q fracc, coefs
	return V_Root	
end
function Ic(w,gam)
	variable w,gam
	
	variable A=(-(1/2)*(1+gam)*ln(gam)-(1-gam))^(-1)
	variable ICfd=(-(1/2)*w^2*((1-gam)*ln(w)-(1-w)*ln(gam))^2)
	ICfd+=((1/2)*w^2*(1-gam)+(1/3)*w^3*ln(gam))*((1-gam)*ln(w)-(1-w)*(ln(gam)))
	ICfd+=(1/4)*(1-w^2)*(1-gam)^2+(5/18)*(1-w^3)*(1-gam)*ln(gam)+(1/12)*(1-w^4)*(ln(gam))^2
	ICfd*=(A)^2
	return ICfd
end
//======================================================================


// Probability of particle diameter dp [m] carrying n charges
// using Xcode fit coefficients for ss charge distribution for air ions and conductive particles at STP
function chargeprob(dp_m,echarges)
	variable dp_m, echarges
	
	wave XcodeS3
	variable ap = dp_m/2		// radius
	variable b0,b1,b2,b3,b4,b5,b6,b7,b8,b9,b10,b11,apmin,apmax,phi_charge
	
	matrixop/o echargesWv=col(XcodeS3,0)
	extract/indx/o echargesWv, echargesIndx, echargesWv==echarges
	variable i=echargesIndx[0]
	b0=XcodeS3[i][1]
	b1=XcodeS3[i][2]
	b2=XcodeS3[i][3]
	b3=XcodeS3[i][4]
	b4=XcodeS3[i][5]
	b5=XcodeS3[i][6]
	b6=XcodeS3[i][7]
	b7=XcodeS3[i][8]
	b8=XcodeS3[i][9]
	b9=XcodeS3[i][10]
	b10=XcodeS3[i][11]
	b11=XcodeS3[i][12]
	apmin=XcodeS3[i][15]
	apmax=XcodeS3[i][16]
	
	variable sumFitCoeffs=0
	if (ap<apmin || ap>apmax)
		phi_charge=0
	else
		sumFitCoeffs=(b0*(log(ap))^0+b1*(log(ap))^1+b2*(log(ap))^2+b3*(log(ap))^3)
		sumFitCoeffs+=(b4*(log(ap))^4+b5*(log(ap))^5+b6*(log(ap))^6+b7*(log(ap))^7)
		sumFitCoeffs+=(b8*(log(ap))^8+b9*(log(ap))^9+b10*(log(ap))^10+b11*(log(ap))^11)
		phi_charge=10^(sumFitCoeffs)
	endif
	
	return max(phi_charge,0)
end
//======================================================================

// Diffusional losses through DMA inlet/outlet; inside pipes/tubes, bipolar charger, dryer
// use equivalent pipe length in diffusional deposition formula for laminar flow (Baron and Willeke)
function eta_pen(dp_m,Leff,Q)
	variable dp_m,Leff,Q				// [m], [m], [m3/s]
	
	variable eta_pen,mu
	mu=pi*diffusion_coeff(dp_m)*Leff/Q
	mu=numtype(mu)==0 ? mu : 0
	if (mu>0.02)
		eta_pen=0.81905*exp(-3.6568*mu)+0.09753*exp(-22.305*mu)
		eta_pen+=0.0325*exp(-56.961*mu)+0.01544*exp(-107.62*mu)
	else
		eta_pen=1.0-2.5638*mu^(2/3)+1.2*mu+0.1767*mu^(4/3)
	endif
	
	return max(0,eta_pen)
end
function diffusion_coeff(dp)
	variable dp
	
	variable D		// diffusion coefficient
	if (dp > 0.3e-9)	// mobility diameter of a particle cannot be below 0.3 nm;
						// ...mass diameter would then be negative
		// based on mobility from Stokes-Millikan equation:
		variable Z = mobility(dp,1)	// mobility [m2/V/s]; assumed singly-charged
		D = Z*kB()*TK()/echg()
	else
		// free molecular regime:		// Li 2003, and Flagan note on nRDMA data analysis
		//	D=kT/f:	f=(2/3) rho dp^2 sqrt(2 pi kB TK / m) ( 1 + pi alpha / 8)
		variable alpha = 0.9
		variable f = (2/3)*rhogas()*dp^2*sqrt(2*pi*Rgas()*TK()/MW())*(1+pi*alpha/8)
		D = kB()*TK()/f
	endif
	
	return D
end

// Mobility calculation, according to Stokes-Millikan equation (Ehn et al. 2011, AST)
function mobility(dp,q)
	variable dp,q						// mobility diameter [m], number of charges in particle
	variable Kn = 2*lambda()/dp		// Knudsen number
	variable mass = pi/6*(dp-0.3e-9)^3*rhop()		// particle mass [kg]
	variable m_g = MW()*1.660538921e-27*1e3	// mass of gas molecule [kg]
	variable mobil = q*echg()/(3*pi*etagas())*(1+Kn*(1.257+0.4*exp(-1.1/Kn)))/dp
	mobil *= (1+m_g/mass)^(-1/2)	// mass correction term
	return mobil							// particle mobility [m2/V/s]
end
//======================================================================

// Conversions between mobility and diameter
function ZpCalc_DpDMA(dp)
	variable dp
	
	// Stokes regime:
	//	D=kTB:	B=Cc / 3 pi eta dp
	//	Zp = n e Cc / (3 pi eta dp)
	variable gam1 = 2.492/2
	variable gam2 = 0.84/2
	variable gam3 = 0.43*2
	variable Kn = 2*lambda()/dp
	variable Cc = 1+ Kn*(gam1+gam2*exp(-gam3/Kn))
	variable Zp = echg()*Cc/(3*pi*etagas()*dp)
	
	return Zp								// mobility [m2/V/s]; assumed singly-charged
end
function DpCalc_Zdma(Zp)
	variable Zp
	
	//	Stokes regime:
	//	D=kTB:	B=Cc / 3 pi eta dp
	//	Zp = n e Cc / (3 pi eta dp)
	variable gam1 = 2.492/2
	variable gam2 = 0.84/2
	variable gam3 = 0.43*2
	variable Kn, Cc, Dslip, comparison
	variable PAR = echg()/(3*pi*etagas()*Zp)
	variable Dinitial = (PAR+sqrt(PAR*(PAR+(8*lambda()*gam1))))/2
	do
		Kn = 2*lambda()/Dinitial
		Cc = 1+ Kn*(gam1+gam2*exp(-gam3/Kn))
		Dslip = Cc*PAR
		comparison = abs((Dinitial-Dslip)/Dslip)
			if (comparison>1e-14)
				Dinitial = (Dinitial+Dslip)/2
			endif
	while(comparison>1e-14)
	variable Dp = Dinitial
	
	return Dp
end
//======================================================================

// Interconversions between diameter, mobility, and voltage
function DpCalc_Vdma(V)
	variable V
	
	variable Zp = ZpCalc_Vdma(V)
	
	return DpCalc_Zdma(Zp)
end
function VCalc_DpDMA(dp)
	variable dp
	
	variable Zp = ZpCalc_DpDMA(dp)
	
	return VCalc_Zdma(Zp)
end
function VCalc_Zdma(Zp)
	variable Zp
	
	return QshDMA()*ln(R2dma()/R1dma())/(2*pi*Ldma()*Zp)
end
function ZpCalc_Vdma(V)
	variable V
	
	return QshDMA()*ln(R2dma()/R1dma())/(2*pi*V*Ldma())
end
//======================================================================


// CPC counting efficiencies; from Wiedensohler 1997 exponential fit of counting efficiency curves
function counteff_CPC3025(dp_m)
	variable dp_m
	
	variable Dp_nm=dp_m*1e9
	variable Dp0_nm=2.22			// 0% activation efficiency
	variable Dp50_nm=2.42			// 50% activation efficiency
	variable Dp2_nm=0.28			// slope
	
	variable cpceff=1
	if (Dp_nm<=Dp0_nm)
		cpceff=0
	else
		cpceff=1-exp((Dp0_nm-Dp_nm)/(Dp2_nm))
	endif
	
	return cpceff
end
//======================================================================