#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Calculate the inversion matrix using cubic B-splines for numerical quadrature of kernel integral
// no assumptions about knot spacing; no added conditions ["natural cubic"]
//
// Input variables and waves [dimensions in brackets]:
//	ChanZ	[m]			vector of mobility values for measurement channels; Z [m2/V/s]
//	ChanV	[m]			vector of voltage values for measurement channels
//	Dpbin		[nDias]	vector of diameter values to define boundaries of size distribution intervals;
//								should be input as ln(Dp) for log-linear channel spacing (Dp in meters)
//	nSub - number of subdivisions for evaluating integral over size bin intervals
//	bbeta, delta - flow parameters
//	chargemax - max charges in kernel function
//	TFformula - string to call form of transfer function
//
// Output is kernel matrix "KernelS3" and Bspline knot waves
function CalcKernel_cubicBspline(ChanZ,ChanV,Dpbin,nSub,bbeta,delta,chargemax,TFformula)
	wave ChanZ,ChanV,Dpbin
	variable nSub,bbeta,delta,chargemax
	string TFformula
	
	// Define spline control points and bounds for integrations
	define_binning(Dpbin)
	wave binvals_geom,binvals_plusuplodiff
	duplicate/o binvals_geom, Dpbinbound
	duplicate/o binvals_plusuplodiff, ControlptBinning
	variable/g Dpbinbound_lo=Dpbinbound[0]
	variable/g Dpbinbound_hi=Dpbinbound[(numpnts(Dpbinbound)-1)]
	variable/g Dpbin_lo=Dpbin[0]
	variable/g Dpbin_hi=Dpbin[(numpnts(Dpbin)-1)]
	
	// Make the wave of knot values needed to calculate Bsplines
	variable degree=3		// degree of spline interpolation; cubic, d=3
	MakeBsplineKnotWave(ControlptBinning,degree)
	wave BsplineKnotWave, BsplineCoeffKnots_j, BsplineCoeffKnots, BsplineControlptKnots
	duplicate/o BsplineKnotWave, Dpbin_knots
	
	variable m = numpnts(ChanZ)
	variable n  = numpnts(Dpbin)
	variable ncoeffs = numpnts(BsplineCoeffKnots)
	variable dDpInt
	if (nSub>1)	// work with even number of subintervals to capture middle Dp of bin
		variable nSub_even = mod(nSub,2)==0 ? nSub : (nSub+1)
		nSub=nSub_even
	endif
	make/o/n= (nSub+1) DpInt	// vector of diameter values for integral over size bins
	make/o/n=(m,ncoeffs) KernelS3=0	// initialize kernel matrix
	variable xk,xk1,xk2,xk3,xk4
	variable h
	variable i,j,k,knotindx
	for(i=0; i<m;i+=1)
		for(k=0;k<ncoeffs;k+=1)
			knotindx=k
			xk=Dpbin_knots[knotindx]
			xk1=Dpbin_knots[knotindx+1]
			xk2=Dpbin_knots[knotindx+2]
			xk3=Dpbin_knots[knotindx+3]
			xk4=Dpbin_knots[knotindx+4]
			// Bpline1
			h=(xk1-xk)
			dDpInt=h/nSub
			DpInt=xk+dDpInt*x
			duplicate/d/o DpInt, ChiSub1
			duplicate/d/o ChiSub1, fSub1; fSub1=BsplineC1(ChiSub1,xk,xk1,xk2,xk3,xk4)
			duplicate/d/o ChiSub1, Ksub1
			Ksub1=KernelFcn_DMA(ChiSub1,ChanZ[i],ChanV[i],bbeta,delta,chargemax,TFformula)
			Ksub1*=fsub1
			// Bpline2
			h=(xk2-xk1)
			dDpInt=h/nSub
			DpInt=xk1+dDpInt*x
			duplicate/d/o DpInt, ChiSub2
			duplicate/d/o ChiSub2, fSub2; fSub2=BsplineC2(ChiSub2,xk,xk1,xk2,xk3,xk4)
			duplicate/d/o ChiSub2, Ksub2
			Ksub2=KernelFcn_DMA(ChiSub2,ChanZ[i],ChanV[i],bbeta,delta,chargemax,TFformula)
			Ksub2*=fsub2
			// Bpline3
			h=(xk3-xk2)
			dDpInt=h/nSub
			DpInt=xk2+dDpInt*x
			duplicate/d/o DpInt, ChiSub3
			duplicate/d/o ChiSub3, fSub3; fSub3=BsplineC3(ChiSub3,xk,xk1,xk2,xk3,xk4)
			duplicate/d/o ChiSub3, Ksub3
			Ksub3=KernelFcn_DMA(ChiSub3,ChanZ[i],ChanV[i],bbeta,delta,chargemax,TFformula)
			Ksub3*=fsub3
			// Bpline4
			h=(xk4-xk3)
			dDpInt=h/nSub
			DpInt=xk3+dDpInt*x
			duplicate/d/o DpInt, ChiSub4
			duplicate/d/o ChiSub4, fSub4; fSub4=BsplineC4(ChiSub4,xk,xk1,xk2,xk3,xk4)
			duplicate/d/o ChiSub4, Ksub4
			Ksub4=KernelFcn_DMA(ChiSub4,ChanZ[i],ChanV[i],bbeta,delta,chargemax,TFformula)
			Ksub4*=fsub4
			
			if (k>3 && k<(ncoeffs-4))
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1)
				KernelS3[i][k]+=AreaXY(ChiSub2,Ksub2)
				KernelS3[i][k]+=AreaXY(ChiSub3,Ksub3)
				KernelS3[i][k]+=AreaXY(ChiSub4,Ksub4)
			endif
			if (k==0)
				KernelS3[i][k]=AreaXY(ChiSub4,Ksub4,Dpbinbound_lo,Dpbin_lo)
			endif
			if (k==1)
				KernelS3[i][k]=AreaXY(ChiSub3,Ksub3,Dpbinbound_lo,Dpbin_lo)
				KernelS3[i][k]+=AreaXY(ChiSub4,Ksub4)
			endif
			if (k==2)
				KernelS3[i][k]=AreaXY(ChiSub2,Ksub2,Dpbinbound_lo,Dpbin_lo)
				KernelS3[i][k]+=AreaXY(ChiSub3,Ksub3)
				KernelS3[i][k]+=AreaXY(ChiSub4,Ksub4)
			endif
			if (k==3)
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1,Dpbinbound_lo,Dpbin_lo)
				KernelS3[i][k]+=AreaXY(ChiSub2,Ksub2)
				KernelS3[i][k]+=AreaXY(ChiSub3,Ksub3)
				KernelS3[i][k]+=AreaXY(ChiSub4,Ksub4)
			endif
			if (k==ncoeffs-4)
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1)
				KernelS3[i][k]+=AreaXY(ChiSub2,Ksub2)
				KernelS3[i][k]+=AreaXY(ChiSub3,Ksub3)
				KernelS3[i][k]+=AreaXY(ChiSub4,Ksub4,Dpbin_hi,Dpbinbound_hi)
			endif
			if (k==ncoeffs-3)
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1)
				KernelS3[i][k]+=AreaXY(ChiSub2,Ksub2)
				KernelS3[i][k]+=AreaXY(ChiSub3,Ksub3,Dpbin_hi,Dpbinbound_hi)
			endif
			if (k==ncoeffs-2)
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1)
				KernelS3[i][k]+=AreaXY(ChiSub2,Ksub2,Dpbin_hi,Dpbinbound_hi)
			endif
			if (k==ncoeffs-1)
				KernelS3[i][k]=AreaXY(ChiSub1,Ksub1,Dpbin_hi,Dpbinbound_hi)
			endif
		endfor
	endfor
end
//======================================================================

// B-spline functional forms
function BsplineC1(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable B1=((xi-xk)^3)/((xk3-xk)*(xk2-xk)*(xk1-xk))
	return B1
end

function BsplineC2(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable B2=(xi-xk)/(xk3-xk)*(xi-xk)/(xk2-xk)*(xk2-xi)/(xk2-xk1)
	B2+=(xi-xk)/(xk3-xk)*(xk3-xi)/(xk3-xk1)*(xi-xk1)/(xk2-xk1)
	B2+=(xk4-xi)/(xk4-xk1)*(xi-xk1)/(xk3-xk1)*(xi-xk1)/(xk2-xk1)
	return B2
end

function BsplineC3(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable B3=(xi-xk)/(xk3-xk)*(xk3-xi)/(xk3-xk1)*(xk3-xi)/(xk3-xk2)
	B3+=(xk4-xi)/(xk4-xk1)*(xi-xk1)/(xk3-xk1)*(xk3-xi)/(xk3-xk2)
	B3+=(xk4-xi)/(xk4-xk1)*(xk4-xi)/(xk4-xk2)*(xi-xk2)/(xk3-xk2)
	return B3
end

function BsplineC4(xi,xk,xk1,xk2,xk3,xk4)
	variable xi,xk,xk1,xk2,xk3,xk4
	
	variable B4=((xk4-xi)^3)/((xk4-xk1)*(xk4-xk2)*(xk4-xk3))
	return B4
end
//======================================================================

// Make the wave of knot values needed to calculate B-splines; specify degree
function MakeBsplineKnotWave(ControlptBin,degree)
	wave ControlptBin
	variable degree
	
	variable deg=degree
	variable nControlpts=numpnts(ControlptBin)
	variable ntotalKnots=nControlpts+2*deg
	variable nBsplCoeffs=nControlpts+deg-1
	
	duplicate/o ControlptBin, BsplineControlptKnots, BsplineKnotWave
	variable hEven=(ControlptBin[nControlpts-1]-ControlptBin[0])/(nControlpts-1)
	variable nEndKnots=deg		// knot points needed at upper and lower bounds of ControlptBin
	variable i,insertindx
	for(i=1;i<=nEndKnots;i+=1)	// add knot points to upper bound
		insertindx=(nControlpts-1)+i
		insertpoints (insertindx),1,BsplineKnotWave
		BsplineKnotWave[insertindx]=ControlptBin[nControlpts-1]+hEven*i
	endfor
	for(i=1;i<=nEndKnots;i+=1)	// add knot points to lower bound
		insertpoints 0,1,BsplineKnotWave
		BsplineKnotWave[0]=ControlptBin[0]-hEven*i
	endfor
	
	duplicate/o/r=(0,(nBsplCoeffs-1)) BsplineKnotWave, BsplineCoeffKnots_j
	duplicate/o/r=(((deg+1)/2),((nBsplCoeffs-1)+(deg+1)/2)) BsplineKnotWave, BsplineCoeffKnots
end
//======================================================================

// Define spline control points and bounds for integrations
function define_binning(wavetobin)
	wave wavetobin
	
	variable numwvpts=numpnts(wavetobin)
	duplicate/o wavetobin, binvals_updiff, binvals_lodiff
	make/o/n=(numwvpts+1) binvals_geom
	make/o/n=(numwvpts) deltavals_updiff, deltavals_lodiff, deltavals_geom
	
	variable/g deltaval_even=(wavetobin[numwvpts-1]-wavetobin[0])/(numwvpts-1)
	insertpoints numwvpts,1,binvals_updiff
	binvals_updiff[numwvpts]=wavetobin[numwvpts-1]+deltaval_even
	insertpoints 0,1,binvals_lodiff
	binvals_lodiff[0]=wavetobin[0]-deltaval_even	
	
	duplicate/o binvals_updiff, binvals_plusuplodiff
	insertpoints 0,1,binvals_plusuplodiff; binvals_plusuplodiff[0]=binvals_lodiff[0]
	
	binvals_geom[1,(numwvpts-1)]=sqrt(wavetobin(p-1)*wavetobin(p))*sign(wavetobin(p))
	binvals_geom[0]=sqrt(binvals_lodiff[0]*wavetobin[0])*sign(wavetobin[0])
	binvals_geom[numwvpts]=sqrt(wavetobin[numwvpts-1]*binvals_updiff[numwvpts])
	binvals_geom[numwvpts]*=sign(wavetobin[numwvpts-1])
	
	deltavals_updiff=binvals_updiff(p+1)-binvals_updiff(p)
	deltavals_lodiff=binvals_lodiff(p+1)-binvals_lodiff(p)
	deltavals_geom=binvals_geom(p+1)-binvals_geom(p)
	
	make/o/n=(2*numwvpts+1) binmidandgeom
	concatenate/np/o {wavetobin,binvals_geom}, binmidandgeom
	sort binmidandgeom, binmidandgeom
end
//======================================================================
