#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Non-negative least squares method to solve problems described as:
//	B = A x Model, where B is measured data, A is m x n matrix, and Model is solution in question
//	input: matrixA (mxn), vectorb (mx1), as well as wave to hold solution vectorx (nx1)
//	output: vectorx>0 such that vectorx = arg min ||Ax - b||^2
//	index sets P & R defined and modified in execution of algorithm
//	initial solution vector x set to (nx1) zero vector
//	n-vectors w & s provide working space
//
// Input waves [dimensions in brackets]:
//	AmatrixInput			[m x n]	kernel matrix, describing measurement collection
//	BvectorInput			[m]			vector of measured signals (raw data)
//	ModelWaveOutput	[n]			wave to hold size distribution solution;
//											number of points must match y-dimension of AmatrixInput
// Output is solution in "ModelWaveOutput"
function NNLS(AmatrixInput,BvectorInput,ModelWaveOutput)
	wave AmatrixInput,BvectorInput,ModelWaveOutput
	
	// create working space
	string OldDf=GetDataFolder(1)
	NewDataFolder/O/S root:Packages
	NewDataFolder/O/S root:Packages:NNLS
	// create local copies of input waves
	duplicate/o AmatrixInput, matrixA
	duplicate/o BvectorInput, vectorb
	redimension/d matrixA, vectorb
	
	variable/g m,n
	m = dimsize(matrixA,0)
	n = dimsize(matrixA,1)
	
	// Initialization: P = null, R = {1, 2,..., n}, x = 0, w = A^T (b - Ax)
	make/o/n=(n) setP; setP = 0
	make/o/n=(n) setR; setR = p+1
	make/d/o/n=(n) vectorx; vectorx = 0
	matrixop/o vectorw = matrixA^t x (vectorb - matrixA x vectorx)
	duplicate/o vectorx, ModelWaveOutputIter
	
	variable/g tolerance = 1e-10
	variable/g i_outer = 0		// initialize count of iteration number
	do
		duplicate/o vectorw, vectorwR
		variable R_index
		for(R_index=0;(R_index+1)<=n;R_index+=1)
			if (setR[R_index]==0)
				vectorwR[R_index] = -inf
			endif
		endfor
		
		matrixop/o wRvtols = sum(greater(tolerance,vectorwR)-1)
		matrixop/o anyRs = sum(equal(setR,0)-1)
		variable wRvtol = wRvtols[0]
		variable anyR = anyRs[0]
		if (wRvtol==0 || anyR==0)			// proceed if R not empty and [max(wi) > tolerance]
			break
		endif
		i_outer+=1
		if (i_outer>=(3*n))
			break
		endif
		
		make/d/o/n=(n) vector_s; vector_s = 0
		wavestats/q vectorwR
		variable t_index = V_maxloc		// t = arg max(wi)
		setP[t_index] = setR[t_index]		// include the index t in P and remove it from R
		setR[t_index] = 0
		
		// Make matrixAp associated with only the variables currently in the passive set P
		make/d/o/n=(m,n) matrixAp; matrixAp=0
		variable jj = 0
		variable j
		for(j=0;(j+1)<=n;j+=1)
			if (setP[j]>0)
				matrixAp[][jj] = matrixA[p][j]
				jj+=1
			endif
		endfor
		redimension/n=(m,jj) matrixAp
		matrixSVD/b matrixAp
		wave M_U,W_W,M_V
		duplicate/o W_W, W_Winv; W_Winv = 1/W_W
		W_Winv = numtype(W_Winv)==1 ? 0 : W_Winv
		matrixop/o W_WinvMatrix = DiagRC(W_Winv,jj,jj)
		matrixop/o M_x = M_V x W_WinvMatrix x M_U^t x vectorb
		make/d/o/n=(jj) vector_sP; vector_sP = M_x[p]
		make/d/o/n=(jj) vectorxP
		variable kk = 0
		variable k
		for(k=0;(k+1)<=n;k+=1)
			if (setP[k]==0)
				vector_s[k] = 0
			else
				vector_s[k] = vector_sP[kk]
				vectorxP[kk] = vectorx[k]
				kk+=1
			endif
		endfor
		
		variable/g i_inner = 0
		do							// sP = [(Ap)^T x Ap]^-1 x (Ap)^T x b
			matrixop/o sPvtols = sum(greater(vector_sP,tolerance)-1)
			variable sPvtol = sPvtols[0]
			if (sPvtol==0)		// proceed if min(sP) <= 0
				break
			endif
			
			i_inner+=1
			if (i_inner>=3*n)
				break
			endif
			
			make/d/o/n=(jj) alphawave
			variable sPpnts = numpnts(vector_sP)
			variable qq = 0
			variable q
			for(q=0;(q+1)<=sPpnts;q+=1)
				if (vector_sP[q]<=tolerance)
					alphawave[qq] = vectorxP(q)/(vectorxP(q)-vector_sP(q))
					qq+=1
				endif
			endfor
			variable qpnts = qq
			redimension/n=(qpnts) alphawave
			wavestats/q alphawave
			variable/g alpha = V_min		// alpha = -min[xi/(xi - si)]
			matrixop/o vectorx = vectorx+alpha*(vector_s-vectorx)		// x := x + alpha(s - x)
			
			// Update R and P: move from setP to setR all indices j in P for which xj = 0
			variable j_index
			for(j_index=0;(j_index+1)<=n;j_index+=1)
				if (setP[j_index]>0 && abs(vectorx[j_index])<tolerance)
					setR[j_index] = setP[j_index]
					setP[j_index] = 0
				endif
			endfor
			
			// sP = [(Ap)^T x Ap]^-1 x (Ap)^T x b
			// sR = 0
			make/d/o/n=(n) vector_s; vector_s = 0
			
			// Make matrixAp associated with only the variables currently in the passive set P
			make/d/o/n=(m,n) matrixAp; matrixAp=0
			variable jj_inner = 0
			Variable j_inner
			for(j_inner=0;(j_inner+1)<=n;j_inner+=1)
				if (setP[j_inner]>0)
					matrixAp[][jj_inner] = matrixA[p][j_inner]
					jj_inner+=1
				endif
			endfor
			redimension/n=(m,jj_inner) matrixAp
			matrixSVD/b matrixAp
			wave M_U,W_W,M_V
			duplicate/o W_W, W_Winv; W_Winv = 1/W_W
			W_Winv = numtype(W_Winv)==1 ? 0 : W_Winv
			matrixop/o W_WinvMatrix = DiagRC(W_Winv,jj_inner,jj_inner)
			matrixop/o M_x = M_V x W_WinvMatrix x M_U^t x vectorb
			make/d/o/n=(jj_inner) vector_sP; vector_sP = M_x[p]
			make/d/o/n=(jj_inner) vectorxP
			variable kk_inner = 0
			variable k_inner
			for(k_inner=0;(k_inner+1)<=n;k_inner+=1)
				if (setP[k_inner]==0)
					vector_s[k_inner] = 0
				else
					vector_s[k_inner] = vector_sP[kk_inner]
					vectorxP[kk_inner] = vectorx[k_inner]
					kk_inner+=1
				endif
			endfor
		while (1)
		vectorx = vector_s(p)		// x = s
		matrixop/o vectorw = matrixA^t x (vectorb - matrixA x vectorx)		// w = A^T x (b - Ax)
		
		redimension/n=(-1,(1+i_outer)) ModelWaveOutputIter
		ModelWaveOutputIter[][i_outer]=vectorx[p]
	while(1)
	
	ModelWaveOutput = vectorx
	SetDataFolder OldDf
end
//======================================================================
