#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Regularization method to solve problems which can be described as:
//	B = A x Model, where B is measured data, A is m x n matrix, and Model is solution in question
// reference: Ilavsky and Jemian, J Appl Chryst Appl 42,2 (2009), pp. 347-353
//
// Input waves [dimensions in brackets]:
//	AmatrixInput				[m x n]	kernel matrix, describing measurement collection
//	ModelWaveOutput		[n]			wave to hold size distribution solution;
//												number of points must match y-dimension of AmatrixInput
//	BvectorInput				[m]			vector of measured signals (raw data)
//	Bvector_uncertainties	[m]			vector of uncertainty/error values associated with
//												measured signals (raw data)
//
// Internally calculated variables and waves [dimensions in brackets]:
//	H_matrix		[n x n]		constraint matrix; here done for second derivative:	MakeHmatrix()
//	B_vector		[n]			calculated from Kmatrix, Bvector and errors:	CalculateBVector()	
//	D_matrix		[n x n]		calculated from Kmatrix and errors:	CalculateDMatrix()
//	A_matrix		[n x n]		calculated from A[][] = D[][] + a * H[][]	CalculateAmatrix()
//	Avalue - for the fitting itself; call with precision (e~0.1 or so):	FindOptimumAvalue(Evalue)
//	Evalue - internal precision parameter, for now hardwired to 0.1; range 0 to 0.5
//				lower value requires resulting chi^2 to be closer to target
//	Chisquared - chi squared sum of the difference value between the fit and measured data
//
// Output is solution in "ModelWaveOutput"
//
// Internal parameter of Regularization:
// Regularization is forced to have at least this * max of SD in each bin to avoid negative values 
constant RegularizationMinRatio = 1e-4

function Regularization(AmatrixInput,ModelWaveOutput,BvectorInput,Bvector_uncertainties)
	wave AmatrixInput,ModelWaveOutput,BvectorInput,Bvector_uncertainties
	
	// create working space
	string OldDf=GetDataFolder(1)
	NewDataFolder/O/S root:Packages
	NewDataFolder/O/S root:Packages:Regularization
	// create local copies of input waves
	duplicate/o AmatrixInput, Kmatrix
	duplicate/o BvectorInput, Bvector
	duplicate/o Bvector_uncertainties, Uncertainties
	Uncertainties = Bvector_uncertainties==0 ? 1 : Bvector_uncertainties
	redimension/d Kmatrix, Bvector, Uncertainties
	
	MakeHmatrix()			// create H matrix
	CalculateBVector()		// create B vector
	CalculateDMatrix()		// create D matrix
	
	variable/g Evalue=0.1	// may not be needed in the future
	variable/g NumberIterations
	NumberIterations=FindOptimumAvalue(Evalue)	// do the fitting for given e value
	
	wave CurrentResultSizeDistribution=root:Packages:Regularization:CurrentResultSizeDistribution
	wave NormalizedResidual=root:Packages:Regularization:NormalizedResidual
	nvar Chisquare=root:Packages:Regularization:Chisquare
	
	if ((numtype(NumberIterations)!=0)||(numberIterations>100))		// no solution found
		CurrentResultSizeDistribution = 0
		NormalizedResidual = 0
		Chisquare = 0	// new Chisquared
	endif
	
	ModelWaveOutput=CurrentResultSizeDistribution
	SetDataFolder OldDf
end
//======================================================================

// Make the H_matrix
function MakeHmatrix()
	
	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization
	
	wave Kmatrix=root:Packages:Regularization:Kmatrix
	variable numOfPoints=dimsize(Kmatrix,1), i=0, j=0
	
	make/d/o/n=(numOfPoints,numOfPoints) H_matrix; H_matrix=0	// make and zero the matrix
	for(i=2;i<numOfPoints-2;i+=1)		// fill most of matrix with 1 -4 6 -4 1
		for(j=0;j<numOfPoints;j+=1)
			if (j==i-2)
				H_matrix[i][j]=1
			endif
			if (j==i-1)
				H_matrix[i][j]=-4
			endif
			if (j==i)
				H_matrix[i][j]=6
			endif
			if (j==i+1)
				H_matrix[i][j]=-4
			endif
			if (j==i+2)
				H_matrix[i][j]=1
			endif
		endfor
	endfor
	H_matrix[0][0]=1		// fill in beginning of the H_matrix
	H_matrix[0][1]=-2
	H_matrix[0][2]=1
	H_matrix[1][0]=-2
	H_matrix[1][1]=5
	H_matrix[1][2]=-4
	H_matrix[1][3]=1
	H_matrix[numOfPoints-2][numOfPoints-4]=1		// fill in end of the H_matrix
	H_matrix[numOfPoints-2][numOfPoints-3]=-4
	H_matrix[numOfPoints-2][numOfPoints-2]=5
	H_matrix[numOfPoints-2][numOfPoints-1]=-2
	H_matrix[numOfPoints-1][numOfPoints-3]=1
	H_matrix[numOfPoints-1][numOfPoints-2]=-2
	H_matrix[numOfPoints-1][numOfPoints-1]=1

	SetDataFolder OldDf
end
//======================================================================

// Make new B_vector, calculated from Kmatrix, Bvector and Uncertainties
function CalculateBVector()
	
	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization
	
	wave Kmatrix	=root:Packages:Regularization:Kmatrix
	wave Bvector	=root:Packages:Regularization:Bvector
	wave Uncertainties=root:Packages:Regularization:Uncertainties
	
	variable M=dimsize(Kmatrix,0)		// rows, i.e, measured points number
	variable N=dimsize(Kmatrix,1)		// columns, i.e., bins in distribution
	variable i=0, j=0
	make/d/o/n=(N) B_vector				// points = bins in size dist.
	B_vector=0
	for(i=0;i<N;i+=1)					
		for(j=0;j<M;j+=1)
			B_vector[i]+=((Kmatrix[j][i]*Bvector[j])/(Uncertainties[j]*Uncertainties[j]))
		endfor
	endfor

	SetDataFolder OldDf
end
//======================================================================

// Make new D_matrix, calculated from Kmatrix and Uncertainties
function CalculateDMatrix()
	
	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization
	
	wave Kmatrix=root:Packages:Regularization:Kmatrix
	wave Uncertainties=root:Packages:Regularization:Uncertainties
	
	variable M=dimsize(Kmatrix,0)		// rows, i.e, measured points number
	variable N=dimsize(Kmatrix,1)		// columns, i.e., bins in distribution
	variable i=0, j=0, k=0
	make/d/o/n=(N,N) D_matrix; D_matrix=0
	duplicate/o Uncertainties, Uncertainties2
	Uncertainties2=Uncertainties^2

	duplicate/o Kmatrix, Kmatrix_ErrScaled
	for(i=0;i<N;i+=1)	
		for(j=0;j<M;j+=1)				
			Kmatrix_ErrScaled[j][i]=Kmatrix[j][i]/(Uncertainties2[j])
		endfor
	endfor	
	matrixop/o testM =  Kmatrix_ErrScaled^t x Kmatrix
	D_matrix = testM
	
	SetDataFolder OldDf
end
//======================================================================


// Generate A_matrix
function CalculateAmatrix(aValue)
	variable aValue

	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization

	wave D_matrix=root:Packages:Regularization:D_matrix
	wave H_matrix=root:Packages:Regularization:H_matrix
	
	duplicate/o D_matrix, A_matrix; A_matrix=0
	A_matrix=D_matrix[p][q]+aValue*H_matrix[p][q]
	
	SetDataFolder OldDf
end
//======================================================================

// Do the fitting itself, call with precision (e~0.1 or so)
function FindOptimumAvalue(Evalue)
	variable Evalue	

	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization

	wave Kmatrix=root:Packages:Regularization:Kmatrix
	wave Bvector=root:Packages:Regularization:Bvector
	wave Uncertainties=root:Packages:Regularization:Uncertainties
	variable numOfPoints=dimsize(Kmatrix,1)
	
	variable ChisquareVal,err
	make/o/n=1 IterCount,errorVals,ChiSqVals, Avals
	make/o/n=(numOfPoints) ModelWaveOutIter; ModelWaveOutIter=0
	MatrixOp/O CurrentResultB = Kmatrix x ModelWaveOutIter	// calculated data from our model
	duplicate/o Bvector, tempWv, NormalizedResidual
	tempWv = (Bvector - CurrentResultB) / Uncertainties
	NormalizedResidual = tempWv
	tempWv = tempWv^2
	ChisquareVal = sum(tempWv)
	err=ChisquareVal/numpnts(Bvector)
	errorVals[0]=err
	ChiSqVals[0]=ChisquareVal
	Avals[0]=1
	
	variable LogAmax=100, LogAmin=-100, M=numpnts(Bvector)
	variable Chisquared, MidPoint, Avalue, i=0, logAval, LogChisquarerdDivN, Smoothness
	do
		MidPoint=(LogAmax+LogAmin)/2
		Avalue=10^MidPoint		// calculate a
		CalculateAmatrix(Avalue)
		wave A_matrix
		variable n = dimsize(A_matrix,1)
		make/o/n=(n) M_x; M_x=0
		wave B_vector
		NNLS(A_matrix,B_vector,M_x)
		wave M_x
		duplicate/o M_x, CurrentResultSizeDistribution	// put the data into the wave 
		
		Chisquared=CalculateChisquared()	// calculate C 	C=|| I - G M_x ||
		i+=1
		if (i>100)		// no solution found
			print i
			return NaN
		endif
		
		// calculate data from model and track results through iterations
		matrixop/o CurrentResultB = Kmatrix x CurrentResultSizeDistribution
		duplicate/o Bvector, tempWv, NormalizedResidual
		tempWv = (Bvector - CurrentResultB) / Uncertainties
		NormalizedResidual = tempWv
		tempWv = tempWv^2
		ChisquareVal = sum(tempWv)
		err=ChisquareVal/numpnts(Bvector)
		insertpoints i,1, IterCount,errorVals,ChiSqVals,Avals
		IterCount[i]=i
		errorVals[i]=err
		ChiSqVals[i]=Chisquared
		Avals[i]=Avalue
		redimension/n=(-1,(1+i)) ModelWaveOutIter
		ModelWaveOutIter[][i]=CurrentResultSizeDistribution[p]
		
		variable tolerance=1e-10
		if (sum(Uncertainties)==M)
			if (Chisquared>(M/10))
				LogAMax=MidPoint
			else
				LogAmin=MidPoint
			endif
		else
			if (Chisquared>(M/100))
				LogAMax=MidPoint
			else
				LogAmin=MidPoint
			endif
		endif
		
		if(i<50)
			continue
		endif
	while((err>1 && i<101))
	
	variable/g Chisquare=Chisquared
	return i
	
	SetDataFolder OldDf
end
//======================================================================

// Calculate Chisquared difference between data in Bvector and result calculated from x_vector
function CalculateChisquared()
	
	string OldDf
	OldDf=GetDataFolder(1)
	SetDataFolder root:Packages:Regularization

	wave Kmatrix=root:Packages:Regularization:Kmatrix
	wave Bvector=root:Packages:Regularization:Bvector
	wave Uncertainties=root:Packages:Regularization:Uncertainties
	wave M_x=root:Packages:Regularization:M_x
	
	duplicate/o Bvector, NormalizedResidual, ChisquaredWave	// waves for data
	
	MatrixMultiply  Kmatrix, M_x	// generate data from current result
	wave M_product	
	redimension/d/n=(-1,0) M_product
	
	NormalizedResidual=(Bvector-M_product)/Uncertainties
	ChisquaredWave=NormalizedResidual^2		// wave with Chisquared	
	return (sum(ChisquaredWave,-inf,inf))			// return sum of Chisquared
	
	SetDataFolder OldDf
end
//======================================================================
