#pragma rtGlobals=3		// Use modern global access method and strict wave access.


// Totally non-negative least squares method to solve problems which can be described as:
//	B = A x Model, where B is measured data, A is m x n matrix, and Model is solution in question
// reference: Michael Merritt and Yin Zhang, J Optim Theory Appl 126,1 (2005), pp. 191-202
//
// Input variables and waves [dimensions in brackets]:
//	AmatrixInput				[m x n]	kernel matrix, describing measurement collection
//	ModelWaveOutput		[n]			wave to hold size distribution solution;
//												number of points must match y-dimension of AmatrixInput
//	BvectorInput				[m]			vector of measured signals (raw data)
//	Bvector_uncertainties	[m]			vector of uncertainty/error values associated with
//												measured signals (raw data)
//	ApproachParameter - "step" - needs to be smaller than 1, usually 0.6 is good;
//										reasonable range seems to be 0.3 - 0.99
//	MaxNumIterations - limit to sensible number...depends on complexity of problem...
//
// Output is solution in "ModelWaveOutput"
function IPG_TNNLS(AmatrixInput,ModelWaveOutput,BvectorInput,Bvector_uncertainties,ApproachParameter,MaxNumIterations)
	wave AmatrixInput,ModelWaveOutput,BvectorInput,Bvector_uncertainties
	variable ApproachParameter,MaxNumIterations
	
	// create working space
	string OldDf=GetDataFolder(1)
	NewDataFolder/O/S root:Packages
	NewDataFolder/O/S root:Packages:TNNLS
	// create local copies of input waves
 	duplicate/o AmatrixInput, AmatrixInputE, AmatrixOrig
	duplicate/o BvectorInput, BvectorInputE, BvectorOrig
	duplicate/o Bvector_uncertainties, Uncertainties
	Uncertainties = Bvector_uncertainties==0 ? 1 : Bvector_uncertainties
	// note this code defaults to use uncertainties;
	// if uncertainties not available, set following parameter to 0 and modify the code
	variable useUncertaintiesInput=1
	if (useUncertaintiesInput)
		AmatrixInputE = AmatrixOrig[p][q] / Uncertainties[p]
		matrixop/o BvectorInputE = BvectorOrig / Uncertainties
	endif
	// create local meaningful waves to work with which account for the errors used
	duplicate/o AmatrixInputE, Amatrix
	duplicate/o ModelWaveOutput, ModelWave
	duplicate/o BvectorInputE, Bvector
	redimension/d Amatrix, ModelWave, Bvector, Uncertainties
	// create some parameters for use after run to find out what happened
	variable/g NumberIterations, Chisquare
	// first we will need some variables and waves, etc. to work with
	variable i,j
	matrixop/o AmatrixT=Amatrix^t
	// start with reliably small positive number; assume 1e-32 is such, but may fail in some cases...
	ModelWave = 1e-32
	// working waves & variables
	variable numIter=0
	variable err=0	
	variable alphaStar, temp1, temp2
	// here the iterations start...
	make/o/n=1 IterCount,errorVals
	duplicate/o ModelWave, ModelWaveIter
	matrixop/o CurrentResultB = AmatrixOrig x ModelWave	// calculated data from our model
	duplicate/o BvectorOrig, tempWv, NormalizedResidual
	tempWv = (BvectorOrig - CurrentResultB) / Uncertainties
	NormalizedResidual = tempWv
	tempWv = tempWv^2
	Chisquare = sum(tempWv)
	err=Chisquare/numpnts(BvectorOrig)
	errorVals[0]=err
	do
	// start of NNLS interior point gradient method itself; step designations relate to original paper
		// step 1
		matrixop/o Qk = AmatrixT x Amatrix x ModelWave - AmatrixT x Bvector
		matrixop/o Dk = ModelWave / (AmatrixT x Amatrix x ModelWave)
		matrixop/o Pk = - Dk * Qk
		for(j=0;j<numpnts(Pk);j+=1)
			if (Qk[j]==0)
				Pk[j] = 0
			endif
		endfor
		// step 2	
		matrixop/o AkSTAR= (Pk^t x AmatrixT x Amatrix x Pk)
		temp1 = AkSTAR[0]
		matrixop/o AkSTAR= - (Pk^t x Qk) 
		temp2 = AkSTAR[0]
		alphaStar = temp2 / temp1
		redimension/n=(numpnts(ModelWave)) AkSTAR
		AkSTAR = numtype(alphaStar)==0 ? alphaStar : 0
		// above is ideal step to make; below is limiting the step so we do not get negative values...		
		matrixop/o AlphaWv =  - ModelWave/Pk	// max alpha, which we can make, if Pk is neg
		AlphaWv = numtype(AlphaWv)==0 ? AlphaWv : 0
		for(i=0;i<numpnts(Pk);i+=1)
			if (Pk[i]<0)	// if Pk negative, may have to limit the step to smaller of the two values
				AkSTAR[i]=min(ApproachParameter*AlphaWv[i],AkSTAR[i])
			endif
		endfor
		// step 3; new model
		matrixop/o ModelWave = ModelWave + (AkSTAR * Pk)	// loop back after calcs below
	// end of NNLS interior point gradient method itself
	// figure out Chisquare so we know if we should bail out...
		matrixop/o CurrentResultB = AmatrixOrig x ModelWave	// calculated data from our model
		duplicate/o BvectorOrig, tempWv, NormalizedResidual
		tempWv = (BvectorOrig - CurrentResultB) / Uncertainties
		NormalizedResidual = tempWv
		tempWv = tempWv^2
		Chisquare = sum(tempWv)
		err=Chisquare/numpnts(BvectorOrig)
		numIter+=1
		NumberIterations=numIter
		
		insertpoints numIter,1, IterCount,errorVals
		IterCount[numIter]=numIter
		errorVals[numIter]=err
		redimension/n=(-1,(1+numIter)) ModelWaveIter
		ModelWaveIter[][numIter]=ModelWave[p]
		
		if (numIter<10)
			continue
		endif
	while((err>1 && numIter<MaxNumIterations))
	// this bails out from TNNLS if:
		//1. Chisquare/number of points is less than 1 (fit within uncertainties)
		//2. Number of iterations is above user input value
	
	ModelWaveOutput = ModelWave<=1e-10 ? 0 : ModelWave
	SetDataFolder OldDf
end
//======================================================================
