% PSV: PSV velocity correction.
% This program computes what PSV velocity data should be given simulation
% velocity data.  
%
% USAGE: [PM, PSV, PFORCE, converged] = psv(data) 
% returns the particle motion (PM) data and particle-streak velocimetry 
% (PSV) data as well as three forces acting on a particle at each point.  
% PM contains three columns.  The first column is the position of the 
% particle.  The second is the velocity of the particle at that point and 
% the third column is the acceleration of the particle at that point.  PSV 
% contains two columns.  The first column is the position where the 
% computed PSV velocity is defiend and the second column is the 
% corresponding velocity data.  PFORCE contains three columns: Stokes drag, 
% Thermophoretic force and gravity.  Each row corresponds to the first 
% column of PM where particle locations are specified.  converged is a
% single integer value that returns 1 if the integration was successful and
% the particle reached to the end of the domain and 0 otherwise.  The 
% function takes one to five input arguments.  The first input is the
% simulation data, which is mandatory and must be arranged in the following
% order: [x u V T r M mu] where the first column (x) is the axial position,
% u is the axial velocity, V is the spreading rate, dv/dr, T is the 
% temperature, r is the density, M is the average molecular weight and mu 
% is the mixture gas viscosity. It is assumed the data is stored such that 
% it is an increasing order in x.  The second to sixth inputs are optional.
% The second input argument can by 'Al2O3', 'Zeeo' or user-specified 
% particle properties.  When particle properties are specified, it must be 
% 1x3 cell with the first cell being the particle diameter (m), the second 
% being the particle density (Kg/m^3) and the last one is the thermal 
% conductivity of the particle.  This can be a single number (constant
% thermal conductivity), or an array with the first column being 
% temperature and the second column being the corresponding thermal 
% conductivity at the given temperature.  (See example below)  When
% particle data is omitted, 'Zeeo' is used by default.  The third to sixth 
% parameters are, the chopping frequency of the PSV, max iteration of
% temporal integration of particle motion, the number of subiterations 
% within each PSV chopping cycle and the integration order (1 or 2).  When 
% omitted, 1600, 100000, 100 and 2 are used by default respectively. 
%
% example 1:
% psv(simdata, {1.0e-5, 1000, 1.0}): simulates a particle whose diameter is
% 10 microns, density is 1000 (kg/m^3), and the thermal conductivity is 1
% (W/m-K) at all temperature.
%
% example 2:
% psv(simdata, {3.0e-6, 2400, [300 100.0; 2000 1000.0]}: simulates a
% particle whose diameter is 3 microns, density is 2400 (kg/m^3), and the
% thermal conductivity is 100 (W/m-K) at 300K and 1000 (W/m-K) at 2000 K.  
% At other temperature, the conductivity is linearly interpolated or
% extrapolated.
%
% example 3:
% psv(simdata, 'Al2O3', 2400, 250000, 200) : simulates a motion of Al2O3
% particle with PSV chopping frequency of 2400.  It tries to iterate up to
% a quater of million iterations at a time step of 1/2400 x 1/200.
%
% For more details, Jeff Bergthorson's thesis should be consulted.
%
% AUTHORS: Jeff M. Bergthorson, Laurent Benezech, and Kazuo Sone.
%
function [PM, PSV, PFORCE, converged] = psv(flowData, varargin)

narg = size(varargin,2);

if (nargin < 1)
   error('psv:NumInputs','Not enough input arguments.');
end

%%% Parameter definitions. %%%
if narg < 1
    % When particle is not specified, use 'Zeeo' as default.
    param = 'Zeeo';
else
    param = varargin{1};
end

% Particle properties, {diameter, density, conductivity}.

if iscell( param )
    % This is the case when property is specified by user.
    % dpart: Diameter of the particle.
    dpart = param{1};
    % rhopart: Particle density in kg/m^3
    rhopart = param{2};
    pcond = param{3};
else
    % Use preset values.
    if strcmp( param, 'Al2O3' )
        dpart = 1.0e-6;     % (m)
        rhopart = 3830.0;   % (Kg/m^3)
        %Thermal conductivity of particles Incropera and deWitt - Fundamentals of
        %Heat and Mass Transfer, 3rd edition 1990
        pcond = [200 55;
            400 26.4;
            600 15.8;
            800 10.4;
            1000 7.85;
            1200 6.55;
            1500 5.66;
            2000 6;
            2500 6.4];
    elseif strcmp( param, 'Zeeo' )
        dpart = 3.0e-6;     % (m)
        rhopart = 2400.0;   %(kg/m^3)
        pcond = 2.3;        % (W/mK)
    else
        % Error!
        error('psv:ParticleInfo','Particle specification incorrect.');
    end
end

% frequency: Chopping frequency of PSV.
if narg > 1
    frequency = varargin{2};
else
    frequency = 1600;
end

% max_iterations: Maximum number of integration steps.
if narg > 2
    max_iterations = varargin{3};
else
    max_iterations=100000;
end

% nsubcycle: number of iterations to get 1 PSV cycle.
% Default value is 100.
if narg > 3
    nsubcycle = uint16( varargin{4} );
else
    nsubcycle = 100;
end

% integorder: Order of time integration (1 or 2).
% Default value is 2.
% Use 1st order only when speed is important.
if narg > 4 
    integorder = uint16( varargin{5} );
    if integorder < 1 || integorder > 2
        error('psv:IntegOrder','Incorrect integration order.  Must be 1 or 2');
    end
else 
    integorder = 2;
end

% deltat: Integration of particle equations.
deltat = 1/frequency * 1/double(nsubcycle);
% halfdeltat: 1/2 deltat (used in 2nd order method)
halfdeltat = 0.5 * deltat;

%%% Universal constants. %%%
% Using the simulated flow, temperature and property fields, we need to model a particles behavior as it
% traverses the flowfield.
%Gravitational acceleration
a_gravity=9.81; %m/s^2

% mpart: Mass of the particle
mpart=rhopart*pi*4/3*(dpart/2)^3; %in kg;
%
% End of parameters.
%

xsim=flowData(:,1);
lsim=max(xsim); % End of the simulation domain.

usim=flowData(:,2);
Vsim=flowData(:,3); %Vsim is not equal to -1/2du/dz, except at the cold inlet!
T=flowData(:,4);
rho=flowData(:,5);
Mbar=flowData(:,6);
visc=flowData(:,7);

gradT=zeros(size(flowData,1),1);
for i=2:1:size(flowData,1)-1;
    gradT(i)=(T(i+1)-T(i-1))/(xsim(i+1)-xsim(i-1));
end
gradT(size(flowData,1))=gradT(size(flowData,1)-1);

%
%Initialize the particle with a corrected particle velocity and
%acceleration
i = 1;
tpart(1)=0;
xpart(1)=0;

% CKWp: approximately 1+1.142*Kn for small Kn.
%     : see Eqn.(A.10) in Jeff Bergthorson's thesis.
CKWp(1)=1.05; 

% vpart: Eqns.(A.23) & (A.24) in Jeff Bergthorson's thesis 
%      : vpart/vfluid = 1/(1+C_KW*tau_S*sigma_f)
%      : sigma_f = du/dx.
dudx = -2*Vsim(1);
vpart(1)=usim(1)/(1+CKWp(1)*(rhopart*dpart^2/(18*visc(1)))*dudx); 
% ufp: fluid velocity at particle location.
ufp(1)=usim(1);
% apart(1)=dudx*usim(1) %du/dt=du/dx*dx/dt, dx/dt=u, therefore du/dt=u*du/dx.  du/dx(0)=-2V(0)
[apart(1), F_thermophoretic(1), F_stokes(1)] = Particle_acceleration(mpart, dpart, pcond, vpart(1), [ufp(1) T(1) gradT(1) rho(1) Mbar(1) visc(1)]);

%The gravitational force is a constant and can be calculated prior to the
%loop
F_gravity=-mpart*a_gravity;

continue_iteration = 1;
while continue_iteration
    % Increment time by deltat, and update x, u,and a through 1st order 
    % explicit Euler method of integration.
    i = i + 1;
    tpart(i)=tpart(i-1)+deltat;
    xpart(i)=xpart(i-1)+deltat*vpart(i-1);
    vpart(i)=vpart(i-1)+deltat*apart(i-1);
    
    % Interpolate the simulated fields at the particle location.
    flow_at_particle = interp1(xsim, [usim T gradT rho Mbar visc], xpart(i), [], 'extrap');

    if integorder == 2
        apart(i) = Particle_acceleration(mpart, dpart, pcond, vpart(i), flow_at_particle);

        % Apply second order correction (2nd order Runge-Kutta method)
        xpart(i) = xpart(i-1) + halfdeltat*(vpart(i-1)+vpart(i));
        vpart(i) = vpart(i-1) + halfdeltat*(apart(i-1)+apart(i));
        
        % Interpolate the simulated fields at the particle location.
        flow_at_particle = interp1(xsim, [usim T gradT rho Mbar visc], xpart(i), [], 'extrap');
    end
    
    [apart(i), F_thermophoretic(i), F_stokes(i)] = Particle_acceleration(mpart, dpart, pcond, vpart(i), flow_at_particle);
    
    % Check if the particle has reached to the wall close enough.
    % and if so, exit the loop.
    % Note xpart has been computed way above here, but I have to get apart
    % otherwise the resulting array does not have the same number of rows.
    if xpart(i) >= lsim || i > max_iterations
        continue_iteration = 0;
    end
end

% Issue an warning message if not converged.
if i > max_iterations
    converged = 0;
    warning('Iteration may not have converged.  Increase maximum number of iterations.');
else
    converged = 1;
end
    
%now want to estimate the PSV velocity from this flowfield.
nb0 = nsubcycle+1;
nb1 = size(tpart,2);
ne0 = 1;
ne1 = size(tpart,2)-nsubcycle;
PSV(:,1) = (xpart(nb0:nb1)+xpart(ne0:ne1)) * 0.5;
PSV(:,2) = (xpart(nb0:nb1)-xpart(ne0:ne1)) * frequency;

% Assign to the returning matrix.
PM = [xpart; vpart; apart]';
PFORCE = [F_stokes; F_thermophoretic; F_gravity*ones(1,i)]';

%-------------------------
% Private functions
%-------------------------

%-------------------------------------------------------------------------
% Particle_acceleration: This function computes a particle acceleration.
function [apart, F_thermophoretic, F_stokes] = Particle_acceleration(mpart, dpart, pcond, vpart, flow_p)
   
ufp = flow_p(1);
Tp = flow_p(2);
gradTp = flow_p(3);
rhop = flow_p(4);
Mbarp = flow_p(5);
viscp = flow_p(6);
 
% R_univ: Universal gas constant.
R_univ=8314; %J/kmolK

% kftransp: The tranlational part of the thermal conductivity
%         : See Eqn.(A.26) in Jeff's thesis.
kftransp = 15*R_univ*viscp/(4*Mbarp);

% kpp : Thermal conductivity of the particle.  When it is not a function of
% temperature, use it.  If it is a function of temperature, pcond should be
% given by a table - simply interpolate.
if length(pcond) == 1
    kpp = pcond;
else
    kpp = interp1(pcond(:,1), pcond(:,2), Tp, [], 'extrap');
end
    
% Now calculate the parameters needed for the forces:
% Knp: Knudsen number.
Knp = Knudsen_number(Tp, Mbarp, viscp, rhop, dpart, R_univ);

% CKWp: Eqn.(A.10)
CKWp = 1+Knp*[1.142+0.558*exp(-0.999/Knp)];

%Now calculate the thermophoretic and stokes drag forces:
F_thermophoretic = Force_Thermophoretic_Talbot(viscp, dpart, rhop, kftransp, kpp, Knp, gradTp, Tp);
F_stokes = Force_Stokes(viscp, dpart, vpart, ufp, CKWp);

%Calculate the acceleration: Eqn.(A.7)
apart = (F_stokes+F_thermophoretic)/mpart; 
%-------------------------------------------------------------------------

%-------------------------------------------------------------------------
% Force_Stokes: This function computes stokes drag.
function [F_stokes] = Force_Stokes(viscp, dpart, vpart, ufp, CKWp)

%Stokes drag: Eqn.(A.9)
F_stokes = -3*pi*viscp*dpart*(vpart-ufp)/CKWp;
%-------------------------------------------------------------------------

%-------------------------------------------------------------------------
% Force_Thermophoretic_Talbot: This function computes thermophoretic force
% by Talbot's model.
function [F_tp] = Force_Thermophoretic_Talbot(viscp, dpart, rhop, kftransp, kpp, Knp,gradTp, Tp)

%Talbot: Eqn.(A.28)
num = -(6*pi*viscp^2/rhop*dpart*1.17*(kftransp/kpp+2.18*Knp)*gradTp/Tp);
den = (1+3*1.14*Knp)*(1+2*kftransp/kpp+2*2.18*Knp);
F_tp = num/den;
%-------------------------------------------------------------------------

%-------------------------------------------------------------------------
% Knudsen_number: This function computes Knudsen number
function [Knp] = Knudsen_number(Tp, Mbarp, viscp, rhop, dpart, R_univ)

% cbarp: Eqn.(A.12)
cbarp = (8*R_univ*Tp/(Mbarp*pi))^0.5;

% Lambdap: Eqn.(A.11)
Lambdap=2*viscp/(rhop*cbarp);

% Knp: Knudsen number.
Knp = 2*Lambdap/dpart;
%-------------------------------------------------------------------------



