function C = CTColoredNoise(sig,param,S,Cnorm)
% CTColoredNoise -- Chirplet Transform in the case of colored noise.
%  Usage
%    C = CTColoredNoise(sig,param,S,Cnorm);
%  Input
%    sig     signal of length n, n=2^J, where J is an integer
%    param   chirplet graph parameters as returned by
%            GetChirpletGraphParam
%    S       In the case of circular stationary process, this should be the 
%            spectrum of the noise process.
%            For non-circular stationary process, this should be the 
%            inverse of the covariance matrix of the sampled process
%            PEND!! If the covariance matrix is badly conditioned, should
%            we use its singular value decomposition instead?
%    Cnorm   Normalizing constants for chirplets assuming stationary Gaussian noise
%            with a particular spectrum/covariance matrix. 
%            For circular statinary noise, Cnorm should be calculated using
%            the function ChirpNormColoredNoise.m.
%            For non-circular statinary noise, Cnorm should be calculated using
%            the function ChirpNormColoredNoiseNonCirc.m.
%            NOTE: Has to be calculated with the same spectrum/covariance
%            matrix as in S
%  Outputs
%    C       chirplet coefficients, stored in a cell array.
%
%  See Also
%    - ChirpletTransform
%    - ChirpNormColoredNoise, ChirpNormColoredNoiseNonCirc
%    - GetChirpletGraphParam for more information on customizing the chirplet transform.
%    - GetChirpletNetwork, CalculateStatistic, for setting up the 
%      chirplet graph and calculating the test statistic. 

if nargin < 4,
  errormsg = 'CTColoredNoise: Chirplet norms have to be provided!';
  error(errormsg);
elseif nargin < 3,
  errormsg = 'CTColoredNoise: Spectrum (or inverse of covariance in case of non-circular noise) and chirplet norm parameters have to be provided!';
  error(errormsg);
elseif nargin < 2,
  errormsg = 'CTColoredNoise: Graph parameters have to be provided!';
  error(errormsg);
end

% ensure that sig is a row vector and change
% it if it isn't
x = reshape(sig,1,length(sig));

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

% PEND! Make the number of chirplets/cells be one of the parameters
%       returned by GetChirpletGraphParam. Avoids chirplet graph
%       specific things to be calculated at more than one place.
%nCells = 2^(finestScale+1)-2^(coarsestScale);
nCells = 2^(finestScale+1) - 1; % would need to change node.m if the
                                % stuff in previous line is used
C = cell(1,nCells);

% Weight data by inverse of noise spectrum and multiply by mask
% PEND! Ensure that the masking works in the case of WSS non-circular noise

[nrows,ncols] = size(S);

% check which type of noise
if (param{8}==0),
  % the noise is assumed to be circular stationary, and reshaped to be a row
  D = reshape(1./S, 1, length(S));
  y = ifft(D.*fft(x).*param{7});
else
  % the noise is assumed to be non-circular stationary
  y = S*(x.'); % x is a row vector, taking the transpose 
  y = y.';     % return y as a row vector
end

for s = finestScale:-1:coarsestScale,
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);
  t = 0:(2^(J-s)-1);  
  for b = 0:(2^s-1),
    ix = dyadindex(s,b,n);
    % Modulation followed by zero-padding
    X = repmat(y(ix),nSlopes,1) .* exp(-i * pi*slopes.' * t.^2/n);
    X = [X.'; zeros(n - length(ix),nSlopes)];
    c = fft(X); % Operates along columns
    C{node(s,b)} = c./Cnorm{1,s+1}; % normalizing
    % NOTE: We could get away with dividing by length(ix) by taking care
    % of the in the code that calcultates the norm
  end
end;


% $RCSfile: CTColoredNoise.m,v $
% $Date: 2007/02/12 23:41:16 $
% $Revision: 1.11 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
