function C = CTPlain(sig,param)
% CTPlain -- Calculates plain chirplet coefficients of a signal
%            
%  Usage
%    C = CTPlain(sig,param);
%  Inputs
%    sig     signal of dyadic length N=2^J, where J is a positive integer
%    param   chirplet graph parameters as returned by GetChirpletGraphParam
%  Outputs
%    C       chirplet coefficients, stored in a cell array.
%
%  Description
%    A chirplet transform that assumes white noise and constant amplitude.
%    See the documentation for ChirpletTransform for a description of how 
%    the chirplet coefficients are stored.
%
%  See Also
%    - ChirpletTransform
%    - GetChirpletGraphParam for information on customizing 
%      the chirplet transform.
%    - GetChirpletNetwork, CalculateStatistic, for setting up the 
%      chirplet graph and calculating the test statistic. 

if (nargin < 2),
  errormsg = 'CTPlain: Chirplet graph parameters have to be provided. Usage: C = CTPlain(sig,param);';
  error(errormsg);
end

% ensure that sig is a row vector and change it if it isn't
x = reshape(sig,1,length(sig));

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

% PEND! Make the number of chirplets/cells be one of the parameters
%       returned by GetChirpletGraphParam. Avoids chirplet graph
%       specific things to be calculated at more than one place.
%nCells = 2^(finestScale+1)-2^(coarsestScale);
nCells = 2^(finestScale+1) - 1; % would need to change node.m if the
                                % stuff in previous line is used
C = cell(1,nCells);

for s = finestScale:-1:coarsestScale,
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);
  t = 0:(2^(J-s)-1);
  chirppart = exp(-i * pi*slopes.' * t.^2/n);
  for b = 0:(2^s-1),
    % handle dyadic interval (s,b)
    ix = dyadindex(s,b,n);
    % Modulation followed by zero-padding
    X = repmat(x(ix),nSlopes,1) .* chirppart;
%    X = [X.'; zeros(n - length(ix),nSlopes)];
%    c = fft(X); % Operates along columns
    c = fft(X.',n); % Operates along columns
    c = c/sqrt(length(ix)); % normalize
    C{node(s,b)} = c;
  end
end


% $RCSfile: CTPlain.m,v $
% $Date: 2006/09/22 18:11:35 $
% $Revision: 1.4 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2005
