function Cnorm = ChirpNormColoredNoise(S,param)
% ChirpNormColoredNoise -- Calculates x'Sigma^(-1)x, for chirplets x
%                          given a noise spectrum.
%  Usage
%    Cnorm = ChirpNormColoredNoise(S,param);
%  Inputs
%    S       spectrum of noise process
%    param   chirplet graph parameters as returned by
%            GetChirpletGraphParam
%  Outputs
%    Cnorm  a cell structure storing x'Sigma^(-1)x for all
%           chirplets for a signal of length N=2^J (given by the chirplet graph param).
%           
% Description
%   Calculates x'Sigma^(-1)x for all the chirplets x in a chirplet graph.
%   Theses coefficients are stored in a J-by-1 cell structure, where
%   cell s corresponds to scale s and is an nx-by-n matrix (nx is the
%   number of slopes allowed at this scale, equal to 2^(J-s) ). The
%   entries in the matrix are as follows:
%        -each column corresponds to one slope
%        -each entry in a column corresponds to an offset
%
%  See Also
%    - ChirpletTransform
%    - CTColoredNoise
%    - GetChirpletGraphParam for more information on customizing the chirplet transform.
%    - GetChirpletNetwork, CalculateStatistic, for setting up the 
%      chirplet graph and calculating the test statistic. 

if nargin < 2,
  errormsg = 'ChirpNormColoredNoise: Chirplet graph parameters have to be provided!';
  error(errormsg);
end

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

% inverse of noise spectrum, reshaped to be a row vector if necessary
D = reshape(1./S, 1, length(S)); 
Cnorm = cell(1,J);

for s = finestScale:-1:coarsestScale,,
  nx = 2^(J-s);
  t = 0:nx-1;

  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);

  % Modulation followed by zero-padding
  phi = pi*slopes.'*t.^2/n; 
  X = exp(i * phi)/sqrt(nx);  
  % NOTE: Each column in X corresponds to one slope.
  X = [X.'; zeros(n - nx,nSlopes)];
  Xfft = abs(fft(X)/sqrt(n)).^2;
  XfftFlip = flipud(Xfft);

  % Do a circular convolution with D and Xfft flipped
  Dfft = fft(D');
  c = repmat(Dfft,1,nSlopes).*fft(XfftFlip); 
  c = ifft(c);
  Cnorm{s+1} = circshift(sqrt(abs(c)*nx),1); % have to do a shift to index right

  % taking the absolute value to
  % force the value to be real and positive in case
  % of some numerical errors.
end


% $RCSfile: ChirpNormColoredNoise.m,v $
% $Date: 2007/02/12 23:41:16 $
% $Revision: 1.7 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
