function C = ChirpletTransform(sig,param,S,CNWeights)
% ChirpletTransform -- configurable chirplet transform
%
%  Usage
%    C = ChirpletTransform(sig,param,S,CNWeights)
%    C = ChirpletTransform(sig,param,S)
%    C = ChirpletTransform(sig,param)
%    C = ChirpletTransform(sig)
%  Inputs
%    sig        signal of dyadic length n=2^J, where J is a positive integer
%    param      chirplet graph parameters as returned by GetChirpletGraphParam.
%               If omitted the default chirplet graph as returned by GetChirpletGraphParam
%               will be used.
%    S          spectrum of noise process
%    CNWeights  weights to use for colored noise 
%               For constant amplitude and colored noise, this should be 
%               the weights given by ChirpNormColoredNoise using the same spectrum S
%               For varying amplitude and colored noise, this should be
%               the weights given by ColNoiseVarAmpWeightMatrix.
%               The helper function ChirpNorm always returns weights appropriate 
%               for the type of transform and can be used instead of these functions.
%               If S is provided but CNWeights is not, CNWeights will be calculated as
%               well. Note however that weights usually only need to be calculated once
%               in advance and can then be used for any input signal as long as the
%               chirplet parameters are the same. It is recommended that weights be
%               calculated in advance using ChirpNorm.
%  Outputs
%    C       chirplet coefficients, stored in a 1d cell array. 
%            The length of the array is equal to the number of dyadic time intervals.
%
%  Description
%    Using the chirplet graph parameters, the chirplet graph can be configured and 
%    the type of chirplet transform can be set (plain, varying amplitutude etc.). 
%    For configuring the transform see the documentation in GetChirpletGraphParam.
%
%    The data structure used for storing the chirplet coefficients is a Matlab cell array.
%    Each element in the array is a table of chirplet coefficients corresponding 
%    to a dyadic time interval [k 2^(-j), (k+1)2^(-j)], labeled (j,k),
%    where k=0,1,...,2^j-1 and j is a scale index which can range from 0 to J-1.
%    A table corresponding to a dyadic interval (j,k) is indexed by (2^j + k)
%    and is an n-by-(#slopes at scale j) matrix whose values are described as follows:
%      -each column in the table corresponds to one slope,
%       column k corresponds to the k-th slope
%      -each entry in a column corresponds to a frequency offset,
%       the m-th entry in each column corresponds to the frequency offset 2*pi*(m-1)/n
%
%  See Also
%    - GetChirpletGraphParam for information on how to customize the chirplet transform.
%    - GetChirpletNetwork, CalculateStatistic, for setting up the 
%      chirplet graph and calculating the test statistic. 
%    - ChirpNorm for calculating chirplet weights

if nargin < 2,
  % use default chirplet graph as returned by GetChirpletGraphParam
  param = GetChirpletGraphParam(length(sig));
end

if nargin < 3,
  if ( strcmpi(param{6},'COLOREDNOISE') | strcmpi(param{6},'COLOREDNOISEVARAMP') ),
    errormsg = 'ChirpletTransform. Spectrum has to be provided in the case of colored noise';
    error(errormsg);
  end
end

if nargin == 3,
  % spectrum provided but chirplet norm has not
  if (length(S)~=param{1}(1)),
    errormsg = 'ChirpletTransform. Spectrum has to be of same length as signal.';
    error(errormsg);
  end
  % norms were not provided, calculate them
  if strcmpi(param{6},'COLOREDNOISE'),
    CNWeights = ChirpNormColoredNoise(S,param);
  elseif strcmpi(param{6},'COLOREDNOISEVARAMP')
    CNWeights = ColNoiseVarAmpWeightMatrix(S,param);
  else
    errormsg = 'ChirpletTransform. Spectrum was provided but xttype is for white noise. To calculate transform for colored noise, set the parameter xttype in GetChirpletGraphParam to the appropriate value.';
    error(errormsg);
  end
end

% check the transform type
switch param{6},
  case 'PLAIN'
    % do a plain transform
    C = CTPlain(sig,param);

  case 'VARAMP'
    % do a varamp chirplet transform
    C = CTVarAmp(sig,param);
 
  case 'COLOREDNOISE'
    % do a colored noise chirplet transform
    C = CTColoredNoise(sig,param,S,CNWeights);

  case 'COLOREDNOISEVARAMP'
    % do a colored noise chirplet transform with varying amplitude
    C = CTCNVA(sig,param,S,CNWeights);

  otherwise
    errormsg = 'Transform not supported. See GetChirpletGraphParam for supported transforms.';
    msgid = 'ChirpletTrans:ChirpletTransform:TransformNotSupported';
    error(msgid,errormsg);
end

% $RCSfile: ChirpletTransform.m,v $
% $Date: 2007/02/12 23:41:16 $
% $Revision: 1.19 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2005
