% An example showing how to apply the code to searching for
% the signal produced in a LIGO detector by a binary inspiral.
%
% Steps:
% 1) Generate a LIGO noise spectrum
% 2) Generate a binary inspiral signal
% 3) Configure the chirplet graph and do a chirplet transform
% 4) Fill in chirplet costs in network
% 5) Calculate the BP for prescribed path lengths test statistic
% 6) Plot chirplet paths corresponding to the lengths
%    chosen for the BP test statistic.

% Reseed the random number generator
rand('state', sum(100*clock));

J = 9;
N =  2^J;       % length of signal in samples

% Set the sampling rate
Fs = 2048.0; % Hz

% Set the scaling factor to bring the strain values to a more
% reasonable level
kappa = 1.0E+20;

% Binary inspiral parameters
% Time of coalescence (seconds)
tc = 0.25;

% Time values to calculate the inspiral at
t = [0:N-1]*(1/Fs);

% Frequency values from DC to Nyquist (Hz)
f = [0:N/2]*(Fs/N);

% Masses of the two bodies (solar mass)
m1 = 14.0;
m2 = 14.0;

% Strength of signal
SNR = 20;

% Settings for test statistic
STATTYPE = 'BPFORPLOTTING'; % Calculates the BP statistic and returns chirplet paths for plotting
maxLength = 8;              % Calculates BP for lengths 1,...,maxLength

% generate a LIGO noise spectrum in units strain^2/Hz, then multiply
% by the sampling rate to get the power spectrum in strain^2
[S, ff, Sh, fh] = MakeLIGO1Psd(N, Fs);
S = kappa^2*Fs*S;
clear ff;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating binary inspiral chirp...');

% h is the gravitational wave strain at the detector at time t,
% p is the phase at time t and fr is the instantaneous fruquency
% of the signal in Hz at time t
[h, p, fr] = inspiral(t, m1, m2, tc);

% Calculate the spectrally-weighted norm of h and normalise it
hn = sqrt((1/N)*sum(abs(fft(h)).^2./S));
h = h/hn;

% plot signal and instantaneous frequency
figure;
subplot(3,1,1);
plot(t, h/kappa);
title(sprintf('Binary inspiral signal for m_1 = %.1f M_0, m_2 = %.1f M_0', m1, m2));
xlabel('time (s)');
ylabel('h(t)');

subplot(3,1,2);
plot(t, fr);
title('Instantaneous frequency of inspiral');
xlabel('time (s)');
ylabel('f (Hz)');

subplot(3,1,3);
semilogx(fh, 0.5*log10(Sh));
title('LIGO-I design noise floor');
xlim([10 Fs/2 ]);
xlabel('frequency (Hz)');
ylabel('PSD (strain Hz^{-1/2})');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM AND GRAPH CONFIGURATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Configuring graph...');
csc = 0; % coarsest scale
fsc = 6; % finest scale
transformType = 'COLOREDNOISE'; % set the transform type to colored noise
fmin = 0;
fmax = N/2;
graphparam = GetChirpletGraphParam(N,csc,fsc,[],[],fmin,fmax,transformType);
disp('Calculating normalization coefficients for the colored noise...');
Cnorm = ChirpNormColoredNoise(S,graphparam);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Generate an instance of real noise
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating noise...');

n = MakeRealNoise2(S);
y = n + SNR*h;

% plot just noise and signal+noise
figure;
subplot(2,1,1);
plot(t,n/kappa);
title('Just noise');
xlabel('time (s)');
ylabel('n(t)');

subplot(2,1,2);
plot(t,y/kappa);
title(sprintf('Noise with binary inspiral signal for m_1 = %.1f M_0, m_2 = %.1f M_0', m1, m2));
xlabel('time (s)');
ylabel('y(t)');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Calculating chirplet coefficients...');
cc = ChirpletTransform(y,graphparam,S,Cnorm); 
disp('Generating chirplet graph and assigning costs...');
cnetwork = GetChirpletNetwork(cc,graphparam);
disp('Running optimization routine for graph...');
[costpath,bestpaths] = CalculateStatistic(cnetwork,STATTYPE,maxLength);
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
figure;
% Plot paths corresponding to dyadic number of chirplets, 1,2,4,...,2^(floor(log2(maxLength)))
nplots = floor(log2(maxLength))+1;
for k= 0:nplots-1,
  subplot(nplots,1,k+1);
  DisplayChirpletPath(bestpaths{2^k},graphparam); % plot the best path
  
  title(strcat('Best path restricted to #chirplets=',num2str(2^k),...
                   ', |cost|=',num2str(abs(costpath(2^k)))));
  hold on;
  % plotting the `instantaneous frequency'
  plot(0:(length(fr)-1),fr*N/Fs);
%  axis([0 N fmin fmax]);
  hold off;
end
