% An example showing how to use the colored noise chirplet transform.
% It also uses the BP test statistic.
%
% Steps:
% 1) Generate a chirping signal
% 2) Configure the chirplet graph and do a chirplet transform
% 3) Fill in chirplet costs in network
% 4) Calculate the BP for prescribed path lengths test statistic
% 5) Plot chirplet paths corresponding to the lengths
%    chosen for the BP test statistic.
disp('--Running ColoredNoiseDemo.m--');

J = 9;
N =  2^J;       % length of signal

% Settings for test statistic
STATTYPE = 'BPFORPLOTTING'; % Calculates the BP statistic and returns chirplet paths for plotting
maxLength = 8;              % Calculates BP for lengths 1,...,maxLength

% generate a noise spectrum
a=0.9; tmp = 1./(1-2*a*cos((0:N/2)/N) + a^2);
S = [tmp fliplr(tmp(2:N/2))];
S = sqrt(N)/norm(S) * S; % normalize S so it has l2-norm sqrt(N)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirp...');
[y,omega]= MakeChirp('CubicPhase',N);
y = y/norm(y); % normalize so the signal has l2-norm 1

% plot real part of signal and instantaneous frequency
figure;
subplot(3,1,1);plot(real(y));
xlabel('time');ylabel('Real Part of Signal');
subplot(3,1,2);plot(0:N-1,omega);
xlabel('time');ylabel('frequency');
subplot(3,1,3);plot(0:N/2-1,S(1:N/2));
xlabel('frequency');ylabel('noise spectrum');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM AND GRAPH CONFIGURATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Configuring graph...');
csc = 0; % coarsest scale
fsc = 6; % finest scale
transformType = 'COLOREDNOISE'; % set the transform type to colored noise
graphparam = GetChirpletGraphParam(N,csc,fsc,[],[],[],[],transformType);
disp('Calculating normalization coefficients for the colored noise...');
Cnorm = ChirpNormColoredNoise(S,graphparam);
disp('Calculating chirplet coefficients...');
cc = ChirpletTransform(y,graphparam,S,Cnorm); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirplet graph and assigning costs...');
cnetwork = GetChirpletNetwork(cc,graphparam);
disp('Running optimization routine for graph...');
[costpath,bestpaths] = CalculateStatistic(cnetwork,STATTYPE,maxLength);
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
figure;
% Plot paths corresponding to dyadic number of chirplets, 1,2,4,...,2^(floor(log2(maxLength)))
nplots = floor(log2(maxLength))+1;
for k= 0:nplots-1,
  subplot(nplots,1,k+1);
  DisplayChirpletPath(bestpaths{2^k},graphparam); % plot the best path
  
  title(strcat('Best path restricted to #chirplets=',num2str(2^k),...
                   ', |cost|=',num2str(abs(costpath(2^k)))));
  hold on;
  % plotting the `instantaneous frequency'
  plot(0:(length(omega)-1),omega);
%  axis([0 N fmin fmax]);
  hold off;
end
