% An example showing how to find the BP test statistic and restrict
% the chirplet graph to non-negative slopes.
%
% Steps:
% 1) Generate a chirping signal
% 2) Set up chirplet graph parameters and slope discretization
% 3) Do a chirplet transform
% 4) Fill in chirplet costs in network
% 5) Calculate the BP for prescribed path lengths test statistic
% 6) Plot chirplet paths corresponding to the lengths
%    chosen for the BP test statistic.
disp('--Running FindBPposSlopesDemo.m--');

J = 9;          % dyadic length of signal
N =  2^J;       % length of signal

% Settings for test statistic
STATTYPE = 'BPFORPLOTTING'; % Calculates the BP statistic and returns chirplet paths for plotting
maxLength = 5;              % Calculates BP for lengths 1,...,maxLength

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirp...');
[y,omega]= MakeChirp('CubicPhase',N);
y = y/norm(y); % normalize so the signal has l2-norm 1

% plot real part of signal and instantaneous frequency
figure;
subplot(2,1,1);plot(real(y));
xlabel('time');ylabel('Real Part of Signal');
subplot(2,1,2);plot(omega);
xlabel('time');ylabel('frequency');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM AND GRAPH PARAMETERS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Taking chirplet transform...');
discrStepFactor = 4; % multiplicative discretization factor for slopes
minslope = 0;        % minimum slope in chirplet graph
maxslope = 0.5;      % maximum slope in chirplet graph
graphparam = GetChirpletGraphParam(N,[],[],discrStepFactor,[minslope maxslope]);
cc = ChirpletTransform(y,graphparam); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirplet graph and assigning costs... (this will take a while)');
cnetwork = GetChirpletNetwork(cc,graphparam);
disp('Running optimization routine for graph...');
[costpath,bestpaths] = CalculateStatistic(cnetwork,STATTYPE,maxLength);
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
npaths = length(bestpaths);
figure;
for k=1:npaths,
  subplot(npaths,1,k);
  DisplayChirpletPath(bestpaths{k},graphparam); % plot the best path
  
  title(strcat('Best path restricted to #chirplets=',num2str(k),...
                   ', |cost|=',num2str(abs(costpath(k)))));
  hold on;
  plot(0:(length(omega)-1),omega);
  axis([0 N 60 200]);
  hold off;
end
