% COMPLEXINSPIRAL Simulate a complex gravitational wave inspiral
%
%   [h, p, f, Ac, Ap] = complexinspiral(t, m1, m2, tc, [ w, [ f0 ] ])
%
% Inputs:
%   t      - vector of times at which to calculate h (seconds)
%   m1, m2 - masses of bodies (solar masses)
%   tc     - time from t = 0 to coalescence (seconds)
%   w      - (optional) angle of inclination of system to Earth
%              (default 0 radians)
%   f0     - optional reference frequency (Hz)
%              (default 0 Hz)
%
% Outputs:
%   h - normalised inspiral waveform (dimensionless strain)
%   thephase - (optional) phase at t (radians)
%   theamplitude - (optional) amplitude of the signal
%
% This function uses the same phase and amplitude as a real
% inspiral but generates the complex envelope of a real inspiral
% signal ie. A(t) exp(i phi(t)).
%
function [h, thephase, theamplitude]=complexinspiral(t, m1, m2, tc, w, f0)

  if (nargin < 6)
    f0 = 0;
  end;

  if (nargin < 5)
    w = 0;
  end;

  % Distance to source (metres). Doesn't matter
  % since we normalise the waveform anyway
  D = 1.0;

  p = phi(t, m1, m2, tc);

  Ap = Aplus(t, m1, m2, tc, w, D);
  [Ac, f] = Across(t, m1, m2, tc, w, D);

  %  h = Ap.*cos(p) + Ac.*sin(p);
  % note that Ap = (1 + cos(w)^2) * Ac/(2*cos(w))
  
  %  h = Ac.* ((1+cos(w)^2)/(2*cos(w)) * cos(p) + sin(p));

  theamplitude = abs(Ac/(2*cos(w))) * sqrt(cos(w)^4 + 6*cos(w)^2 + 1);
  phaseoffset = atan(-2*cos(w)/(1+cos(w)^2));
  thephase = p + phaseoffset;

  h = -theamplitude.*exp(i*thephase);

  % Zero out everything where t >= tc
  h(t >= tc) = 0;
  thephase(t >= tc) = 0;
  f(t >= tc) = 0;
  theamplitude(t >= tc) = 0;

  % Zero out everything where inst. frequency is < f0
  h(f < f0) = 0;
  thephase(f < f0) = 0;
  f(f < f0) = 0;
  theamplitude(f < f0) = 0;

  % Normalise the template
  h = h/norm(h);

return;

