% INSPIRAL Simulate binary inspiral
%
%   [h, p, f, Ac, Ap] = inspiral(t, m1, m2, tc, [ w, [ f0 ] ])
%
% Inputs:
%   t      - vector of times at which to calculate h (seconds)
%   m1, m2 - masses of bodies (solar masses)
%   tc     - time from t = 0 to coalescence (seconds)
%   w      - (optional) angle of inclination of system to Earth
%              (default 0 radians)
%   f0     - optional reference frequency (Hz)
%              (default 0 Hz)
%
% Outputs:
%   h - normalised inspiral waveform (dimensionless strain)
%   p - (optional) phase at t (radians)
%   f - (optional) instantaneous frequency at t (Hz) (f = 2*pi*p')
%   Ac - (optional) cross component of amplitude
%   Ap - (optional) plus component of amplitude
%
% This function simulates the gravitational wave signal from a binary
% inspiral where the two bodies have masses m1 and m2 respectively.
% The waveform is calculated in the time-domain using a power-series
% expansion for the phase phi(t) and amplitude, which is proportional to
% velocity-squared (in the case of negligible spin-spin interactions).
% The amplitudes of the plus- and cross-polarised components are calculated
% using a Newtonian approximation. The total gravitational wave strain
% is given by
%
%   h(t) = A_\plus(t) \cos(\phi(t)) + A_\cross(t) \sin(\phi(t))
%
% In this model of an inspiral, the waveform extends infinitely into the past,
% and coalescesces at some time t = tc. The returned waveform is normalised
% to be a unit vector in the sense that sum(abs(h).^2) = 1. It is also
% zeroed out for any t >= t, and any times where the instantaneous frequency
% is less than the reference frequency f0.
% In interferometric detection schemes, the reference frequency is usually
% taken to be at the point below which seismic disturbances would make
% detection impossible - around 40 Hz for LIGO.
% In order to see the complete evolution of the waveform,
% from the reference frequency f0 up to the coalescence at t = tc,
% the input vector of times t must be a set of values in order from 0 to tc.
% Although the waveform is set to zero for all t >= tc, in reality there
% would be significant gravitational signal after this time from the
% coalescence itself, but this is not accurately modelled.
%
% REFERENCES
%
% "LAL Software Documentation",
%    http://www.aei.mpg.de/research/grawave/software/lsd.pdf
%    especially section 16.3 and chapter 25
function [h, p, f, Ap, Ac]=inspiral(t, m1, m2, tc, w, f0)

  if (~exist('w'))
    w = 0.0;
  end;

  if (~exist('f0'))
    f0 = 0.0;
  end;

  % Distance to source (metres)
  D = 1.0;

  p = phi(t, m1, m2, tc);

  Ap = Aplus(t, m1, m2, tc, w, D);
  [Ac, f] = Across(t, m1, m2, tc, w, D);
  
  h = Ap.*cos(p) + Ac.*sin(p);

  % Zero out everything where t >= tc
  h(t >= tc) = 0;
  p(t >= tc) = 0;
  f(t >= tc) = 0;
  Ap(t >= tc) = 0;
  Ac(t >= tc) = 0;

  % Zero out everything where inst. frequency is < f0
  h(f < f0) = 0;
  p(f < f0) = 0;
  f(f < f0) = 0;
  Ap(f < f0) = 0;
  Ac(f < f0) = 0;

  % Normalise the template
  h = h/norm(h);

return;
