% INSPIRALSPA Simulate binary inspiral using stationary-phase approximation.
%
%   [h, t, tau0, tau1] = inspiralspa(m1, m2, srate, f0)
%
% Inputs:
%   m1, m2 - masses of bodies (solar masses)
%   srate - sampling rate (samples/second)
%   f0 - initial frequency of waveform (Hz)
%
% Outputs:
%   h - inspiral waveform sampled at srate samples/second
%   t - array of time steps (seconds)
%   tau0 - Newtonian contribution to coalescence time (seconds)
%   tau1 - first post-Newtonian contribution to coalescence time (seconds)
%
% This function simulates the gravitational wave signal from a binary
% inspiral where the two bodies have masses m1 and m2 respectively.
% The waveform is calculated in the frequency-domain using the stationary
% phase approximation to the time-domain waveform. The time-domain waveform
% h(t) is obtained via an inverse Fourier transform (this puts some artifacts
% into the waveform which would not be there if we generated the waveform
% in the time domain directly. That's a bit harder to do, so for the moment
% I just apply some filtering to reduce the artifacts).
%
% The waveform shows the evolution of the gravitational wave up to
% time time tc = tau0 + tau1 (time of coalescence). The quantity tc
% represents the time taken for the waveform to evolve from instantaneous
% frequency f0 until coalescence ie. the waveform turns on at t = 0 with
% instantaneous frequency f0, and coalesceses at t = tc.
function [h, t, tau0, tau1]=inspiralspa(m1, m2, srate, f0)

%%
%% Constants
%%

T0 = 4.925E-06;    % conversion factor to geometrised units (seconds)

%%
%% Derived values
%%

M = m1 + m2;        % total mass
eta = m1*m2/(M^2);  % symmetric mass ratio (unitless)

%
% These are the "chirp times" corresponding to this choice of masses,
% representing the contribution to time-of-coalescence from the
% Newtonian (tau0) and 1st post-Newtonian (tau1) terms of the expansion
%
% It takes tc = tau0 + tau1 seconds for the waveform to evolve from frequency
% f0 to coalescence. If you want to see the full wave form from f0 on,
% T should be chosen so that T > tc.
%
tau0 = 5/256 * (M*T0)^(-5/3) * eta^(-1) * (pi*f0)^(-8/3);
tau1 = 5/192 * (M*T0)^(-1) * (743/336*eta^(-1) + 11/4) * (pi*f0)^(-2);
tc = tau0 + tau1;

N = 2^nextpow2(tc*srate);  % number of samples
T = N/srate;               % actual length of sequence (seconds)
Nyq = srate/2.0;           % Nyquist frequency (Hz)
dt = 1/srate;              % time-step size (seconds)
df = srate/N;              % frequency-step size (Hz)

f = [1:N/2-1]*df;  % positive frequencies, up to but not including Nyq,
                   % used for calculating chirp as a function of frequency
                   % (zero excluded to prevent divide-by-zero later)

% Calculate the phase functions
phi0 = (3/5)*f0*(f/f0).^(-5/3);    % Newtonian
phi1 = f0*(f/f0).^(-1);            % 1st post-Newtonian

% Calculate the chirp in the frequency domain
chirp = f.^(-7/6).*exp(-i*2*pi*(tau0*phi0 + tau1*phi1));

c = [ 0 chirp ];   % Insert the DC component (wasn't done above to
                   % avoid divide-by-zero error)
f = [ 0 f ];

% Kill off the bad frequencies (probably should use a nice smooth
% window for this instead of a boxcar)
w = zeros(1, length(f));
w(f0 <= f) = 1.0;

c = w.*c;

% fake up the Fourier transform of the above chirp (inserting a Nyquist
% component which we just set to zero)
H = [ c 0 conj(c(end:-1:2)) ];

% transform to time-domain
h = real(ifft(H));

% kill everything below (T - tc), as it's junk (do this before filtering)
t = [0:N-1]*dt;
h = h((T-tc) <= t);

% correct vector of time steps (seconds)
t = [0:length(h)-1]*dt;

% filter out some of the low-frequency garbage with a high-pass
% Butterworth filter (really, need to generate waveforms properly!)
[b, a] = butter(4, 1.4*f0/Nyq, 'high');
h = filter(b, a, h);

% Filter out some of the high-frequency garbage with a low-pass
% Butterworth filter
[b, a] = butter(4, 0.98);
h = filter(b, a, h);

return;
