function [cost,paths] = CSPcell(G,n,startnodes,endnodes,maxLength)        
% CSPcell -- solves the shortest path problem for an acyclic graph with
%            a constraint on the maximum number of arcs that the path can have. 
%
%  Usage
%    [cost,paths] = CSPcell(G,n,startnodes,endnodes,maxLength)
%    cost = CSPcell(G,n,startnodes,endnodes,maxLength)
%  Inputs
%    G		a cell array of size (k,2) where k is the number of nodes in the graph
%               for which arcs go out from. cell (m,1) is a vector with all the nodes 
%               that node m connects to and cell (m,2) is a vector with the corresponding
%               arc costs
%    n          total number of nodes in graph.
%    startnodes	list of nodes from where paths can start in the acyclic network.
%    endnodes	list of nodes from where paths can end in the acyclic network.
%    maxLength	Maximum number of arcs that the best path is allowed to have.
%
%  Outputs
%    paths	a 1d cell array of length alpha. Entry k in the
%           cell array corresponds to the best path with k chirplets.
%           If there does not exist a path with k chirplets, for
%           example because of restriction of the coarsest allowable
%           scale, paths{k} will be empty.
%           Note that if cs is the coarsest allowable scale
%           then the minimum number of chirplets in a path is 2^cs.
%    cost	a vector of length maxLength where costs(k) is the value of
%           the best path with number of chirplets equal to k.
%
%  Description
%    This is an inner function in ChirpLab.
%    It is recommended to use CalculateStatistic instead.
%
%  See Also:
%    - CalculateStatistic
%    - CSPstatistic

if (nargout < 2),
  warningmsg = 'If you are only interested in the best costs see the documentation for CalculateStatistic for speeding up calculations.';
  warning(warningmsg);
end
  
nG = size(G,1);  % returns the number of starting nodes in G
% Assume that the nodes in G appear in topological order.
ord = 1:nG;

% Initialize the distance label matrix
d = inf*ones(maxLength+1,n);    

% all the startnodes have distance label 0
%d(:,startnodes) = 0;
d(1,startnodes) = 0;

if (nargout > 1),
  % if we are only interested in costs, we do not need
  % to keep predessor indices. Can save a bit of memory
  % by considering these two cases

  % Initialize the predecessor indices
  pred = zeros(maxLength+1,n);

  pred(1,startnodes) = startnodes;


  % Examine nodes in topological order
  for start = 1:nG,
    A = G{start,1};     % A is the set of endpoints and costs of arcs emanating out of node 'start'.
    L = G{start,2}; % costs
    for k = 1:length(A),
     for l=2:(maxLength+1),
        if (isinf(d(l-1,start))),
          % PEND: Could check for Inf values earlier and then
          % loop over the resulting set.
          continue;
        end
        if ( d(l,A(k)) > d(l-1,start)+L(k) )
          d(l,A(k)) = d(l-1,start)+L(k);
          pred(l,A(k)) = start;
        end
      end
    end
  end

else
  % we are only interested in costs

  % Examine nodes in topological order
  for start = 1:nG,
    A = G{start,1};     % A is the set of endpoints and costs of arcs emanating out of node 'start'.
    L = G{start,2}; % costs
    for k = 1:length(A),
      for l=2:(maxLength+1),
        if (isinf(d(l-1,start))),
          % PEND: Could check for Inf values earlier and then
          % loop over the resulting set.
          continue;
        end
        if ( d(l,A(k)) > d(l-1,start)+L(k) )
          d(l,A(k)) = d(l-1,start)+L(k);
        end
      end
    end
  end
  
end


% Now we figure out which of the endnodes has the lowest distance label. That is going to be
% the end node of our shortest path. To find the shortest path we just have to start at
% the endnode and track the predecessor nodes.
cost = zeros(1,maxLength+1);
ind = zeros(1,maxLength+1);

for m = 1:(maxLength+1),
  [cost(m),ind(m)] = min(d(m,endnodes));
end
cost = cost(2:(maxLength+1));

if (nargout > 1),
  % the caller wanted to get the paths, so find them
  paths = cell(1,maxLength);
  
  for k=1:maxLength,
    % if cost(k)==Inf, then we know there is no path of length(k)
    % Note: This depends on the coarsest scale in the graph
    % PEND: Perhaps it would be good to pass the graph parameters
    % and use the fact that if cs is the coarsest allowable scale
    % then the minimum number of chirplets in a path is 2^cs
    if (cost(k)==Inf),
        continue;
    end
 
    paths{k} = zeros(1,k+1);
    paths{k}(k+1) = endnodes(ind(k+1));
    for l=k:-1:1,
        paths{k}(l) = pred(l+1,paths{k}(l+1));
    end
  end
end

%mctr = min(cost(2:(maxLength+1))./(1:(maxLength)));


% $RCSfile: CSPcell.m,v $
% $Date: 2006/10/11 22:57:50 $
% $Revision: 1.8 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2005
