function [costpath,chirpletpath,nchirplets] = CalculateStatistic(cnetwork,statType,alpha)
% CalculateStatistic -- for solving different optimization problems on chirplet graphs
%		
%  Usage
%    [costpath,chirpletpath,nchirplets] = CalculateStatistic(cnetwork,statType,alpha)
%    [costpath,chirpletpath,nchirplets] = CalculateStatistic(cnetwork,statType)
%    [costpath,chirpletpath,nchirplets] = CalculateStatistic(cnetwork)
%  Inputs
%    cnetwork   a chirplet network as returned by GetChirpletNetwork
%    statType   parameter for choosing which optimization problem to solve
%               'MCTR' or 0 to calculate the minimum cost to time ratio (MCTR), 
%               'SP' or 1 to calculate the shortest path (SP) in the graph, 
%               'BPFORPLOTTING' or 2 to calculate the constrained SP (BP) and return paths (SLOW)
%               'BP' or 3 to calculate BP but doesn't return paths (FAST)
%               If omitted defaults to 'SP'.
%    alpha      For MCTR this parameter should be set to a known upper bound on the MCTR 
%               (default value is 0, since costs are negative). 
%               For SP this is a number to be added to every arc cost in the graph.
%               For BP this parameter corresponds to the maximum number of chirplets 
%               allowed in the path and cannot be omitted. Calculates the best paths
%               with number of chirplets equal 1,2 up to alpha
%               Default value is 0 in the case of MCTR and SP.
%  Outputs
%    costpath	For MCTR and SP this is the cost of the optimal path.
%               For BP, this is an array of length alpha. Entry k in the
%               array is the cost of the best path with k chirplets.
%    chirpletpath
%               For MCTR and SP, a vector with information about which chirplets are 
%               on the optimal path
%               For BP, this is a 1d cell array of length alpha. Entry k in the
%               cell array corresponds to the best path with k chirplets.
%               If there does not exist a path with k chirplets, for
%               example because of restriction of the coarsest allowable
%               scale, paths{k} will be empty.
%               Note that if cs is the coarsest allowable scale
%               then the minimum number of chirplets in a path is 2^cs.
%    nchirplets the number of chirplets on the optimal path. For BP this is value is set to [].
%
%  Description
%    For solving optimization problems in the chirplet graph like the shortest path (SP)
%    the constrained shortest path (BP) and the minimum cost-time ratio problem (MCTR)
%    Note that the graph is acyclic. 
%    Before doing the optimization, the chirplet graph parameters have to set,
%    chirplet transform of data has to be done using the function and the chirplet
%    costs filled in the chirplet network.
%    Currently there is only support for chirplet graphs with connected offsets.
%    That is, near-continuity of the chirplet paths in the time-frequency plane.
%    In later versions support for connected chirplets slopes, i.e. curvature
%    constraints, will be provided.
%
%  See also 
%    - ChirpletTransform for doing chirplet transforms of the data
%    - GetChirpletNetwork for initializing the chirplet network
%    - DisplayChirpletPath for plotting chirplet paths

if (nargin < 3),
    alpha = 0;
end;

if (nargin < 2),
  % calculates the SP by default
  statType = 'SP';
else
  if(isnumeric(statType)),
    switch statType
     case 0,    
      statType = 'MCTR';
     case 1,
      statType = 'SP';
     case 2,
      statType = 'BPFORPLOTTING';
     case 3,
      statType = 'BP';
     otherwise,
      errormsg = 'CalculateStatistic: Wrong value for parameter statType. See documentation.'
      error(errormsg);
    end % switch
  else
    statType = upper(statType);
  end % if
end % if

if ((strcmp(statType, 'BP') | strcmp(statType, 'BPFORPLOTTING')) & nargin < 3),
  % alpha corresponds to the BP statistic
  % in which case a value must be provide
  errmsg = 'CalculateStatistic: When calculating BP, maximum path length must be provided (i.e. the parameter alpha)';
  error(errmsg);
end

if (strcmp(upper(statType),'BP') & nargout > 1)
  warningmsg = 'CalculateStatistic: For returning chirplet paths set statType to BPFORPLOTTING. Some of the returned arguments will be empty.';
  warning(warningmsg);
end

% Calculate the test statistic

if strcmp(upper(statType),'MCTR'),
    [costpaths,lpaths,paths] = minctratiocells(...
                              cnetwork{1},alpha,cnetwork{2},...
                              cnetwork{5},cnetwork{3},cnetwork{4});
    costpath = costpaths(length(costpaths));
    nchirplets = lpaths(length(costpaths));
    chirpletpath = paths{length(costpaths)};
elseif strcmp(upper(statType),'SP'),
    [costpath, chirpletpath] = ShortestPathCell(...
                                cnetwork{1},cnetwork{2},alpha,...
                                cnetwork{5},cnetwork{3},cnetwork{4});
    nchirplets = length(chirpletpath) - 1;
elseif strcmp(upper(statType),'BPFORPLOTTING'),
    if (nargout > 1),
      % caller wanted to find the chirplet paths
        [costpath, chirpletpath] = CSPcell(...
                                cnetwork{1},cnetwork{2},...                                
                                cnetwork{3},cnetwork{4},alpha);
      nchirplets = []; % this parameter doesn't apply here, let it be empty
    else
      % only find the cost, use a faster routine
      costpath  = CSPstatistic(...
                                cnetwork{1},cnetwork{2},...                                
                                cnetwork{3},cnetwork{4},alpha);
    end
elseif strcmp(upper(statType),'BP'),
    % only find the cost,
	costpath  = CSPstatistic(...
                                cnetwork{1},cnetwork{2},...                                
                                cnetwork{3},cnetwork{4},alpha);
	chirpletpath = [];
	nchirplets = [];
else
    errormsg = 'CalculateStatistic: Wrong value for parameter statType. See documentation.';
    error(errormsg);
end

% $RCSfile: CalculateStatistic.m,v $
% $Date: 2006/10/11 22:57:50 $
% $Revision: 1.14 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
