function [pmin,ind] = EstimateMinPvalues(nullData,altData,whichIndArray,rejectForBig,doPlot)
% EstimateMinPvalues -- for estimating the minimum p-values from 
%                       realizations of random vectors
%
%  Usage
%    [pmin,ind] = EstimateMinPvalues(nullData,altData,whichIndArray,rejectForBig,doPlot)
%    [pmin,ind] = EstimateMinPvalues(nullData,altData,whichIndArray,rejectForBig)
%    [pmin,ind] = EstimateMinPvalues(nullData,altData,whichIndArray)
%    [pmin,ind] = EstimateMinPvalues(nullData,altData)
%    [pmin,ind] = EstimateMinPvalues(nullData)
%  Inputs
%    nullData       each row in nullData corresponds to one realization of a random vector.
%                   This data is used to estimate the distribution of this random vector.
%                   If the parameter altData is provided this estimation is used to
%                   estimate the p-values for the data in altData.
%                   If altData is not provided, it estimates minimum p-values under the null
%                   using a `leave-one-out' method (see below).
%    altData        each row in altData corresponds to one realization of a random vector.
%                   p-values for the data in this parameter are estimated from nullData.
%                   To skip this parameter set it to []
%    whichIndArray  a vector with the coordinates in the random vector to restrict to.
%                   For example, whichIndArray = [1 2 8], would use theData(:,[1 2 8])
%                   If omitted or set to [], all coordinates are used.
%    rejectForBig   if set to 1, big values are "unlikely" under the null
%                   if set to 0, small values are "unlikely" under the null
%                   If omitted or set to [] the value will be set to 1.
%    doPlot         set this parameter to plot the ecdf for the minimum p-values. 
%  Outputs
%    pmin           a vector with the estimated minimum p-values for each realization in
%                   nullData if altData was not provided, 
%                   otherwise for each realization in altData
%    ind            the indices for which gave minimum p-values in pmin.
%                   For the k-th realization, whichIndArray(ind(k)) is the
%                   the coordinate in signal(k,:) which gave the minimum p-value.
%
%  Description
%    For estimating the cumulative distribution function (ecdf) at a point x given
%    a set of data, this method uses the simple estimate (#samples <= x)/(total no. samples).
%    For estimating a p-values it uses a `leave-one-out' method, that is, the p-value
%    for every sample in a data vector is estimated from the ecdf of the rest of the samples.
%
%  See Also
%    - DetectionRate
%    - RunSimulations
%    - ExperimentSetup

[nNullSamples,nCoordinates] = size(nullData);

% check the inputs
if (nargin < 5),
  % do not generate plot
  doPlot = 0;
  if (nargin < 4),
    rejectForBig = 1;
    if (nargin < 3),
      % use all coordinates by default
      whichIndArray = 1:nCoordinates;
	  if (nargin < 2),
	    altData = [];
	  end
    end
  end
end

if (isempty(whichIndArray)),
  % use all coordinates
  whichIndArray = 1:nCoordinates;
end

if (isempty(rejectForBig)),
  % use all coordinate
  rejectForBig = 1;
end

leaveOneOut = 0;
if (isempty(altData)),
  leaveOneOut = 1;
  nSamples = nNullSamples;
else
  [nSamples,nAltCoordinates] = size(altData);
  if (nAltCoordinates~=nCoordinates),
    warningMsg = 'EstimateMinPvalues: The random vectors for the null and alternative are not of the same dimension';
    warning(warningMsg);
  end
end
  
nCoordinates = length(whichIndArray);

pvalues = zeros(nSamples,nCoordinates);

% estimate the p-values, correct for `rejectForBig' afterwards
for m=1:nCoordinates,
  coord = whichIndArray(m);
  [X,ind] = sort( nullData(:,coord) );
  
  if(leaveOneOut),
    % Assume the sorted vector is x=[x_1 x_2 ... x_N]
    % Leave the k-th element x_k out. Then there are k-1 entries
    % out of N-1 that are smaller than x_k. This gives us an
    % estimate of the p-value for x_k as 1-(k-1)/(N-1) 
    tmp = (0:nSamples-1)/(nSamples-1);
  
    % assign p-values to right indices
	pvalues(ind,m) = 1-tmp;
  else
  
    for k=1:nSamples,
      pvalues(k,m) = 1-sum(X<=altData(k,coord))/length(X);
    end

%    F = (0:(length(X)-1))/length(X);
%    pvalues(:,m) = 1-interp1(X,F,altData(:,coord),'linear',0);
%    outside = (altData(:,ind) < X(1));
%    pvalues(outside,m) = 0;
  end
end

if (rejectForBig==0),
  % small values are "unlikely" under the null, correct the calculations above
  pvalues = 1-pvalues;
end

% find the minimum along rows
[pmin, ind] = min(pvalues,[],2);

if(doPlot),
  figure;
  [f,x] = ecdf(pmin);
  plot(x,f);
  xlabel('minimum p-value');ylabel('ecdf');
end

% $RCSfile: EstimateMinPvalues.m,v $
% $Date: 2006/05/01 17:52:55 $
% $Revision: 1.3 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
