function expSetup = ExperimentSetup(graphParam,statType,statParam,numSims,sig,snr);
% ExperimentSetup -- wrapper for settings for simulations. 
%                    Note that this implementations assumes complex white noise.
%
%  Usage
%    expSetup = ExperimentSetup(graphParam,statType,statParam,numSims,sig,snr);
%    expSetup = ExperimentSetup(graphParam,statType,statParam,numSims);
%    expSetup = ExperimentSetup(graphParam,statType,statParam);
%
%  Inputs
%    graphParam   chirplet graph parameters as returned by GetChirpletGraphParam.
%                 This parameter has to be provided.
%    statType     a char array for choosing the type of test statistic to use. 
%                 Use 
%                   'BP' for the best path test statistic, 
%                   'MCTR' for the minimum cost to time ratio
%                   'SP' for the shortest path
%                 NOTE: This parameter has to be provided.
%    statParam    For statType equal to:
%                   'BP' this parameter corresponds to the maximum number of chirplets 
%                    allowed in the path. Best paths of all lengths up to this value
%                    will be calculated.
%                    'MCTR' this parameter corresponds to an array of scale indices of
%                    coarsest scales in the graph for which the MCTR should be calculated. 
%                    Say, if statParam = [0 1 2 3] the MCTR costs for the same realization
%                    of noise will be calculated for the chirplet graphs that differ only
%                    in the coarsest allowable scale, i.e. 2^0,2^(-1),2^(-2),2^(-3) resp.
%                    The value 0 will be used as an upper bound for calculating the MCTR.
%                    'SP' this is a number to be added to every arc cost in the graph.
%                 NOTE: This parameter has to be provided if statType='BP' or 'MCTR'.
%                 If omitted and statType='SP', this will be set to 0. 
%    numSims      the number of simulations to perform. If this parameter is omitted
%                 or set to [], the value 1 will be used.
%    sig          the signal to use in the experiment, an optional parameter.
%                 If omitted, this will have an empty value.
%                 NOTE: For simulations under the null, omit this parameter or set it to [].
%    snr          the signal to noise ratio (SNR) to use in the experiment, an optional parameter.
%                 The definitition of SNR used in ChirpLab, in case of white noise, is
%                     || a S ||/sqrt(N)
%                 where the data vector is y = a S + z, is of length N, S is the signal we wish
%                 to detect, `a' is a scalar and z is complex-valued white noise sequence:
%                 z=z_r + i z_i, where z_r and z_i are two independent vectors of iid N(0,1/2)
%                 variables.
%                 If z_r and z_i where vectors of iid N(0,1) variables, the SNR would be
%                     || a S ||/sqrt(2*N)
%                 If omitted, this variable will have an empty value. Has to be set if the
%                 parameter sig is set.
%                 NOTE: For simulations under the null, omit this parameter or set it to [].
%    description  an optional parameter. A string for describing this experiment setup in words
%                 if omitted, set to an empty string: '' 
%  Outputs
%    expSetup    a MATLAB structure with parameters describing the experiment setup
%                The fields are 
%                  graphParam, statType, statParam, numSims, sig, snr, description
%                and store the same values given by the input parameters above.
%                Value of the fields can retrieved for example by 
%                expSetup.graphParam, expSetup.statType etc.
%
%  See also 
%    - GetChirpletGraphParam for information on how to customize the chirplet transform.
%    - CalculateStatistic, for information on which test statistics can be calculated
%    - RunSimulations for running simulations as described in an experiment setup

if (nargin < 7),
  % no description provided, set it to an empty string
  description = '';
  if (nargin < 6),
    % snr was not provided, set it to an empty value
    snr = [];
    if (nargin < 5),
      % sig not provided, set it to an empty value
	  sig = [];
	  if (nargin < 4),
	    % number of simulations not set, set it to default value
	    numSims = 1;
	    if (nargin < 3),
		  if (nargin<2),
		    errormsg = 'ExperimentSetup: The input parameters graphParam and statType have to be provided.';
		    error(errormsg);
		  else
		    % check
			if (strcmp(statType,'SP'))
			  statParam = 0;
			else
              % the first three parameters have to be provided in case of statType='BP' or 'MCTR'
		      errormsg = 'ExperimentSetup: The input parameter statParam has to be provided.';
		      error(errormsg);
            end
          end
        end
	  end
    end
  end
end

if (~isempty(sig)),
  n = graphParam{1}(1); % signal length according to graph
  if (n~=length(sig)),
    errormsg = 'ExperimentSetup: The signal in the parameter sig and the signal length according to graph have to be equal.';
	error(errormsg);
  end
end

% create the structure
expSetup = struct('graphParam',{graphParam},'statType',statType,'statParam',statParam,...
                  'numSims',numSims,'sig',sig,'snr',snr,'description',description);
				  
% $RCSfile: ExperimentSetup.m,v $
% $Date: 2006/05/01 17:52:55 $
% $Revision: 1.4 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
