% LIGOPsd - simulated 2-sided LIGO PSD as a function of frequency
%
% Inputs
%   f - array of frequency values in Hz
%
% Outputs
%   S - PSD evaluated at f, in units strain^2/Hz
%
% This function returns the 2-sided LIGO spectrum evaluated at f.
% The power spectrum is based on a fit to the power spectral density
% of the LIGO I detector given in the paper
%   http://www.arxiv.org/pdf/astro-ph/0008481 on page 63.
%
% To simulate the spectrum of data that has been low-pass filtered
% prior to A-D conversion, the spectrum is multiplied by the gain
% function of a 12th-order Butterworth low-pass filter with cutoff
% frequency 6000 Hz.
%
% The fitting function is invalid below around 40 Hz. To obtain a
% a spectrum with reasonable dynamic range in that band, the spectrum
% is fitted with the gain function of a Butterworth low-pass filter
% with cutoff frequency 20 Hz.
%
% To get the usual LIGO 1-sided spectrum Sh(f), S must be multiplied by 2.
% It must be multiplied by 1/dt to give the equivalent of a Matlab PSD as
% given by the psd() function.
function S = LIGOPsd(f)

  % The frequency where the minimum value of the power spectrum
  % occurs ie. where the detector is most sensitive.
  f0 = 150; % Hz

  % Minimum value of S
  S0 = 9.0E-46;

  % Seismic cutoff at 40 Hz
  xc = 40/f0;

  % Low-pass filter cutoff at 6000 Hz
  xLP = 6000/f0;

  % Start of seismic band. Spectrum is flat below this value
  xs = 20/f0;

  % Convert frequencies to unit-free parameter
  x = f/f0;

  S = zeros(1, length(f));

  % Select all the frequencies outside of the seismic band
  f1 = (x < -xc | xc < x);
  S(f1) = psdFn(x(f1));

  % Smoothly continue S(f) using a Butterworth gain profile
  f2 = (-xc <= x & x <= xc);
  S(f2) = seismicPsd(x(f2), xc, xs, S0);

  % Multiply by the low-pass filter function
  G = butterworth_LPgain(x, xLP, 12);
  S = G.*S;

  S = S0*S;

return;

function S = psdFn(x)

  xAbs = abs(x);
  S = (4.49*xAbs).^(-56) + 0.16*xAbs.^(-4.52) + 0.32*xAbs.^2 + 0.52;

return;

function DS = DpsdFn(x)

  if (sum(x <= 0) ~= 0)
    error('Cannot evaluate dS/df at non-positive frequencies');
  end;

  DS = -56*4.49*(4.49*x).^(-57) - 4.52*0.16*x.^(-5.52) + 2*.32*x;

return;

function S = seismicPsd(x, xc, xs, S0)

  % Set the order ie. steepness of the transition band
  n = 6;

  % Find the gain so that the function is continuous with S at x = xc
  Sc = psdFn(xc);
  Gsq = (1 + (xc/xs)^(2*n))*Sc;

  S = Gsq*butterworth_LPgain(x, xs, n);

return;

% BUTTERWORTH_LPGAIN Gain of an n-th order Butterworth filter with
% cutoff at fc
%
%                1
%   G(f) = ---------------
%          1 + (f/fc)^(2n)
%
function G=butterworth_LPgain(f, fc, n)

  G = (1 + (f/fc).^(2*n)).^(-1);

return;
