function [x,omega] = MakeChirp(Name,n,param)
% MakeChirp -- Make artificial chirp signal
%
%   Usage
%     [x,omega] = MakeChirp(Name,n)
%   Inputs
%     Name   One of the following strings:
%           'CubicPhase'                   - constant amplitude, cubic phase
%           'CubicPhaseAmplitude'          - same as 'CubicPhase', except with cosine amplitude modulation
%           'CubicPhaseQuadraticAmplitude' - same as 'CubicPhase', except with quadratic amplitude modulation
%           'QuadraticPhase'               - constant amplitude, quadratic phase
%           'CosinePhase'                  - constant amplitude, cosine phase
%           'CosinePhaseAmplitude'         - same as 'CosinePhase', except with cosine amplitude modulation
%           'CosinePhaseModulated'         - same as 'CosinePhase', but modulated by exp(i*2*pi*100*n/1024)
%           'CosinePhaseModulatedAmplitude' - same as 'CosinePhaseModulated', except with cosine amplitude modulation
%     n      desired signal length
%   Outputs
%     x      chirp signal
%     omega  the "instantaneous frequency" of the chirp signal (the phase differentiated).

switch upper(Name)

% COSINE PHASE CHIRP
  case {'COSINEPHASE', 'COSINEPHASEAMPLITUDE', 'COSINEPHASEMODULATED', 'COSINEPHASEMODULATEDAMPLITUDE'},
    t = 0:(n-1);

    if (nargin<3),	
      % set the default value
      b = 1/(4*pi);
      omega0 = n/2^10 * 100;
    elseif (length(param)==2),
      % set the default value
      b = param(1);
      omega0 = param(2);
    else,
      errormsg = sprintf('MakeChirp: The input param has to have three entries when Name=%s.',Name);
      error(errorsmg);
    end

    phase = n*b*sin(2*pi*t./n);
    omega = n*b*2*pi*cos(2*pi*t./n)/(2*pi);

    a = ones(1,n);
    if (strcmp(Name,'COSINEPHASEAMPLITUDE')),
      a = cos(2*pi*t/n);
    end
    x = exp(i*phase);

    if ( strcmp(upper(Name),'COSINEPHASEMODULATED') | strcmp(upper(Name),'COSINEPHASEMODULATEDAMPLITUDE') ),
      phase = phase + 2*pi*omega0*t./n;
      omega = omega + omega0;
      if strcmp(upper(Name),'COSINEPHASEMODULATEDAMPLITUDE'),
        a = 2+cos(2*pi*t/n+pi/4);
      end
    end

    % amplitude modulation
    x = exp(i*phase) .* a;

    % normalize so x has l2-norm sqrt(n)
    x = x/norm(x)*sqrt(n);

% CUBIC PHASE CHIRP
  case {'CUBICPHASE','CUBICPHASEFORPAPER', 'CUBICPHASEAMPLITUDE', 'CUBICPHASEQUADRATICAMPLITUDE'},
    t = (0:n-1)/n;
    if (nargin<3),
	  if (strcmp(upper(Name),'CUBICPHASEFORPAPER')),
        a = n/16;    
        b = 0;
        % Want a+c/2 <= n/2 so final frequency is <= Nyquist
        c = n/4;
	  else
       % set the default value
		a = n/8;   
        b = 0;
        % Want a+c/2 <= n/2 so final frequency is <= Nyquist
        c = n/2;
	  end
    elseif (length(param)==3),
      % set the default value
      a = param(1);
      b = param(2);
      c = param(3);
    else,
      errormsg = sprintf('MakeChirp: The input param has to have three entries when Name=%s.',Name);
      error(errorsmg);
    end
    x = exp(i*2*pi*(a*t + b*t.^2/2 + c*t.^3/6));
    omega = a + b*t + c*t.^2/2;

    if (strcmp(upper(Name),'CUBICPHASEAMPLITUDE')),
      x = cos(2*pi*t).*x;
      x = sqrt(n)*x/norm(x);
    elseif (strcmp(upper(Name),'CUBICPHASEQUADRATICAMPLITUDE')),
      x = (1 + 2*t.*t).*x;
      x = sqrt(n)*x/norm(x);
    end

% QUADRATIC PHASE CHIRP
  case 'QUADRATICPHASE',
    t = (0:n-1)/n;
    if (nargin < 3),
      % set the default value
      a = n/8;
      % Want a+b <= n/2 so final frequency is <= Nyquist
      b = n/4;
    elseif (length(param)==2),
      a = param(1);
      b = param(2);
    else
      errormsg = sprintf('MakeChirp: The input param has to have three entries when Name=%s.',Name);
      error(errorsmg);
    end
    x = exp(i*2*pi*(a*t + b*t.^2/2));
    omega = a + b*t;

% UNKNOWN SIGNAL NAME
  otherwise,
    errormsg = sprintf('MakeChirp: Unknown signal name: %s. Run help MakeChirp to see list of supported signals.',Name);
    error(errormsg);
end


% $RCSfile: MakeChirp.m,v $
% $Date: 2006/05/02 21:38:07 $
% $Revision: 1.11 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006

