% MakeLIGO1Psd -- Generates 2-sided LIGO noise PSD of length N (strain^2/Hz)
% Usage
%   [S, f, Sh, fh] = MakeLIGO1Psd(N, Fs)
% Inputs
%   N      number of samples to generate in the spectrum
%   Fs     sampling rate
% Output
%   S      the 2-sided power spectrum of the noise in FFT order and
%          units strain^2/Hz
%   f      the frequencies where S is evaluated
%   Sh     the 1-sided power spectrum
%   fh     the frequencies where Sh is evaluated
% Description:
%  The noise power spectrum is based on a fit to the noise power
%  spectral density of the LIGO I detector given in the paper
%  http://www.arxiv.org/pdf/astro-ph/0008481 on page 63.
% Note - this spectrum must be multiplied by the sampling rate to give
% the equivalent of a Matlab PSD
function [S, f, Sh, fh] = MakeLIGO1Psd(N, Fs)

  % Construct the frequency series
  df = Fs/N;

  % Positive frequencies for Sh
  fh = [0:N/2]*df;

  % All frequencies, in FFT order
  f = [ [0:N/2] -[N/2-1:-1:1] ]*df;

  % Generate the 2-sided PSD
  S = LIGOPsd(f);

  % Make a 1-sided PS
  Sh = [ S(1) 2*S(2:N/2) S(N/2+1) ];

return;

% $RCSfile: MakeLIGO1Psd.m,v $
% $Date: 2007/05/23 04:58:31 $
% $Revision: 1.7 $
%
% Copyright (c) Philip Charlton, 2006
