function [T,seedUsed] = RunSimulations(expSetup,fname,randSeed)
% RunSimulations -- for running simulations as described an experiment setup
% 
%  Usage
%    [T,seedUsed] = RunSimulations(expSetup,fname,randSeed)
%    [T,seedUsed] = RunSimulations(expSetup,[],randSeed)
%    [T,seedUsed] = RunSimulations(expSetup,fname)
%    [T,seedUsed] = RunSimulations(expSetup)
%
%  Inputs
%    expSetup    a MATLAB structure as returned by ExperimentSetup with parameters 
%                describing the experiment setup.
%                If no signal is provided in expSetup (that is expSetup.sig is empty)
%                this routine will simulate the test statistic under the null. That
%                is the data is just noise of the form, z=z_r + i z_i, where 
%                z_r and z_i are two independent vectors of iid N(0,1) variables.
%                In this case the SNR would be
%                     || S ||/sqrt(2*N)
%                where || S || is the l2-norm of the signal to detect.
%    fname       A character array for a file name. If set, expSetup, T and seedUsed 
%                will be saved to a file with this file name.
%                This is an optional parameter. To set randSeed but not
%                write the data to file, set this parameter to an empty square brackets: []
%    randSeed    Forces the simulations to be done with this specific random seed
%                This is an optional parameter. 
%  Outputs
%    T           results from simulations using the settings from expSetup stored in a matrix.
%                Every row in this matrix corresponds to one simulation. 
%                In the case of the BP test statistic with maximum length set
%                to 16, T(n,:) correponds to the best costs for lengths 1,...,16 in simulation n.  
%                In the case of SP, T(n) is the shortest path from simulation n
%    seedUsed    the random seed used to do the simulations. Equal to 
%                randseed if it was set
%
%  See also 
%    - ExperimentSetup a wrapper for the settings for the simulations

if (nargin < 3),
  % generate a random seed based on the current time
  randSeed = sum(100*clock);
  if (nargin < 2),
    % no file name provided, set it to an empty string, nothing will
	% be written out to file
    fname = '';
  end
else
  % check whether fname is a character array
  if(~ischar(fname) & ~isempty(fname)),
    errormsg = 'RunSimulations: the input parameter fname has to be a string/char array';
	error(errormsg);
  end
end

% Set the state of the random generator
seedUsed = randSeed;
randn('state',seedUsed);

% get parameters from the experiment setup
graphParam = expSetup.graphParam;
sig = expSetup.sig;
snr = expSetup.snr;
statType = expSetup.statType;
statParam = expSetup.statParam;
numSims = expSetup.numSims;

N = graphParam{1}(1); % signal length

if (~isempty(sig)),
  % multiply signal with a scalar to get the requested SNR, 
  % the l2-norm should be snr*sqrt(2*N)
  theSignal = snr*sqrt(2*N) * sig/norm(sig); 
  % we take care of the sqrt(2) factor here instead of in the noise generation
end

if (strcmp(statType,'BP')),
  T = zeros(numSims,statParam);
elseif (strcmp(statType,'MCTR')),
  T = zeros(numSims,length(statParam));
elseif (strcmp(statType,'SP')),
  T = zeros(numSims,1);
end

% DO SIMULATIONS
for runNo=1:numSims,
  % generate noise
  z = randn(1,N) + i *randn(1,N);
  if isempty(sig),
    % do null simulations
	cc = ChirpletTransform(z,graphParam);    
  else
    % we are doing simulations under the alternative
	y = theSignal + z;
	cc = ChirpletTransform(y,graphParam);
  end
  if (strcmp(statType,'BP')),
    cnetwork = GetChirpletNetwork(cc,graphParam);
    T(runNo,:) = CalculateStatistic(cnetwork,statType,statParam);
  elseif (strcmp(statType,'MCTR')),
    for k=1:length(statParam),
      cnetwork = GetChirpletNetwork(cc,graphparam,statParam(k));
      T(runNo,k) = CalculateStatistic(cnetwork,STATTYPE); 
    end
  elseif (strcmp(statType,'SP')),
    cnetwork = GetChirpletNetwork(cc,graphParam);
    T(runNo) = CalculateStatistic(cnetwork,statType,statParam);
  end
end

if (~isempty(fname)),
  disp(sprintf('Saving results from simulations to file: %s',fname));
  save(fname,'expSetup','T','seedUsed');
end

% $RCSfile: RunSimulations.m,v $
% $Date: 2006/05/01 17:52:56 $
% $Revision: 1.5 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
