/* 
   ShortestPathCellStub.c
   Used to solve the shortest path problem for an acyclic graph, starting from given
   start nodes to given endnodes. Uses a cell array data structure for the network.
   
   See also documentation in ShortestPathCellStub.m
*/

#include "mex.h"

/* 
#include <math.h>
   Gateway function 

   Inputs:
	prhs[0] - a cell array where cell k is a vector corresponding to node k in the 
		graph and holds all the nodes that node k connects to.
	prhs[1] - a cell array with the costs corresponding to the arcs described in prhs[0].
     	prhs[2] - total number of nodes in graph.
	prhs[3] - a constant to update the costs on the arcs, the updated cost will be the
		old cost as described in prhs[1] minus alpha. (Needed to find the MCTR)
	prhs[4] - a vector with the topological ordering of the nodes in G
	prhs[5] - list of nodes from where paths can start in the acyclic network.

   Outputs:
	plhs[0] - a vector with the distance labels for all the nodes in the graph
	plhs[1] - list of predecessor nodes for the shortest path
*/
void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray
                 *prhs[]) 
{ 
  double *arcs;         /* the set of endpoints and costs of arcs 
                           emanating out of node k */
  int nArcs;            /* number of arcs */
  double *arcCosts;     /* the arc costs */
  double *startNodes;   /* starting nodes in the graph */ 
  int nStartNodes;      /* length of the array startnodes */
  int nLeftEndpoints;   /* number of nodes that arcs emanate from  */
  int nNodes;           /* number of nodes in graph */
  double alpha;         /* a constant to update the costs on the arcs */
  double *ord;          /* a vector with the topological ordering of the 
                           nodes in G */

  const mxArray *cellElementPtr;
  int startNode;
  int arcIndex;
  double tentativeDistance;
  int k;
  
  /* Output Variables */
  double *pred;         /* predecessor nodes */
  double *d;            /* distance labels */

  double inf;           /* a variable to store the value of infinity on 
                           the system running Matlab */
  inf = mxGetInf();     /* get system's infinity */
  
  /* Get input data */
  nNodes = (int) mxGetScalar(prhs[2]);
  alpha = mxGetScalar(prhs[3]);
  ord = mxGetPr(prhs[4]);
  startNodes = mxGetPr(prhs[5]);
  nStartNodes = mxGetNumberOfElements(prhs[5]);
  nLeftEndpoints = (int) mxGetScalar(prhs[6]);

  /* Set output */
  plhs[0] = mxCreateDoubleMatrix(1,nNodes, mxREAL);
  d = mxGetPr(plhs[0]);
  plhs[1] = mxCreateDoubleMatrix(1,nNodes, mxREAL);
  pred = mxGetPr(plhs[1]);
  
  /* Initialize d by setting all distance labels to infinity */
  for (k=0; k < nNodes; k++) {
    d[k] = inf;
  }
  /* starting nodes have the distance label 0 */
  for (k=0; k < nStartNodes; k++) {
    d[(int) startNodes[k]-1 ] = 0; /* -1 because C indices from 0 */
  }  
 
  /* Examine nodes in topological order */
  for (startNode=0; startNode < nLeftEndpoints; startNode++) {
    cellElementPtr = mxGetCell(prhs[0], startNode);
    /* the cell could be empty, so in that case we do nothing */
    if (cellElementPtr == NULL) {
      continue;
    }
    /* we reach this statement if the cell is not empty */
    arcs = mxGetPr(cellElementPtr);
    /* arcs is the set of endpoints of arcs emanating from startNode */
    nArcs = mxGetNumberOfElements(cellElementPtr);
    
    cellElementPtr = mxGetCell(prhs[1], startNode);
    arcCosts = mxGetPr(cellElementPtr);
    for (k=0; k < nArcs; k++) {
      /* calculate a tentative distance label, updating cost by 
         subtracting alpha */
      tentativeDistance = d[(int) ord[startNode]-1]+ arcCosts[k] - alpha;
      arcIndex = (int) arcs[k] - 1;
      if( d[arcIndex] > tentativeDistance ) {
        d[arcIndex] = tentativeDistance;
        pred[arcIndex] = ord[startNode];
      }
    }
  }

}
