function [allYhat, nChirplets, omega, phi, amp, phi0] = BPDenoiseWithSpline(y,sigma,graphparam,maxLength,alpha)
% BPDenoiseWithSpline -- estimates an unknown chirp in noisy data using the BP estimator
%            
%  Usage
%    [allYhat, nChirplets, omega, phi, amp, phi0] = BPDenoiseWithSpline(y,sigma,graphparam,maxLength,alpha)
%  Inputs
%    y            vector with noisy data of the form y= s + z, 
%                 where s is an unknown chirp and z is Gaussian white noise
%    sigma        variance of the Gaussian noise 
%    graphparam   chirplet graph parameters as returned by GetChirpletGraphParam
%    maxLength    maximum allowed numbers of chirplets in a chirplet path
%    alpha        vector of smoothing parameters for the spline smoothing for the amplitude
%                 These are the values for which to calculate 
%                 the cross-validation score function CV(alpha) (see Chapter 5 in Thesis)
%  Outputs
%    allYhat      matrix with two estimations of the unknown chirp:
%                 1st row is the estimation using the best path and piecewise constant
%                 amplitude fit.
%                 2nd row is the estimation using the best path to estimate the phase of the chirp
%                 and amplitude is fitted globally afterwards using splines smoothing.
%    nChirplets   number of chirplets for the "best" chirplet path
%    y            signal reconstructed from path and original data
%    phi          phase estimate of the chirp in the noisy data
%    omega        estimate of instantaneous frequency chirp
%    amp          piecewise constant estimate of amplitude of the chirp
%    phi0         local estimates of phase offsets
%
%  Note: The penalization coefficient used in TollCTPlain is Lambda = 2*sigma^2*(2 * log(N)),
%        where N is the length of the data.
% 
%  For further information about the estimation procedure, see chapter 5 of the Thesis.


if (nargin<5),
  % hardcoding spline smoothing penalization vector
  alpha = 0.001:0.001:0.05;
end

N = length(y);

Lambda = 2*sigma^2*(2 * log(N)); 

cc = TollCTPlain(y,graphparam,Lambda); 

[costpath,allpaths] = TollBPDirect(cc,graphparam,maxLength);
[minval,ind] = min(costpath);
pp = allpaths{ind};
nChirplets = length(pp)-1;

[yhat,phi,omega,amp,phi0] = GenerateSignalFromPath(pp,graphparam,y);

% Global amplitude fit:
% Estimation based on fitting amplitude seperately using spline smoothing
s = exp(-i*phi);
noisyAmp = real(y.*s); 
t = (0:(N-1))/N;

[ahat,CV] = SplineSmoother(noisyAmp,t,alpha);
 
ahat = reshape(ahat,1,N);
yhatGAafter = abs(ahat).*exp(i*phi);
  
% return estimates
allYhat = zeros(2,length(y));
allYhat(1,:) = yhat;
allYhat(2,:) = yhatGAafter;

% $RCSfile: BPDenoiseWithSpline.m,v $
% $Date: 2008/05/19 23:08:24 $
% $Revision: 1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
