function [y,phi,omega,amp,phi0s] = GenerateSignalFromPath(p,gp,yy)
% GenerateSignalFromPath -- reconstructs a signal from a chirplet path
%            
%  Usage
%    [y,phi,omega,amp,phi0s] = GenerateSignalFromPath(p,gp,yy)
%  Inputs
%    p       chirplet path resulting from the the BP algorithm
%    gp      chirplet graph parameters as returned by GetChirpletGraphParam
%    yy      original (noisy) data fed into the BP algorithm and outputting
%            the path p
%  Outputs
%    y       signal reconstructed from path and original data
%    phi     phase estimate of the chirp in the noisy data
%    omega   estimate of instantaneous frequency chirp
%    amp     piecewise constant estimate of amplitude of the chirp
%    phi0    local estimates of phase offsets


J = gp{1}(2);    % dyadic length of signal
fsc = gp{2}(2);  % finest scale in the chirplet graph 
fmin = gp{4}(1); % lowest frequency in the chirplet graph 
fmax = gp{4}(2); % highest frequency in the chirplet graph 
[t,freq] = Path2TF(p,J,fmin,fmax,fsc);
n = 2^J;
phi = zeros(1,2^J);
phi0s = zeros(1,2^J);
omega = zeros(1,2^J);
amp = zeros(1,2^J);
y = zeros(1,2^J);
nchirplets = length(t) - 1;
% loop over chirplets
phi0 = 0;
for k = 1:nchirplets,
  t0 = t(k);
  t1 = t(k+1);
  f0 = freq(k); f1 = freq(k+1);
  
  nblock = t1-t0;
  deep = log2(n/nblock);
 
  slope = FreqIndexToSlope(deep,[f0 f1],n);
  tt = (t0:(t1-1))/n;
  tind =(t0:(t1-1))+1; 
  f0 = f0-1;
  omega(tind) = f0 +  n*slope .* (tt - t0/n);
  
  
  phi(tind) = 2*pi*(f0*(tt-t0/n) + n*slope/2 * (tt-t0/n).^2);
  chirplet = exp(i*phi(tind))/sqrt(length(tt));
  phi0 = angle(yy(tind)*chirplet'); % the mle of the phase-offset (local estimate)
  phi0s(tind) = angle(yy(tind)*chirplet');
  coeff = yy(tind)*chirplet';
  y(tind) = coeff.*chirplet;

  phi(tind) = phi(tind)+phi0;
  phi(1:t1) = unwrap(phi(1:t1));
  amp(tind) = abs( yy(tind)*chirplet');
end

% $RCSfile: GenerateSignalFromPath.m,v $
% $Date: 2008/05/19 22:47:59 $
% $Revision: 1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
