function C = TollCTPlain(sig,param,lambda)
% TollCTPlain -- Calculates penalized chirplet costs for a signal.
%                To use for the BP Estimator.
%            
%  Usage
%    C = TollCTPlain(sig,param,lambda);
%  Inputs
%    sig     signal of dyadic length N=2^J, where J is a positive integer
%    param   chirplet graph parameters as returned by GetChirpletGraphParam
%    lambda  toll, or penalization parameter, assigned to each chirplet
%  Outputs
%    C       penalized chirplet costs, stored in a cell array.
%            For the chirplet coefficient c, the chirplet cost stored is 
%            abs(c).^2 - lambda
%
%  Description
%    A chirplet transform that assumes white noise and constant amplitude.
%    See the documentation for 
%    the chirplet coefficients are stored as described in ChirpletTransform.
%
%  See Also
%    - GetChirpletGraphParam for information on customizing 
%      the chirplet transform.
%    - TollBPDirect.m

% ensure that sig is a row vector and change it if it isn't
x = reshape(sig,1,length(sig));

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

nCells = 2^(finestScale+1) - 1; 

C = cell(1,nCells);

for s = finestScale:-1:coarsestScale,
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);
  t = 0:(2^(J-s)-1);
  chirppart = exp(-i * pi*slopes.' * t.^2/n);
  for b = 0:(2^s-1),
    % handle dyadic interval (s,b)
    ix = dyadindex(s,b,n);
    % Modulation followed by zero-padding
    X = repmat(x(ix),nSlopes,1) .* chirppart;
    c = fft(X.',n); % Operates along columns
    c = c/sqrt(length(ix)); % normalize
    c = abs(c).^2 - lambda;
    C{node(s,b)} = c; 
  end
end


% $RCSfile: TollCTPlain.m,v $
% $Date: 2008/05/19 21:40:41 $
% $Revision: 1.1 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2007
