function C = CTVarAmp(sig,param)
% CTVarAmp -- calculates amplitude adjusted chirplet coefficients 
%             for chirp signals with slowly varying amplitude
%
%  Usage
%    C = CTVarAmp(sig,param);
%  Inputs
%    sig     signal of dyadic length n=2^J, where J is a positive integer
%    param   chirplet graph parameters as returned by GetChirpletGraphParam
%  Outputs
%    C       _square root_ of chirplet coefficients, stored in a cell array.
%
%  Description
%    Assumes slowly varying amplitude for the chirp signal under
%    consideration and uses the least-squares method with orthogonal
%    polynomials to try to fit the amplitude. The degree of the 
%    amplitude fit to use for each scale is set in the chirplet
%    graph parameters passed to the function.
%
%  See Also
%    - ChirpletTransform
%    - GetChirpletGraphParam for information on customizing 
%      the chirplet transform and setting the degrees of the polynomials
%      for the amplitude fit.
%    - GetChirpletNetwork, CalculateStatistic, for setting up the 
%      chirplet graph and calculating the test statistic. 

if nargin < 2,
  errormsg = 'CTVarAmp: Chirplet graph parameters have to be provided!';
  error(errormsg);
end

% ensure that sig is a row vector and change
% it if it isn't
x = reshape(sig,1,length(sig));

n = param{1}(1);
J = param{1}(2);
coarsestScale = param{2}(1);
finestScale = param{2}(2);

% PEND! Make the number of chirplets/cells be one of the parameters
%       returned by GetChirpletGraphParam. Avoids chirplet graph
%       specific things to be calculated at more than one place.
%nCells = 2^(finestScale+1)-2^(coarsestScale);
nCells = 2^(finestScale+1) - 1; % would need to change node.m if the
                                % stuff in previous line is used
C = cell(1,nCells);


% Handle the finest scale specifically, since there
% we only have two data-points per chirplet
if (finestScale==J-1),
  s = J-1;
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes);
  t = [0 1];
  for b = 0:(2^s-1),
    ix = dyadindex(s,b,n);
	% Modulation followed by zero-padding
	X = repmat(x(ix),nSlopes,1) .* exp(-i * pi*slopes.' * t.^2/n);
	X = [X.'; zeros(n - length(ix),nSlopes)];
	c = fft(X); % Operates along columns
	c = c/sqrt(length(ix)); % normalize
	C{node(s,b)} = c;	
  end
  finestScale = finestScale-1; % update finest scale
end

% Other scales:
for s = finestScale:-1:coarsestScale,
  A = PolynomialsInt(2^(J-s),param{5}(s-coarsestScale+1));
  nA = size(A,1); % number of row vectors in A;
  slopes = param{3}{s-coarsestScale+1};
  nSlopes = length(slopes); 
  for b = 0:(2^s-1),
    tmpcost = 0;
    ix = dyadindex(s,b,n);
    Ax = A.*repmat(x(ix),nA,1);
    for k=1:nA,
      % do a chirplet transform
      t = 0:(length(ix)-1);
      % Modulation followed by zero-padding
      X = repmat(Ax(k,:),nSlopes,1) .* exp(-i * pi*slopes.' * t.^2/n);
      X = [X.'; zeros(n - length(ix),nSlopes)];
      c = fft(X); % Operates along columns
      tmpcost = tmpcost + abs(c).^2;
    end
    % NOTE!! The reason a square root is taken is that
    % InitChirpNetwork.m squares the chirplet coefficient
    % and we don't want to do it twice.
    % PEND! Perhaps we should do this differently. 
    C{node(s,b)} = sqrt(tmpcost);
  end

end

% $RCSfile: CTVarAmp.m,v $
% $Date: 2006/05/01 17:44:15 $
% $Revision: 1.6 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2005
