function c = ChirpTransInt(x,n,slsym,maxsl)
% ChirpTransInt -- Chirplet Transform of a Dyadic Interval
%  Usage
%    c = ChirpTransInt(x,n,slsym,maxsl);
%  Input
%    x     signal of length nx, x must be a row vector.
%    n     length of the original signal
%    slsym   set to 1 if slopes discretized symmetrically,
%            0 otherwise
%            PEND!! Get rid of this parameter. Slopes should
%                   normally be discretized symmetrically.
%    maxsl   parameter to choose between different
%            discretization of slopes. Sets the range
%            of slopes [-maxslrange,maxslrange]. 
%            Uses three slopes at the finest scale, -0,5,0,0.5.
%            If omitted, set to default discretization
%            which has range [-0.5,0.5].
%  Outputs
%    c       chirplet coefficients
% Description
%   Calculates the Chirplet Transform of a Dyadic Interval for different
%   slopes and offsets. The number of slopes depends on the length of the
%   signal being transformed. The current version of this function
%   calculates it in the following manner (here nx is the length of the
%   input signal x):
%        slopes = [-maxsl -maxsl+maxsl/nx ... maxsl]
%        i.e. ranging from -maxsl to maxsl with increment maxsl/nx.
%        NOTE: Uses three slopes at the finest scale, -0,5,0,0.5.
%   The choice of offsets is incorporated into the Fourier Transform and there
%   are n offsets (we zero-pad to interpolate in the frequency domain).
%
%   The table of Chirplet coefficients which is returned in an n-by-nx matrix whose
%   values are as follows:
%        -each column in c corresponds to one slope,
%         column k corresponds to the k-th slope
%        -each entry in a column corresponds to an offset,
%         the m-th entry in each column is the offset (m-1)/n

if nargin < 4
  maxsl = 0;
end

nx = length(x);
deep = ceil(log2(n/nx));
  if 2^deep ~= n/nx,
      disp('Warning: interval is not dyadic')
  end

if (slsym==1),
  nSlopes = nx+1;
else
  nSlopes = nx;
end

t = 0:(nx-1);  

% choose slope discretization
if (maxsl == 0 || nx == -1),
  slopes = (0:(nSlopes-1))/nx - 1/2;
else
    slopes = (0:maxsl:(2*maxsl*(nSlopes-1)))/nx - maxsl;    
    nSlopes = length(slopes);
% if (nx == 2),
%    slopes = (0:(nSlopes-1))/nx - 1/2;
% else
%    slopes = (0:maxsl:(2*maxsl*(nSlopes-1)))/nx - maxsl;    
%    nSlopes = length(slopes);
%  end
end


% Modulation followed by zero-padding
X = repmat(x,nSlopes,1) .* exp(-i * pi*slopes.' * t.^2/n);
X = [X.'; zeros(n - nx,nSlopes)];

c = fft(X); % Operates along columns
c = c/sqrt(nx);

% Copyright (c) 2005 Emmanuel J. Candes, Hannes Helgason

% $RCSfile: ChirpTransInt.m,v $
% $Date: 2006/05/01 17:44:15 $
% $Revision: 1.5 $
%
% Copyright (c) Emmanuel J. Candes, Hannes Helgason, California Institute of Technology, 2005
