function parameters = GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange,minfreq,maxfreq,xttype,degrees,mask,noisetype)
% GetChirpletGraphParam -- parameter configuration for the chirplet graph
%
%  Usage
%    parameters =
%      GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange,minfreq,maxfreq,xttype,degrees,mask,noisetype)
%    parameters =
%      GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange,minfreq,maxfreq,xttype,degrees)
%    parameters =
%      GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange,minfreq,maxfreq,xttype)
%    parameters =
%      GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange,minfreq,maxfreq)
%    parameters = GetChirpletGraphParam(N,csc,fsc,sldf,slopeRange)
%    parameters = GetChirpletGraphParam(N,csc,fsc,sldf)
%    parameters = GetChirpletGraphParam(N,csc,fsc)
%    parameters = GetChirpletGraphParam(N,csc)
%    parameters = GetChirpletGraphParam(N)
%  Inputs
%    N          length of the signal to analyze, has to be of dyadic length, N=2^J
%    csc        the coarsest scale index for which to calculate the chirplet transform,
%               corresponds to the scale 2^(-csc). 
%               Has to be a value smaller than or equal to the input parameter fsc.
%               If value is set to [], default value will be used.
%               Default value is csc=0.
%    fsc        the finest scale for which to calculate the chirplet transform,
%               corresponds to the scale 2^(-fsc). 
%               Has to be a value bigger than or equal to csc.
%               If value is set to [], default value will be used.
%               Default value is fsc=J-1.
%    sldf       for adjusting how fine/coarse the slope discretization is. 
%               Sets the difference between adjacent slopes at scale 2^(-s) to
%                     sldf * (maximumslope-minimumslope)/2 * 2^(s-J)
%               The smaller the value is, the finer the slope discretization will be.
%               If it is a power of 2 the time-frequency endpoints of the chirplets 
%               coincide with the discrete frequencies.
%               If value is set to [], default value will be used.
%               Default value is 2.
%    slopeRange Sets the range of slopes. 
%               Can either be a scalar or an array of length 2. 
%               In case of a scalar, the slope range will be set to [-slopeRange,slopeRange].
%               In case of an array, the slope range will be set to [slopeRange(1),slopeRange(2)] 
%               If omitted or value is set to [], the default range [-0.5,0.5] will be used.
%    minfreq    lowest frequency in chirplet graph, corresponds to the frequency 2*pi*minfreq/N. 
%               The value has to be an integer satisfying 0<=minfreq<=maxfreq.
%               If value is set to [], default value will be used.
%               Default value is 0.
%    maxfreq    highest frequency in chirplet graph, corresponds to the frequency 2*pi*maxfreq/N. 
%               The value has to be an integer satisfying minfreq<=maxfreq<=N-1.
%               If value is set to [], default value will be used.
%               Default value is N-1.
%    xttype     Determines the type of chirplet transform to use.
%               Possible configurations:
%               'PLAIN' or 1              - for constant amplitude and white noise
%               'VARAMP' or 2             - for varying amplitude and white noise
%               'COLOREDNOISE' or 3       - for constant amplitude and colored noise
%               'COLOREDNOISEVARAMP' or 4 - for varying amplitude and colored noise
%               If omitted, plain chirplet transform assuming constant
%               amplitude and white noise will be used. 
%    degrees    Degrees of the amplitude fit. Used only for chirplet transforms
%               that consider varying amplitude.
%               Can be either empty, in which case the default
%               values are used, or it must be a vector of the same
%               length as number of allowable scales. 
%               Entry k has the degree of the polynomial fit to the amplitude at the 
%               k-th coarsest scale (goes from coarsest to finest scale).
%               For example if csc=0 and fsc=4 and one wants to use a quadratic amplitude fit
%               for the coarsest scale and linear fit to other scales, set degrees = [2 1 1 1 1].
%               If omitted or empty and a varying amplitude transform is chosen, 
%               degree 2 will be used for all scales.
%               It is an error to provide a non-empty array as the
%               degrees parameter when the constant amplitude
%               transform is chosen.
%    mask       Frequency mask of length N, in FFT order.
%               At present this parameter is only used in the
%               colored noise transform for circular noise to eliminate frequency bins.
%               If omitted or set to [] no frequency masking will be done.
%    noisetype  In case of colored noise, this parameter tells whether the noise is circular
%               or not. In case of white noise (xttype=1 or 2) this parameter is ignored.
%               Possible configurations:
%                 0 - for circular colored noise
%                 1 - for non-circular colored noise
%               If omitted, circular colored noise will be assumed.
%  Output
%    parameters  a cell array with parameters for the chirplet graph and transform.
%                The entries are:
%      parameters{1}  vector of length 2, first entry lenght of signal, N,
%                     second entry dyadic length of signal, J (N=2^J)
%      parameters{2}  vector of length 2, first entry is the coarsest 
%                     scale for which to calculate the chirplet transform, 
%                     second entry is the finest scale for which to
%                     calculate the chirplet transform.
%      parameters{3}  slopes stored in a cell array of size 
%                     1-by-(number of allowable scales).
%                     Entry k has the slopes at the k-th coarsest scale,
%                     i.e. goes from coarsest to finest scale.
%      parameters{4}  vector of length 2, first entry is the lowest frequency
%                     in the graph, second entry is the highest frequency.
%      parameters{5}  Degrees of the amplitude fit stored in an array. 
%                     See the documentation for the input parameter 'degrees' above.
%                     If a constant amplitude chirplet transform is chosen, the
%                     value of this parameter will not matter.
%      parameters{6}  The type of transform as character array. 
%                     See documentation for the parameter 'xttype' above.
%      parameters{7}  The frequency mask
%      parameters{8}  The type of noise (same value as noisetype)
%
%  Description
%    Creates a cell array with parameter configuration which determine the chirplet graph. 
%    These parameters are then later on passed to the chirplet transform, optimization
%    routines and plotting utilities. The main purpose of wrapping these parameters in one 
%    structure is to simplify parameter passing between functions. 
%
%    NOTE: This function only allows for evenly spaced slopes at each fixed scale but
%    it is possible to overwrite the slope discretization parameter. See the user's manual.
%
%  See Also
%    - GetSlopes - can be used to custom make slope discretization schemes

% PEND! Perhaps this structure should also have information about
%       number of chirplets in the graph, whether it assumes connected
%       offsets or slopes (would need to include the curvature constraint)
%       and so forth.
% PEND! Switch to struct if it does not affect performance. 
%       The output parameters, parameters{4} etc. look cryptic.
% PEND! For input parameters, switch to ('Property','Value') style.

J = ceil(log2(N)); % dyadic length of signal

if (2^J ~= N),
  msg = 'GetChirpletGraphParam: Signal length is not dyadic. Should be of the form N=2^J.';
  msgid = 'ChirpLab:ChirpletGraph:NonDyadicLength';
  error(msgid,msg);
end

if nargin < 10
  mask = 1;
end;

if nargin<11,
  noisetype = 0;
end

% check number of arguments and set to default values
if (nargin < 8),
  xttype = 'PLAIN';
  if (nargin < 7),
    maxfreq = N-1;
    if (nargin < 6),
      minfreq = 0;
      if (nargin < 5),
        % use default slope discretization range
		slopeRange = [-0.5 0.5];
        if (nargin < 4),
          sldf = 2;
          if (nargin < 3),
            fsc = J-1;
            if (nargin < 2),
              csc = 0;
            end
          end
        end
      end
    end
  end
end

% check whether transform type is supported
if(ischar(xttype)),
  if ~(strcmpi(xttype,'PLAIN') | strcmpi(xttype,'VARAMP') |...
       strcmpi(xttype,'COLOREDNOISE') | strcmpi(xttype,'COLOREDNOISEVARAMP') )
    errormsg = 'GetChirpletGraphParam: Unsupported transform type.';
	error(errormsg);
  end
elseif (isnumeric(xttype))
  switch xttype
    case 1,    
	  xttype = 'PLAIN';
	case 2,
	  xttype = 'VARAMP';
	case 3,
	  xttype = 'COLOREDNOISE';
	case 4,
	  xttype = 'COLOREDNOISEVARAMP';
	otherwise,
	  errormsg = 'GetChirpletGraphParam: Unsupported transform type.';
	  error(errormsg);
  end
else
  errormsg = 'GetChirpletGraphParam: Unsupported transform type. See documentation.';
  error(errormsg);
end

% check if any parameter has an empty value
if(isempty(csc)),
  csc = 0; 
end
if(isempty(fsc)),
  fsc = J-1; 
end
if(isempty(sldf)),
  sldf = 2; 
end
if(length(slopeRange) == 0),
  % empty slopeRange, use default
  slopeRange = [-0.5 0.5];
elseif(length(slopeRange) == 1),
  % slopeRange is a scalar
  slopeRange = [-slopeRange slopeRange];
elseif (length(slopeRange) > 2)
  errormsg = 'GetChirpletGraphParam: slopeRange can only be empty or scalar or vector of length 2';
  error(errormsg);
end;
% Now check the final value of slopeRange
if (slopeRange(1) > slopeRange(2))
  errormsg = 'GetChirpletGraphParam: slopeRange(1) must be less than slopeRange(2)';
  error(errormsg);
end;
if(isempty(minfreq)),
  minfreq = 0; 
end
if(isempty(maxfreq)),
  maxfreq = N-1; 
end
if(isempty(mask)),
  mask = 1;
end


if (csc>fsc),
  msg = 'The parameters csc and fsc must satisfy csc<=fsc.';
  msgid = 'ChirpLab:ChirpletGraph:ScalesNotRight';
  error(msgid,msg);
end

if (minfreq>maxfreq | minfreq < 0 | maxfreq >= N),
  msg = 'The parameters minfreq and maxfreq must satisfy 0<= minfreq <= maxfreq <= N-1.';
  msgid = 'ChirpLab:ChirpletGraph:ForbiddenFrequencyRange';
  error(msgid,msg);
end


% set the chirplet graph parameters
parameters = cell(1,8);
parameters{1} = [N J];
parameters{2} = [csc fsc];

% get the slopes
slopeparam = cell(1,3);
slopeparam{1} = slopeRange;
slopeparam{2} = zeros(1,fsc-csc+1); % #allowable scales=fsc-csc+1
slopeDifference = abs(slopeRange(2)-slopeRange(1));
for s=csc:fsc,
    nx = 2^(J-s);
    if (sldf/nx > 1),
      % just to make sure that we have at least 2 slopes per scale
	  % in the case of the finest allowable scale (s=J-1)
      slopeparam{2}(s-csc+1) = slopeDifference/2;
    else
      slopeparam{2}(s-csc+1) = (slopeDifference/2)/nx*sldf;
    end
end
slopeparam{3} = parameters{2};

parameters{3} = GetSlopes(slopeparam);

% set maximum and minimum frequencies
parameters{4} = [minfreq maxfreq];

% set the type of chirplet transform to use
parameters{6} = xttype; 

if (strcmpi(xttype,'VARAMP') | strcmpi(xttype,'COLOREDNOISEVARAMP')),
  % set the degrees for the polynomial fit of the amplitude
  % If degrees is of non-zero length it's length must be equal
  % to the number of scales
  if (nargin >= 9 & length(degrees) ~= 0)
    % Check its length.
    if (length(degrees) ~= fsc-csc+1),
      errormsg = [ 'GetChirpletGraphParam: The degrees parameter ' ...
                   'should be an array of a length equal to the ' ...
                   'number of scales in the chirplet graph.' ];
      error(errormsg);
    end
  else
    % Degrees param was not provided or was empty, using a default setting
    % for the polynomial fit of the amplitude
    degrees = 2*ones(1,fsc-csc+1); % set degree to 2
  end
else
  % Constant amplitude case
  if (nargin >= 9 & length(degrees) ~= 0)
    % Non-empty degrees was provided for constant-amplitude
    % transform, possibly as a placeholder so mask could 
    % be given as well.
    error([ 'GetChirpletGraphParam: The degrees parameter must be ' ...
            'the empty array when using the constant-amplitude ' ...
            'transform' ]);
  else
    % Degrees not provided or empty, using constant amplitude transform,
    % set the degrees to an empty array
    degrees = [];
  end;
end

% set the degrees for the polynomial fit of the amplitude
parameters{5} = degrees;

% Set the mask parameter. At present, it is an error to provide
% the mask for the white noise transform unless the mask is
% identically 1
if (nargin >= 10)
  % Mask was provided
  if (~(strcmpi(xttype,'COLOREDNOISE') | strcmpi(xttype,'COLOREDNOISEVARAMP')) ...
      & mask ~= 1)
    error('GetChirpletGraphParam: mask is not used in white noise transform');
  elseif (length(mask) ~= N & mask ~= 1)
    error('GetChirpletGraphParam: mask must be either length N or the scalar 1');
  else
    parameters{7} = mask;
  end;
else
  % Use default value, the identity mask
  parameters{7} = 1;
end;

if (noisetype == 0 | noisetype == 1)
  parameters{8} = noisetype;
else
  error('GetChirpletGraphParam: noisetype must be 0 or 1');
end;


% $RCSfile: GetChirpletGraphParam.m,v $
% $Date: 2007/02/12 23:41:17 $
% $Revision: 1.22 $
%
% Copyright (c) Hannes Helgason, California Institute of Technology, 2006
