% An example showing how to find the MCTR test statistic.
%
% Steps:
% 1) Generate a chirping signal
% 2) Do a chirplet transform
% 3) Fill in chirplet costs in network
% 4) Calculate the MCTR. 
% 5) Plot chirplet path.
disp('--Running FindMCTRDemo.m--');

J = 9;          % dyadic length of signal
N =  2^J;       % length of signal

STATTYPE = 'MCTR'; % set the test statistic type to MCTR

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirp...');
[y,omega]= MakeChirp('CosinePhase',N);
y = y/norm(y); % normalize so the signal has l2-norm 1

% plot real part of signal and instantaneous frequency
figure;
subplot(2,1,1);plot(real(y));
xlabel('time');ylabel('Real Part of Signal');
subplot(2,1,2);plot(omega);
xlabel('time');ylabel('frequency');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Taking chirplet transform...');
graphparam = GetChirpletGraphParam(N);
cc = ChirpletTransform(y,graphparam); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
coarsestScaleArray = [0 1 2 3]; % restricts the chirplet graph to scales 
                            % as fine and finer than 0,1,2,3 resp.
mctrCost = zeros(1,length(coarsestScaleArray));
mctrPath = cell(1,length(coarsestScaleArray));
nchirplets = zeros(1,length(coarsestScaleArray));

disp('Assigning costs to chirplet graph and running optimization routine...');
disp('(this will take a while)');
for k=1:length(coarsestScaleArray),
  disp( strcat( 'Coarsest scale=',num2str(coarsestScaleArray(k)) ) );
  cnetwork = GetChirpletNetwork(cc,graphparam,coarsestScaleArray(k));
  [mctrCost(k),mctrPath{k},nchirplets(k)] = CalculateStatistic(cnetwork,STATTYPE); 
end
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
npaths = length(mctrPath);
figure;
for k=1:npaths,
  subplot(npaths,1,k);
  DisplayChirpletPath(mctrPath{k},graphparam); % plot MCTR path
  title(strcat('MCTR path. Coarsest scale=',num2str(coarsestScaleArray(k)),...
                   ', |cost/L|=',num2str(abs(mctrCost(k)))));
  hold on;
  plot(0:(length(omega)-1),omega);
%  axis([0 N 60 200]);
  hold off;
end
