% An example showing how to find the Shortest Path (SP) through the
% chirplet graph.
%
% Steps:
% 1) Generate a chirping signal
% 2) Do a chirplet transform
% 3) Fill in chirplet costs in network
% 4) Calculate the SP
% 5) Plot chirplet path associated with the SP.
disp('--Running FindSPDemo.m--');

J = 9;          % dyadic length of signal
N =  2^J;       % length of signal

% Settings for test statistic
STATTYPE = 'SP'; % Calculates the Shortest Path in the chirplet graph

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirp...');
[y,omega]= MakeChirp('CubicPhase',N);
y = y/norm(y); % normalize so the signal has l2-norm 1

% plot real part of signal and instantaneous frequency
figure;
subplot(2,1,1);plot(real(y));
xlabel('time');ylabel('Real Part of Signal');
subplot(2,1,2);plot(omega);
xlabel('time');ylabel('frequency');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Taking chirplet transform...');
graphparam = GetChirpletGraphParam(N);
cc = ChirpletTransform(y,graphparam); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirplet graph and assigning costs... (this will take a while)');
cnetwork = GetChirpletNetwork(cc,graphparam);
disp('Running optimization routine for graph...');
[costpath,shortestpath,nchirplets] = CalculateStatistic(cnetwork,STATTYPE);
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
figure;
DisplayChirpletPath(shortestpath,graphparam); % plot the best path
title(strcat('Shortest Path through the chirplet graph. #chirplets=',num2str(nchirplets),...
                   ', |cost|=',num2str(abs(costpath))));
hold on;
plot(0:(length(omega)-1),omega);
axis([0 N 0 200]);
hold off;
