% An example showing how to use the varying amplitude chirplet transform.
% It also uses the BP test statistic.
%
% Steps:
% 1) Generate a chirping signal
% 2) Configure the chirplet graph and do a chirplet transform
% 3) Fill in chirplet costs in network
% 4) Calculate the BP for prescribed path lengths test statistic
% 5) Plot chirplet paths corresponding to the lengths
%    chosen for the BP test statistic.
disp('--Running VarAmpDemo.m--');

J = 10;         
N =  2^J;       % length of signal

% Settings for test statistic
STATTYPE = 'BPFORPLOTTING'; % Calculates the BP statistic and returns chirplet paths for plotting
maxLength = 8;              % Calculates BP for lengths 1,...,maxLength

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% GENERATE A SIGNAL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirp...');
[y,omega]= MakeChirp('CosinePhaseModulatedAmplitude',N);
y = y/norm(y); % normalize so the signal has l2-norm 1

% plot real part of signal and instantaneous frequency
figure;
subplot(2,1,1);plot(real(y));
xlabel('time');ylabel('Real Part of Signal');
subplot(2,1,2);plot(omega);
xlabel('time');ylabel('frequency');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHIRPLET TRANSFORM AND GRAPH CONFIGURATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Configuring graph and doing chirplet transform...');
csc = 0; % coarsest scale
fsc = 6; % finest scale
fmin = 0; % minimum frequency
fmax = N/2-1; % maximum frequency
transformType = 'VARAMP';
degreeArray = [2 2 1 1 1 1 1]; % degrees for polynomials, coarsest to finest scale
graphparam = GetChirpletGraphParam(N,csc,fsc,[],[],fmin,fmax,transformType,degreeArray);

cc = ChirpletTransform(y,graphparam); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SET UP CHIRPLET NETWORK AND FILL IN CHIRPLET COSTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
disp('Generating chirplet graph and assigning costs...');
cnetwork = GetChirpletNetwork(cc,graphparam);
disp('Running optimization routine for graph...');
[costpath,bestpaths] = CalculateStatistic(cnetwork,STATTYPE,maxLength);
disp('Done!');

%%%%%%%%%%%%%%%%%%%
% PLOT PATHS
%%%%%%%%%%%%%%%%%%%
figure;
% Plot paths corresponding to dyadic number of chirplets, 1,2,4,...,2^(floor(log2(maxLength)))
nplots = floor(log2(maxLength))+1;
for k= 0:nplots-1,
  subplot(nplots,1,k+1);
  DisplayChirpletPath(bestpaths{2^k},graphparam); % plot the best path
  
  title(strcat('Best path restricted to #chirplets=',num2str(2^k),...
                   ', |cost|=',num2str(abs(costpath(2^k)))));
  hold on;
  % plotting the `instantaneous frequency'
  plot(0:(length(omega)-1),omega);
  axis([0 N fmin fmax]);
  hold off;
end
